package eu.gronos.beschriftungen.util;

import java.util.prefs.Preferences;

import eu.gronos.beschriftungen.interfaces.ConfigOption;
import eu.gronos.beschriftungen.model.ConfigSchluessel;
import eu.gronos.kostenrechner.Kostenrechner;

/**
 * Die abstrakte Oberklasse für den Zugriff auf Registry oder
 * Benutzereinstellungen und Standard-Implementierung von {@link ConfigOption}
 * speichert den Ast /eu/gronos/kostentenor und den Schlüssel
 *
 * @author Peter Schuster (setrok)
 * @date 18 Feb 2019
 *
 * @param <E> der Typ des Wertes für den Eintrag in der Registry bzw. den
 *            Preferences, kann bisher {@link String}, {@link Boolean} oder
 *            {@link Integer} sein.
 */
public abstract class PreferencesConfigOption<E> implements ConfigOption<E> {

	/**
	 * NODE speichert den Pfad in der Registry zu diesem Programm als
	 * {@link String}.
	 */
	public static final String NODE = "/eu/gronos/kostentenor";
	private static final Preferences userPrefs = Preferences.userRoot().node(NODE);
	private final String schluessel;
	protected final E standardWert;

	/**
	 * Der Konstruktor erfordert den Wert für den finalen {@link String}
	 * {@link #getSchluessel()} und den <code>E</code> {@link #standardWert} und
	 * zieht sie sich aus dem {@link ConfigSchluessel}
	 * 
	 * @param parameter ein {@link ConfigSchluessel} mit demselben generischen
	 *                  Typ-Parameter <code>E</code> wie die
	 *                  {@link PreferencesConfigOption}
	 */
	public PreferencesConfigOption(ConfigSchluessel<E> parameter) {
		this.schluessel = parameter.getSchluessel();
		this.standardWert = parameter.getStandardWert();
	}

	/**
	 * Die Klasse Preferences hat große Ähnlichkeit mit den Klassen Properties
	 * beziehungsweise HashMap (vergleiche Kapitel 13, »Einführung in
	 * Datenstrukturen und Algorithmen«). Schlüssel/Werte-Paare lassen sich
	 * einfügen, löschen und erfragen. Allerdings ist die Klasse Preferences kein
	 * Mitglied der Collection-API, und es existiert auch keine Implementierung von
	 * Collection-Schnittstellen.
	 * 
	 * @return gibt {@link #userprefs} als Preferences zurück.
	 */
	public Preferences getUserprefs() {
		return userPrefs;
	}

	/**
	 * 
	 * @return gibt {@link #schluessel} als String zurück.
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#getSchluessel()
	 */
	@Override
	public String getSchluessel() {
		return schluessel;
	}

	/**
	 * @return gibt {@link #standardWert} als <code>E</code> zurück.
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#getStandardWert()
	 */
	@Override
	public E getStandardWert() {
		return standardWert;
	}

	/**
	 * Die Methode schreibt den Wert in die Registry/Preferences und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @param value als <code>E</code>
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#setValue(E)
	 */
	@Override
	public void setValue(E value) {// boolean
		inRegistrySchreiben(value);
	}

	/**
	 * Die Methode liest den Wert aus der Registry (den Preferences) und setzt das
	 * Oberflächenelement entsprechend
	 * 
	 * @return value als <code>E</code>
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#getValue()
	 */
	@Override
	public E getValue() {// boolean
		E value = ausRegistryLesen();
		return value;
	}

	/**
	 * Die Methode setzt wieder den Standardwert in die Registry/Preferences.
	 * 
	 * @see eu.gronos.beschriftungen.interfaces.ConfigOption#reset()
	 */
	@Override
	public void reset() {
		setValue(getStandardWert());
	}

	/**
	 * Die Methode loggt einen Schreibvorgang
	 * 
	 * @param wert der in die Registry/Preferences geschriebene Wert
	 */
	protected void logInfo(String wert) {
		String msg = String.format("Schreibe in Registry ... %s/%s: %s", getUserprefs().absolutePath(), getSchluessel(),
				wert);
		Kostenrechner.getLogger().info(msg);
	}

	/**
	 * Liest einen Wert aus dem Java-Zweig der Registry, unter dem Schluessel, der
	 * dem Konstruktor mitgegeben wurde. Wird von {@link #setValue(Object)}
	 * aufgerufen.
	 * 
	 * @return den <code>E</code>-Wert aus der Registry oder den Standardwert, der
	 *         dem Konstruktor mitgegeben wurde
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 */
	protected abstract void inRegistrySchreiben(E wert);

	/**
	 * Schreibt den übergebenen <code>E</code>-Wert in den Java-Zweig der Registry,
	 * und zwar in den Schlüssel, der dem Konstruktor übergeben wurde. Wird vom
	 * Konstruktor aufgerufen.
	 * 
	 * @param wert der Wert, der in die Registry soll (<code>E</code>)
	 * 
	 * @url {@link "http://openbook.galileocomputing.de/javainsel/javainsel_11_009.html#dodtp1f33b78d-493f-43a5-be9c-afc95d5f903b"}
	 *      Java ist auch eine Insel Kapitel 11.9.2 Einträge einfügen, auslesen und
	 *      löschen
	 */
	protected abstract E ausRegistryLesen();

}
