/**
 * FontUndFensterHelfer.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.beschriftungen.view;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GraphicsEnvironment;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.Window;
import java.util.List;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.text.AttributeSet;
import javax.swing.text.StyleConstants;

/**
 * Die Klasse erweitert {@link FontHelfer} um Abfragen aus einem {@link Window}.
 * Das ist die gemeinsame Oberklasse für {@link JFrame} und {@link JDialog}. Man
 * kann Fonts suchen und die Fenstergröße nach Schriftgröße setzen.
 *
 * @author Peter Schuster (setrok)
 * @date 15.08.2018
 *
 */
public class FontHelferForWindow extends FontHelfer {
	private final Window window;

	/**
	 * Der Konstruktor braucht den {@link JDialog}, um daran die Größe usw. setzen
	 * zu können.
	 * 
	 */
	public FontHelferForWindow(Window window) {
		this.window = window;
	}

	/**
	 * Die Methode ermittelt die bevorzugte Größe des {@link JDialog}s, indem die
	 * Größe der Ränder hinzugerechnet wird.
	 * 
	 * @param insets {@link Insets} des {@link JDialog}s
	 * @return eine {@link java.awt.Dimension} mit der vergrößerten bevorzugten
	 *         Größe
	 * 
	 * @see java.awt.Container#getPreferredSize()
	 * @see java.awt.Container#getMinimumSize()
	 */
	public Dimension calculatePreferredSizeWith(Insets insets) {
		final Dimension minimum = window.getMinimumSize();//
		final Dimension preferred = window.getPreferredSize();//
		return calculatePreferredSizeWith(insets, minimum, preferred);
	}

	/**
	 * Zusehen, dass das Fenster nicht breiter als maxSpalten Zeichen und nicht
	 * höher als maxZeilen Zeilen wird.
	 * 
	 * @param attrib     ein {@link AttributeSet}
	 * @param maxSpalten Spaltenzahl für die Berechnung
	 * @param maxZeilen  Zeilenanzahl für die Berechnung
	 */
	public void setPreferredSizeFromAttributeSet(AttributeSet attrib, int maxSpalten, int maxZeilen) {
		Dimension maximumSize = window.getMaximumSize();
		Dimension preferredSize = window.getPreferredSize();
		calculatePreferredSizeFrom(maxSpalten, maxZeilen, getCharDimension(attrib), maximumSize, preferredSize);
		window.setPreferredSize(preferredSize);
	}

	/**
	 * Die Methode ermittelt den Standard-{@link Font} des {@link #window}
	 * 
	 * @return den {@link Component#getFont()} als {@link Font}
	 */
	public Font baseFont() {
		return window.getFont();
	}

	/**
	 * Zieht aus der Liste die Dimensionen aller Panels und errechnet den
	 * Durchschnitt. Dabei werden einzelne Elemente, die die Bildschirmmaße
	 * überschreiten, auf die Bildschirmmaße gestutzt
	 * 
	 * @param dimensionen Liste die Dimensionen aller Panels
	 * @return den Durchschnitt
	 */
	public static Dimension averageDimension(List<Dimension> dimensionen) {
		long width = 0L, height = 0L, zaehler = 0L;
		Dimension screenSize = getScreenSize();

		// Breiten- und Höhen-Werte aufaddieren
		for (Dimension d : dimensionen) {
			long w, h;
			// höchstens die Bildschirmbreite nehmen
			if (d.width > screenSize.width) {
				w = screenSize.width;
			} else {
				w = d.width;
			}
			width += w;

			// höchstens die Bildschirmhöhe nehmen
			if (d.height > screenSize.height) {
				h = screenSize.height;
			} else {
				h = d.height;
			}
			height += h;
			zaehler++;
		}

		// Durchschnitt zurückgeben
		return new Dimension((int) (width / zaehler), (int) (height / zaehler));
	}

	/**
	 * @return
	 * @throws HeadlessException
	 */
	private static Dimension getScreenSize() throws HeadlessException {
		// Bildschirmgröße ermitteln
		Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		// In Mehr-Monitor-Systemen gibt das hier die Größe des mittleren Monitors
		Dimension maxBounds = GraphicsEnvironment.getLocalGraphicsEnvironment().getMaximumWindowBounds().getSize();

		screenSize.height = Integer.min(screenSize.height, maxBounds.height);
		screenSize.width = Integer.min(screenSize.width, maxBounds.width);
		
		return screenSize;
	}

	/**
	 * Die Methode dient dazu, die Breite eines Zeichens ('n') und die
	 * Standardzeilenhöhe zu ermitteln
	 * 
	 * @param attrib ein {@link AttributeSet}
	 * @return Zeichenbreite und Zeilenhöhe in Pixeln als {@link Dimension}
	 */
	private Dimension getCharDimension(AttributeSet attrib) {
		FontMetrics fontMetrics = window.getFontMetrics(getFontForName(StyleConstants.getFontFamily(attrib)));
		return getCharDimension(fontMetrics);
	}
}
