/*
 * KleinDialogAction.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller;

import java.awt.Component;
import java.awt.HeadlessException;

import javax.swing.Icon;
import javax.swing.JDialog;
import javax.swing.JOptionPane;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.controller.BeschriebeneAktion;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.interfaces.AbfrageLieferant;
import eu.gronos.kostenrechner.view.RechnerhammerIcons;

/**
 * Kleine Oberklasse, die Methoden bereithält, um mit {@link JOptionPane} einen
 * {@link JDialog} zu erzeugen.
 *
 * @author Peter Schuster (setrok)
 * @date 06.01.2020
 *
 */
public abstract class KleinDialogAction extends BeschriebeneAktion implements AbfrageLieferant<JDialog> {

	private static final long serialVersionUID = -5606764523838104402L;
	private JDialog dialog = null;

	public KleinDialogAction(LangBeschriftung beschriftung) {
		super(beschriftung);
	}

	/**
	 * @return gibt {@link #dialog} als {@link JDialog} zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.AbfrageLieferant#getDialog()
	 */
	public JDialog getDialog() {
		return dialog;
	}

	/**
	 * Die Methode baut eine {@link JOptionPane}, einen {@link JDialog}, setzt den
	 * {@link #getDialog()} und zeigt ihn an. Dann gibt sie die Auswahl zurück.
	 * 
	 * @param beschriftung eine {@link LangBeschriftung}, der auch
	 *                     ({@link javax.swing.Action#NAME}),
	 *                     ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                     {@link javax.swing.Action#MNEMONIC_KEY},
	 *                     {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *                     ({@link javax.swing.Action#ACTION_COMMAND_KEY}) entnommen
	 *                     werden.
	 * @param message      the <code>Object</code> to display
	 * @param optionType   die Wahlmöglichkeiten, die der Dialog anzeigen soll:
	 *                     <code>DEFAULT_OPTION</code>, <code>YES_NO_OPTION</code>,
	 *                     <code>YES_NO_CANCEL_OPTION</code>,
	 *                     <code>OK_CANCEL_OPTION</code>
	 * @param messageType  den Typ der anzuzeigenden Nachricht:
	 *                     <code>ERROR_MESSAGE</code>,
	 *                     <code>INFORMATION_MESSAGE</code>,
	 *                     <code>WARNING_MESSAGE</code>,
	 *                     <code>QUESTION_MESSAGE</code>, or
	 *                     <code>PLAIN_MESSAGE</code>
	 * @return den {@link JOptionPane#getValue()}
	 */
	protected int showDialog(Beschriftung beschriftung, String message, int optionType, int messageType) {
		/* JOptionPane */
		JOptionPane pane = createPane(Kostenrechner.getInstance(), message, optionType, messageType,
				RechnerhammerIcons.getRechnerhammer64());
		// JDialog
		setDialog(createDialog(Kostenrechner.getInstance(), pane, beschriftung));

		int antwort = show(pane, getDialog());
		setDialog(null);
		return antwort;
	}

	/**
	 * @param dialog d. {@link #dialog}, d. gesetzt werden soll als {@link JDialog}.
	 */
	protected void setDialog(JDialog dialog) {
		this.dialog = dialog;
	}

	/**
	 * Die Methode lässt {@link JOptionPane} den {@link JDialog} bauen.
	 * 
	 * @param parentComponent determines the frame in which the dialog is displayed;
	 *                        if the <code>parentComponent</code> has no
	 *                        <code>Frame</code>, a default <code>Frame</code> is
	 *                        used
	 * @param pane            den gebauten {@link JOptionPane}
	 * @param beschriftung    eine {@link LangBeschriftung}, der auch
	 *                        ({@link javax.swing.Action#NAME}),
	 *                        ({@link javax.swing.Action#SHORT_DESCRIPTION}),
	 *                        {@link javax.swing.Action#MNEMONIC_KEY},
	 *                        {@link javax.swing.Action#ACCELERATOR_KEY} und
	 *                        ({@link javax.swing.Action#ACTION_COMMAND_KEY})
	 *                        entnommen werden.
	 * @return den gebauten {@link JDialog}
	 * @exception HeadlessException if <code>GraphicsEnvironment.isHeadless</code>
	 *                              returns <code>true</code>
	 */
	private JDialog createDialog(Component parentComponent, JOptionPane pane, Beschriftung beschriftung)
			throws HeadlessException {
		JDialog dialog = pane.createDialog(parentComponent, "");
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.beschrifte(dialog, beschriftung);
		pane.selectInitialValue();
		return dialog;
	}

	/**
	 * Die Methode legt eine Instanz von {@link JOptionPane} an.
	 * 
	 * @param parentComponent determines the frame in which the dialog is displayed;
	 *                        if the <code>parentComponent</code> has no
	 *                        <code>Frame</code>, a default <code>Frame</code> is
	 *                        used
	 * @param message         the <code>Object</code> to display
	 * @param optionType      the options to display in the pane:
	 *                        <code>DEFAULT_OPTION</code>,
	 *                        <code>YES_NO_OPTION</code>,
	 *                        <code>YES_NO_CANCEL_OPTION</code>,
	 *                        <code>OK_CANCEL_OPTION</code>
	 * @param messageType     the type of message to be displayed:
	 *                        <code>ERROR_MESSAGE</code>,
	 *                        <code>INFORMATION_MESSAGE</code>,
	 *                        <code>WARNING_MESSAGE</code>,
	 *                        <code>QUESTION_MESSAGE</code>, or
	 *                        <code>PLAIN_MESSAGE</code>
	 * @param icon            the Icon image to display
	 * @return den gebauten {@link JOptionPane}
	 */
	private JOptionPane createPane(Component parentComponent, String message, int optionType, int messageType,
			Icon icon) {
		JOptionPane pane = new JOptionPane(message, messageType, optionType, icon, null, null);

		pane.setInitialValue(null);
		pane.setComponentOrientation(
				((parentComponent == null) ? JOptionPane.getRootFrame() : parentComponent).getComponentOrientation());
		return pane;
	}

	/**
	 * Die Methode zeigt den {@link JDialog} und gibt die Auswahl zurück.
	 * 
	 * @param pane   den gebauten {@link JOptionPane}
	 * @param dialog den gebauten {@link JDialog}
	 * @return den {@link JOptionPane#getValue()}
	 */
	private int show(JOptionPane pane, JDialog dialog) {
		dialog.setVisible(true);
		dialog.dispose();

		Object value = pane.getValue();

		if (value == null)
			return JOptionPane.CLOSED_OPTION;
		if (value instanceof Integer)
			return ((Integer) value).intValue();
		return JOptionPane.CLOSED_OPTION;
	}

}