/*
 * GesamtschuldPruefer.java
 * eu.gronos.kostenrechner.controller.baumbach (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.baumbach;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JList;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.GenusNumerus;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenListModel;
import eu.gronos.kostenrechner.util.baumbach.GesamtschuldPruefer;

/**
 * Der Pruefer kontrolliert alle Werte zum Bau einer
 * {@link BaumbachGesamtschuldnerschaft}
 *
 * @author Peter Schuster (setrok)
 * @date 15.07.2019
 *
 */
public class GesamtschuldParser implements ParsendUndBauend<BaumbachGesamtschuldnerschaft> {

	private BaumbachGesamtschuldnerschaft rueckgabewert;
	private JList<BaumbachBeteiligter> beteiligtenAuswahlListe;
	private JFormattedTextField ftfUnterliegen;
	private BaumbachBeteiligtenListe baumbachBeteiligtenListe;

	/**
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ParsendUndBauend<BaumbachGesamtschuldnerschaft> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("36011".equals(component.getName())) {
					beteiligtenAuswahlListe = (JList<BaumbachBeteiligter>) component;
					baumbachBeteiligtenListe = ((BeteiligtenListModel) beteiligtenAuswahlListe.getModel()).getValues();
				} else if ("36025".equals(component.getName())) {
					ftfUnterliegen = (JFormattedTextField) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob alle {@link JComponent}s zur
	 * Eingabe mit gültigen Werten gefüllt sind und diese zueinander nicht im
	 * Widerspruch stehen.
	 * 
	 * Hier wirft die Methode Fehler, wenn weniger als zwei Beteiligte ausgewählt
	 * sind oder wenn das Gesamtunterliegen bei einem Beteiligten größer als der
	 * Streitwert ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		try {
			rueckgabewert = leseFelder();
			return parseEingabe(rueckgabewert);
		} catch (IllegalArgumentException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
	}

	/**
	 * Die Methode wird aufgerufen, um zu prüfen, ob der Rückgabewert gültige
	 * Teilwerte hat, die zueinander nicht im Widerspruch stehen.
	 * 
	 * Hier wirft die Methode Fehler, wenn weniger als zwei Beteiligte ausgewählt
	 * sind oder wenn das Gesamtunterliegen bei einem Beteiligten größer als der
	 * Streitwert ist.
	 * 
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(BaumbachGesamtschuldnerschaft gesamtschuld)
			throws IllegalArgumentException, NullPointerException {
		this.rueckgabewert = gesamtschuld;
		return new GesamtschuldPruefer().pruefeEingabe(rueckgabewert);
	}

	/**
	 * Die Methode soll aus den Eingabefeldern des Dialogs einen Rückgabewert
	 * zusammenzubauen, der dann von {@link #showDialogAndAsk()} an die aufrufende
	 * Maske zurückgegeben werden kann.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public BaumbachGesamtschuldnerschaft getRueckgabewert() {
		return rueckgabewert;
	}

	private BaumbachGesamtschuldnerschaft leseFelder() throws IllegalArgumentException, NullPointerException {
		NumberExtractor extractor = new NumberExtractor(ftfUnterliegen,
				ftfUnterliegen.getAccessibleContext().getAccessibleName());

		BeteiligtenTyp typ = getLetztenBeteiligten().getTyp();
		if (typ == BeteiligtenTyp.KLAEGER)
			typ = BeteiligtenTyp.DRITTWIDERBEKLAGTE;

		int[] indices = beteiligtenAuswahlListe.getSelectedIndices();
		GenusNumerus genusNumerus = getLetztenBeteiligten().getGenusNumerus();
//		if (genusNumerus < Beteiligter.PLURAL)
//			genusNumerus += Beteiligter.PLURAL;
		if (genusNumerus.getNumerus() == GenusNumerus.Numerus.SINGULAR)
			genusNumerus = GenusNumerus.of(genusNumerus.getGenus(), GenusNumerus.Numerus.PLURAL);
		/*
		 * anWiderklageBeteiligt darf beim gesamtschuldnerischen Beklagten // nicht
		 * gesetzt werden (eine Widerklage führt man nicht // gesamtschuldnerisch)
		 */
		final boolean anWiderklageBeteiligt = baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()
				&& typ != BeteiligtenTyp.BEKLAGTE;
		double unterliegen = extractor.extractDouble(); 
		return new BaumbachGesamtschuldnerschaft(typ, genusNumerus, Euro.ofEuros(unterliegen), anWiderklageBeteiligt, indices,
				baumbachBeteiligtenListe);
	}

	private Beteiligter getLetztenBeteiligten() {
		return this.beteiligtenAuswahlListe.getModel().getElementAt(beteiligtenAuswahlListe.getModel().getSize() - 1);
	}

}