/*
 * RtfWerkzeugKasten.java
 * eu.gronos.kostenrechner.controller.files (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.files;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;

import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextPane;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Document;
import javax.swing.text.EditorKit;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;
import javax.swing.text.TabSet;
import javax.swing.text.TabStop;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.rtf.RTFEditorKit;

import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.view.FontHelfer;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenTableModel;
import eu.gronos.kostenrechner.data.tenordaten.CaretRange;
import eu.gronos.kostenrechner.interfaces.Tabulierend;
import eu.gronos.kostenrechner.interfaces.Tenorierend;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.util.files.RTFStringCreator;
import eu.gronos.kostenrechner.view.KostenFileChooser;
import eu.gronos.kostenrechner.view.result.BegruendungsZahlenRenderer;
import eu.gronos.kostenrechner.view.result.StringButtonRenderer;

/**
 * Die Klasse ist ein {@link RTFEditorKit}, das zusätzlich Methoden
 * bereitstellt, um RTF-Daten in die Zwischenablage zu kopieren:
 * {@link #kopiereRtfInZwischenablage()}, und in eine Datei zu speichern:
 * {@link #speichereRTF()}. Zudem eine Klasse, damit ein RTF in die
 * Zwischenablage kopiert werden kann. Man kann auch Formatierungen setzen.
 * 
 * Defines the interface for classes that can be used to provide data for a
 * transfer operation. For information on using data transfer with Swing, see
 * How to Use Drag and Drop and Data Transfer, a section in The Java Tutorial,
 * for more information.
 *
 * @author Peter Schuster (setrok)
 * @date 15.08.2018
 * 
 * @url "http://lists.apple.com/archives/java-dev/2004/Jul/msg00359.html"
 * @url "http://docs.oracle.com/javase/tutorial/uiswing/dnd/index.html"
 *
 */
public class RtfWerkzeugKasten extends RTFEditorKit implements ClipboardOwner, Transferable {

	private static final long serialVersionUID = 2201251699302376878L;

	/**
	 * flavors speichert ein {@link DataFlavor}[] mit den DataFlavors. Das Array
	 * muss so zusammengesetzt sein, dass df[i] den passenden DataFlavor zum String
	 * s[i] enthält.
	 */
	private final DataFlavor[] flavors = new DataFlavor[] { new DataFlavor("text/rtf", "Rich Formatted Text"),
			new DataFlavor("text/plain", "Plain Text") };

	private String[] data;
	private KostenFileChooser fileChooser;

	private Document doc;

	private Tenorierend tenorierend;

	private final RTFStringCreator writer;// = new RTFStringCreator();

	private final boolean alsBruch;

	public static final KostenFileFilter DATEI_FILTER = new KostenFileFilter("tenor", "Rich Text Format (RTF)", "rtf");

	private static final String[] ARIALS = new String[] { "Arial", "Helvetica", "Helv" };

	/**
	 * @param alsBruch
	 */
	public RtfWerkzeugKasten(boolean alsBruch) {
		super();
		this.alsBruch = alsBruch;
		writer = new RTFStringCreator(alsBruch);
	}

	/**
	 * Die Methode fügt den Text der JTextPane in die Zwischenablage ein. Versucht
	 * das jetzt mit text/rtf!
	 * 
	 * @fixed auch hier funktionierte der Zeichensatz nicht.
	 */
	public void kopiereRtfInZwischenablage() {
		Clipboard zwischenAblage = Toolkit.getDefaultToolkit().getSystemClipboard();
		try {
			setTransferData(writer.createRTFString(tenorierend), getAllText());
			zwischenAblage.setContents(this, this);
		} catch (BadLocationException | UnsupportedEncodingException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		}
	}

	/**
	 * 
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard,
	 *      java.awt.datatransfer.Transferable)
	 */
	@Override
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
		Kostenrechner.getLogger()
				.info(String.format("Och nö, gehört mir doch die Zwischeablage (%s) nicht mehr!", clipboard.getName()));
	}

	/**
	 * Die Methode setzt in einer {@link JTextPane} das {@link HTMLEditorKit} und
	 * das {@link StyledDocument} und lädt
	 * 
	 * @param textPane die {@link JTextPane}
	 * @return das {@link StyledDocument}, namentlich das
	 *         {@link EditorKit#createDefaultDocument()} des {@link RTFEditorKit}s
	 * 
	 * @see HtmlWerkzeugKasten#createDefaultDocument(JTextPane)
	 * @see javax.swing.text.StyledEditorKit#install(JEditorPane)
	 */
	@Override
	public void install(JEditorPane textPane) {
		super.install(textPane);
		this.doc = this.createDefaultDocument();

		textPane.setEditable(false);

		createStyles();
	}

	/**
	 * Die Methode erstellt ein leeres neues {@link StyledDocument} und merkt es
	 * sich. Wenn bereits eines erstellt wurde, nimmt sie das zuvor erstellte (und
	 * erstellt kein neues).
	 * 
	 * @return das {@link StyledDocument} als {@link Document}
	 * 
	 * @see javax.swing.text.StyledEditorKit#createDefaultDocument()
	 */
	@Override
	public Document createDefaultDocument() {
		if (getDefaultStyledDocument() != null) {
			return getDefaultStyledDocument();
		}
		doc = super.createDefaultDocument();
		return getDefaultStyledDocument();
	}

	/**
	 * @return gibt das {@link #doc} als {@link StyledDocument} zurück
	 */
	public DefaultStyledDocument getDefaultStyledDocument() {
		return (DefaultStyledDocument) getDocument();
	}

	/**
	 * Die Methode speichereRTF dient dazu, einen {@link KostenFileChooser} zu
	 * erzeugen, der fürs Speichern von RTF-Dateien vorkonfiguriert ist, um den
	 * Inhalt der JTextPane als RTF zu speichern. Die Variante ohne Parameter soll
	 * vom ActionListener der Schaltfläche selbst aufgerufen werden. Sie ruft
	 * nacheinander den Dateiauswahldialog auf, überprüft ob die Datei schon
	 * existiert und ruft dann die eigentliche Methode zum Speichern auf.
	 * 
	 * Write content from a document to the given stream in a format appropriate for
	 * this kind of content handler.
	 * 
	 * @return bei Erfolg das {@link File}, sonst <code>null</code>
	 */
//	public File speichereRTF(Component parent) {
	public Path speichereRTF(Component parent) {
		boolean erfolg = false;
//		File datei = null;
		Path datei = null;
		fileChooser = new KostenFileChooser(DATEI_FILTER,
				(Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(75000));

		int option = fileChooser.showSaveDialog(parent);
		if (option == JFileChooser.APPROVE_OPTION) {
//			datei = fileChooser.getSelectedFile().getAbsoluteFile();
			datei = fileChooser.getSelectedPath().toAbsolutePath();
			// Existenz wird schon vom fileChooser geprüft
			erfolg = write(datei);
		}
		fileChooser = null;
		if (erfolg)
			return datei;
		else
			return null;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be
	 * provided in. The array should be ordered according to preference for
	 * providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return an array of data flavors in which this data can be transferred
	 * 
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 */
	@Override
	public DataFlavor[] getTransferDataFlavors() {
		return flavors;
	}

	/**
	 * @return gibt {@link #doc} als {@link Document} zurück.
	 */
	public Document getDocument() {
		return doc;
	}

	/**
	 * Returns whether or not the specified data flavor is supported for this
	 * object.
	 * 
	 * @param flavor the requested flavor for the data
	 * @return boolean indicating whether or not the data flavor is supported
	 * 
	 * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
	 */
	@Override
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		for (DataFlavor df : getTransferDataFlavors())
			if (df.isMimeTypeEqual(flavor))
				return true;
		return false;
	}

	/**
	 * Returns an object which represents the data to be transferred. The class of
	 * the object returned is defined by the representation class of the flavor.
	 * 
	 * @param flavor the requested flavor for the data
	 * @return an object which represents the data to be transferred. The class of
	 *         the object returned is defined by the representation class of the
	 *         flavor.
	 * @throws UnsupportedFlavorException
	 * @throws IOException
	 * 
	 * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
	 */
	@Override
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
		for (int i = 0; i < flavors.length; i++)
			if (flavors[i].isMimeTypeEqual(flavor))
				return new ByteArrayInputStream(data[i].getBytes());
		throw new UnsupportedFlavorException(flavor);
	}

	/**
	 * @return gibt {@link #fileChooser} als {@link KostenFileChooser} zurück.
	 */
	public KostenFileChooser getFileChooser() {
		return fileChooser;
	}

	/**
	 * Die Methode dient dazu, den Tenor der übergegebenen Tenorklasse im JTextPane
	 * mit der passenden Schriftart und Formatierungen zu setzen. Als Dialogtitel
	 * wird die Tenorbeschreibung des Tenorierend gesetzt.
	 * 
	 * @param str               den Text des Tenors als {@link String}
	 * @param title             {@link Tenorierend#getTenorBeschreibung()} als
	 *                          {@link String}
	 * @param grosseTabulatoren ein <code>boolean</code>, ob große Tabulatorabstände
	 *                          eingebaut werden sollen
	 * @throws BadLocationException wenn der Einfügepunkt kein gültiger Punkt im
	 *                              {@link Document} ist.
	 * 
	 * @see javax.swing.text.Document#insertString(int offset, String str,
	 *      AttributeSet a)
	 * @see javax.swing.text.Document#putProperty(Object key, Object value)
	 */
	public void insertString(String str, String title, boolean grosseTabulatoren) throws BadLocationException {
		writer.setTenor(str);
		writer.setTitle(title);
		doc.remove(0, doc.getLength());
		doc.insertString(0, str, ((StyledDocument) doc).getStyle("base"));
		// Die Beschreibung auch beim Speichern als Titel setzen
		doc.putProperty(Document.TitleProperty, title);
		setParagraphAttributes(0, doc.getLength(), grosseTabulatoren);
	}

	/**
	 * Die Methode fügt weiteren Text ins {@link #getDefaultStyledDocument()} ein,
	 * vorzugsweise die Begründung.
	 * 
	 * @param str                 die Gründe
	 * @param tenorierend         ein {@link Tenorierend}. Wenn es auch ein
	 *                            {@link Tabulierend} ist, wird auch die
	 *                            Begründungs-Tabelle eingefügt.
	 * @param paneSize            eine {@link Dimension} mit der
	 *                            {@link JComponent#getPreferredSize()} der
	 *                            {@link JTable}
	 * @param gruendeUeberschrift normalerweise
	 *                            {@link TenorTexter#GRUENDE_UEBERSCHRIFT}
	 * @param alsBruch            ob Kostenquoten (in der Tabelle) als Bruch
	 *                            dargestellt werden. Sonst werden sie als Prozent
	 *                            geschrieben
	 * @throws BadLocationException wenn der {@link Document#getLength()} kein
	 *                              gültiger Einfügepunkt im {@link Document} ist
	 */
	public int insertString(String str, Tenorierend tenorierend, Dimension paneSize, String gruendeUeberschrift)
			throws BadLocationException {
		// ,boolean alsBruch
		this.tenorierend = tenorierend;
		writer.setGruende(str);
		writer.setGruendeUeberschrift(gruendeUeberschrift);
		int anfang = doc.getLength();
		doc.insertString(anfang, writer.getGruendeUeberschrift(), getDefaultStyledDocument().getStyle("bold"));

		int nachUeberschrift = doc.getLength();
		doc.insertString(nachUeberschrift, str, getDefaultStyledDocument().getStyle("base"));

		Kostenrechner.getLogger().info(String.format("doc.Length()-anfang: %04d", (doc.getLength() - anfang)));

		// an dieser Stelle ist das vorerst besser
		if (tenorierend instanceof Tabulierend) {
			createTableStyle(baueGruendeTabelle((Tabulierend) tenorierend, paneSize, alsBruch));
			gruendeTabelleEinpassen(str, (Tabulierend) tenorierend, nachUeberschrift);
		}

		return nachUeberschrift;
	}

	/**
	 * Die Methode speichereRTF dient dazu, die <code>datei</code> wirklich zu
	 * speichern. Dazu wird {@link #createRTFString(Tenorierend)} benutzt.
	 * 
	 * @param datei die Datei, in die gespeichert werden soll
	 * @return hat's geklappt, dann true TODO Muss {@link File} oder {@link Path}
	 *         zurückgeben
	 * 
	 * @url {@link http://stackoverflow.com/questions/2725141/java-jtextpane-rtf-save}
	 * @url {@link http://openbook.rheinwerk-verlag.de/javainsel/07_006.html#u7.6.3}
	 *      (try with resources)
	 * 
	 * @see javax.swing.text.rtf.RTFEditorKit#write(OutputStream,Document,int,int)
	 */
//	private boolean write(File datei) {
	private boolean write(Path datei) {
		boolean erfolg = false;
//		try (FileWriter fw = new FileWriter(datei); BufferedWriter bw = new BufferedWriter(fw);
		try (BufferedWriter bw = Files.newBufferedWriter(datei, StandardCharsets.UTF_8)) {
			bw.write(writer.createRTFString(tenorierend));
			erfolg = true;
		} catch (FileNotFoundException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		} catch (IOException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
		}

		return erfolg;
	}

	/**
	 * Die Methode holt sich dem vollständigen Text der JTextPane
	 * 
	 * @return den gesamten Text der {@link JTextPane} als {@link String}
	 * @throws BadLocationException
	 */
	private String getAllText() throws BadLocationException {
		return doc.getText(doc.getStartPosition().getOffset(), doc.getLength());
	}

	/**
	 * Die Methode, die die String[] aufnehmen kann.
	 * 
	 * @param data ein String[] mit den Inhalten. Das Array muss so zusammengesetzt
	 *             sein, dass flavors[i] den passenden DataFlavor zum String data[i]
	 *             enthält.
	 */
	private void setTransferData(String... data) {
		this.data = data;
	}

	/**
	 * Die Methode erstellt die {@link Style}s für das {@link StyledDocument}
	 * 
	 * @return den baseStyle
	 */
	private Style createStyles() {
		DefaultStyledDocument doc = getDefaultStyledDocument();

		Style baseStyle = createBaseStyle();
		createBoldStyle(doc, baseStyle);
		createTabStoppedStyle(doc, baseStyle);
		return baseStyle;
	}

	/**
	 * Die Methode definiert erstmal den grundlegenden Style ("base"), wobei diese
	 * direkt ins übergebene {@link StyledDocument} geschrieben werden. Zusätzlich
	 * wird der Style noch zurückgegeben.
	 * 
	 * @return ein {@link Style} mit den grundlegenden Formatierungen
	 * @see #createBaseStyle()
	 */
	private Style createBaseStyle() {
		Style baseStyle = getDefaultStyledDocument().addStyle("base", null);
		setFontArial(baseStyle);
		setFontSize(baseStyle);

		StyleConstants.setLineSpacing(baseStyle, .33f);
		StyleConstants.setSpaceBelow(baseStyle, 7f);

		return baseStyle;
	}

	/**
	 * Die Methode baut einen weiteren Style für fetten Text und schreibt ihn ins
	 * übergebene {@link StyledDocument}. Zusätzlich wird der Style noch
	 * zurückgegeben.
	 * 
	 * @param doc       ein {@link StyledDocument}
	 * @param baseStyle ein {@link Style} mit den grundlegenden Formatierungen
	 * @return ein {@link Style} mit den neuen Formatierungen
	 * @see #createBaseStyle()
	 */
	private Style createBoldStyle(StyledDocument doc, Style baseStyle) {

		Style style = doc.addStyle("bold", baseStyle);
		StyleConstants.setBold(style, true);

		return style;
	}

	/**
	 * Die Methode baut einen Style mit Tabstopps ("tabulatoren") und schreibt ihn
	 * ins übergebene {@link StyledDocument}. Zusätzlich wird der Style noch
	 * zurückgegeben.
	 * 
	 * @param doc       ein {@link StyledDocument}
	 * @param baseStyle ein {@link Style} mit den Formatierungen
	 * @return ein {@link Style} mit den neuen Formatierungen
	 * @url "http://www.java2s.com/Tutorial/Java/0240__Swing/TabStopandTabSetClasses.htm"
	 *      "http://www.java2s.com/Code/Java/Swing-JFC/Createadecimalalignedtabstopat400pixelsfromtheleftmargin.htm"
	 * @see #createBaseStyle()
	 */
	private Style createTabStoppedStyle(StyledDocument doc, Style baseStyle) {
		Style style = doc.addStyle("tabulatoren", baseStyle);
		StyleConstants.setTabSet(style,
				new TabSet(new TabStop[] { new TabStop(600, TabStop.ALIGN_RIGHT, TabStop.LEAD_DOTS),
						new TabStop(950, TabStop.ALIGN_RIGHT, TabStop.LEAD_DOTS) }));

		return style;
	}

	/**
	 * Die Methode definiert einen Style mit einer JTable ("table"), sofern das
	 * übergebene {@link Tenorierend} auch das Interface {@link Tabulierend}
	 * implementiert
	 * 
	 * @param table eine {@link JTable}, die in den {@link Style} eingefügt wird
	 * @return der {@link Style} mit {@link JTable}
	 * @url "http://www.java-forums.org/awt-swing/2260-how-insert-tables-into-jtextpane.html#post4296"
	 */
	private Style createTableStyle(JTable table) {
		Style style = getDefaultStyledDocument().addStyle("table", getDefaultStyledDocument().getStyle("base"));
		JScrollPane jScrollPane = new JScrollPane(table);
		StyleConstants.setComponent(style, jScrollPane);
		return style;
	}

	/**
	 * Die Methode dient dazu, eine Begründungstabelle einzufügen, wenn möglich
	 * 
	 * @param gruende die Begründung als {@link String}
	 * @param offset  hier soll das Einsetzen anfangen als <code>int</code>
	 * 
	 * @throws BadLocationException
	 * 
	 * @FIXED: Folgendes sorgte für eine NullPointerExeption beim
	 *         Speichern/Kopieren: setParagraphAttributes(doc, anfang,
	 *         doc.getLength() - anfang, false); Wohl wegen doc.getLenght-anfang
	 * 
	 *         {@url http://stackoverflow.com/questions/2600960/nullpointerexception-in-javax-swing-text-simpleattributeset-addattribute}
	 */
	private void gruendeTabelleEinpassen(final String gruende, Tabulierend tab, int offset)
			throws BadLocationException {
		// Wenn eine Begründungstabelle erzeugt werden kann, dann diese einfügen.
		final CaretRange range = tab.getRange();
		int anfangGruende = offset + range.beginn + 1;
		int laenge = range.ende - range.beginn;
		// Damit es keine Dopplungen gibt, erst einmal den alten Text herausnehmen
		getDefaultStyledDocument().remove(anfangGruende, laenge - 1);
		// Dann den neuen Text mit der richtigen Formatierung (d.h. mit JTable in
		// JScrollpane) einfügen
		getDefaultStyledDocument().insertString(anfangGruende, gruende.substring(range.beginn + 1, range.ende),
				getDefaultStyledDocument().getStyle("table"));
		Kostenrechner.getLogger()
				.info(String.format("anfangGruende: %04d, doc.getLength(): %04d, range: (%04d, %04d); laenge: %04d",
						anfangGruende, getDefaultStyledDocument().getLength(), range.beginn, range.ende, laenge));
	}

	/**
	 * Die Methode dient dazu, mit den Tabellendaten eines {@link Tabulierend} eine
	 * JTable zu bauen und mit den passenden Renderern zu versehen.
	 * 
	 * @param tabulierend das {@link Tabulierend}, in dem die Begründung steckt
	 * @param paneSize    die bevorzugte Größe ({@link Dimension}) des
	 *                    {@link JTextPane}s
	 * @param alsBruch    ob Kostenquoten als Bruch dargestellt werden. Sonst werden
	 *                    sie als Prozent geschrieben
	 * @return die gebaute JTable
	 */
	private JTable baueGruendeTabelle(Tabulierend tabulierend, Dimension paneSize, boolean alsBruch) {
		JTable table = new JTable(new BegruendungsZahlenTableModel(tabulierend));

		Dimension d = table.getPreferredSize();
		d.width = paneSize.width;
		table.setPreferredScrollableViewportSize(d);
//		table.setDefaultRenderer(Double.class, new NachkommastellenRenderer());
		table.setDefaultRenderer(Number.class, new BegruendungsZahlenRenderer(alsBruch));
		table.setDefaultRenderer(String.class, new StringButtonRenderer());
		return table;
	}

	/**
	 * Die Methode dient dazu, die Schriftart Arial , sonst eine verwandte
	 * Schriftart aus den Schriftarten des Systems zu finden. Sie geht nacheinander
	 * die Schriftnamen durch. Ist einer davon der aktuelle Font, gewinnt dieser
	 * immer, sonst wird einer der anderen aus der Liste genommen, sofern er
	 * existiert.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 */
	private boolean setFontArial(MutableAttributeSet attrib) {
		final FontHelfer helper = new FontHelfer();
		boolean ok = false;
		final String ff = StyleConstants.getFontFamily(attrib);
		for (String arial : ARIALS) {
			if (helper.getFontForName(arial) != null) {
				writer.setArial(arial);
			}
			if (arial.equals(ff)) {
				StyleConstants.setFontFamily(attrib, arial);
				ok = true;
			} else if (!ok && helper.getFontForName(arial) != null) {
				StyleConstants.setFontFamily(attrib, arial);
				ok = true;
			}
		}

		return ok;
	}

	/**
	 * Die Methode findet die Schriftgröße heraus. Die Schriftgröße wird auf 12px
	 * gesetzt, solange nicht die Systemschriftgröße größer ist.
	 * 
	 * @param attrib ein {@link MutableAttributeSet} oder {@link Style}
	 */
	private void setFontSize(MutableAttributeSet attrib) {
		final Font font = baseFont();

		// TODO Kann man herausfinden, ob die Schriftart die Größe unterstützt?
		// Nimmt jetzt auch den Zoom-Faktor
		final int fs = (int) (StyleConstants.getFontSize(attrib) * FontHelfer.zoom);
		StyleConstants.setFontSize(attrib, fs > font.getSize() ? fs : font.getSize());
	}

	/**
	 * @return gibt den UI-Standard für EditorPane.font als {@link Font} zurück
	 */
	private Font baseFont() {
		UIDefaults uid = UIManager.getLookAndFeel().getDefaults();
		return uid.getFont("EditorPane.font");
	}

	/**
	 * Die Methode setzt die Absatzformatierung auf anderthalbfachen
	 * Zeilenabsabstand (ok, 1,33-fach) und 7px Abstand danach.
	 * 
	 * @param offset            the start of the change >= 0
	 * @param length            the length of the change >= 0
	 * @param grosseTabulatoren sollen große Tabulatorabstände eingebaut werden?
	 * 
	 * @see javax.swing.JTextPane#getParagraphAttributes()
	 */
	private void setParagraphAttributes(int offset, int length, boolean grosseTabulatoren) {
		if (grosseTabulatoren)
			getDefaultStyledDocument().setParagraphAttributes(offset, length,
					getDefaultStyledDocument().getStyle("tabulatoren"), false);
		else
			getDefaultStyledDocument().setParagraphAttributes(offset, length,
					getDefaultStyledDocument().getStyle("base"), false);
	}

}
