package eu.gronos.kostenrechner.controller.files;

import java.io.IOException;
import java.net.URI;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.logging.Level;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import com.sun.org.apache.xerces.internal.parsers.XMLParser;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.GenusNumerus;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.util.files.XJustizErrorHandler;

/**
 * Verarbeitet die Ereignisse eines {@link XMLParser} für eine Xjustiz-Datei
 * 
 * @author Peter Schuster (setrok)
 * @date 06.07.2014
 * 
 */
public class XjustizContentHandler implements ContentHandler {

	private static final String INSTANZDATEN_TAG = "Instanzdaten";
	private static final String BETEILIGUNG_TAG = "Beteiligung";
	private static final String ROLLE_TAG = "Rolle";
	private static final String ROLLENBEZEICHNUNG_TAG = "Rollenbezeichnung";
	private static final String NR_TAG = "Nr";
	private static final String BETEILIGTER_TAG = "Beteiligter";
	private static final String GESCHLECHT_TAG = "Geschlecht";
	private static final String CONTENT_TAG = "content";
	private static final String GEGENSTANDSWERT_TAG = "Gegenstandswert";
	private static final String ZAHL_TAG = "Zahl";
	private static final String RECHTSFORM_TAG = "Rechtsform";
	private static final String AKTENZEICHEN_TAG = "Aktenzeichen";

	private final static String KLAEGER_CONTENT = "Kläger(in)";
	private final static String BEKLAGTER_CONTENT = "Beklagte(r)";
	private final static String DRITTWIDERBEKLAGTE_CONTENT = "Drittwiderbeklagte(r)";
	private final static String WIDERBEKLAGTE_CONTENT = "Widerbeklagte(r)";
	private final static String WIDERKLAEGER_CONTENT = "Widerkläger(in)";
	private final static String DRITTWIDERKLAEGER_CONTENT = "Drittwiderkläger(in)";
	private final static String MAENNLICH_CONTENT = "männlich";
	private final static String WEIBLICH_CONTENT = "weiblich";

	private XMLReader xmlReader;
	private String xmlDatei;
	private int betZaehler;
	private boolean betOffen;
	private boolean rolleOffen;
	private ArrayList<String> beteiligterElemente;
	private ArrayList<String> rolleElemente;
	private boolean betrOffen;
	private String aktGenus;
	private ArrayList<String> aktRollen;
	private ArrayList<Integer> nrRollen;
	private Locator locator;
	private StringBuffer stringBuffer;
	private boolean instOffen;
	private ArrayList<String> instanzElemente;
	private Euro gegenstandswert;
	private BaumbachBeteiligtenListe klaeger;
	private BaumbachBeteiligtenListe drittwiderbeklagte;
	private BaumbachBeteiligtenListe beklagte;
	private String rechtsform;
	private String aktenzeichen;
	public static final KostenFileFilter XJUSTIZ_FILTER = new KostenFileFilter(null, "Xjustiz (XML)", "xml");

	/**
	 * Baut einen {@link ContentHandler} für Xjustiz-Dateien und setzt dazu einen
	 * {@link org.xml.sax.XMLReader XmlReader}
	 * 
	 * @param uri einen String mit der URI der Datei (file:// usw)
	 * 
	 * @throws ParserConfigurationException Indicates a serious configuration error.
	 * @throws SAXException                 Encapsulate a general SAX error or
	 *                                      warning.
	 */
	public XjustizContentHandler(String uri) throws ParserConfigurationException, SAXException {
		if (uri == null)
			xmlDatei = getClass().getClassLoader().getResource("resources/xjustizimport.xml").toString();
		else
			xmlDatei = uri;
		xmlReader = setzeParser();
	}

	/**
	 * Die Methode baut eine {@link URI} als String aus einem absoluten Dateipfad
	 * 
	 * @param pfad ein File
	 * @return einen String mit der URI der Datei (file:// usw)
	 */
	public static String vonPathNachURI(Path pfad) {
		if (pfad == null)
			return null;
		return pfad.toAbsolutePath().toUri().toString();
	}

//	public static String vonFileNachURI(File file) {
//		if (file == null)
//			return null;
//		String path = file.getAbsolutePath();
//		if (File.separatorChar != '/')
//			path = path.replace(File.separatorChar, '/');
//		if (!path.startsWith("/"))
//			path = "/" + path;
//		return "file:" + path;
//	}

	/**
	 * Erstellt eine SAXParserFactory, holt sich einen SAXParser von dieser und von
	 * diesem wiederum einen XMLReader
	 * 
	 * @return den so ermittelten XMLReader
	 * @throws ParserConfigurationException Indicates a serious configuration error.
	 *                                      If a parser cannot be created which
	 *                                      satisfies the requested configuration.
	 * @throws SAXException                 Encapsulate a general SAX error or
	 *                                      warning. If any SAX errors occur during
	 *                                      processing.
	 */
	private XMLReader setzeParser() throws ParserConfigurationException, SAXException {
		XMLReader xmlReader = null;
		// Eine Instanz der SAXParserFactory erstellen (die ermittelt die
		// richtige Art, XML zu parsen aus den Systemeinstellungen) und auf
		// Namespare-Support einschwören
		SAXParserFactory spf = SAXParserFactory.newInstance();
		spf.setNamespaceAware(true);
		// Instanz von SAXParser von der ParserFactory holen = Wrapperklasse für
		// Parser
		SAXParser saxParser = spf.newSAXParser();
		// und schließlich einen XMLReader
		xmlReader = saxParser.getXMLReader();
		xmlReader.setContentHandler(this);
		xmlReader.setErrorHandler(new XJustizErrorHandler());
		return xmlReader;
	}

	/**
	 * Die Methode löscht die gepufferten Beteiligtendaten
	 * 
	 */
	public void clearRecent() {
		aktGenus = null;
		rechtsform = null;
		if (aktRollen == null)
			aktRollen = new ArrayList<String>();
		else
			aktRollen.clear();
		if (nrRollen == null)
			nrRollen = new ArrayList<Integer>();
		else
			nrRollen.clear();
		stringBuffer = new StringBuffer();
	}

	/**
	 * Prüft, ob die übergebene <code>rechtsform</code> einer der Rechtsformen aus
	 * <code>WL_Rechtsform</code> entspricht, die grammatikalisch männlich sind. Der
	 * Vergleich erfolgt über {@link java.lang.String#equals(Object) equals}, so
	 * dass ein ggfls. nötiges {@link java.lang.String#trim() trim} schon vorher
	 * stattfinden muss.
	 * 
	 * @param rechtsform die Rechtsform
	 * @return true, wenn die Rechtsform in der Liste enthalten und grammatikalisch
	 *         männlich ist.
	 */
	private boolean istRechtsformGrammatikalischMaeenlich(String rechtsform) {
		boolean zwischen = false;
		for (String rf : new String[] { "B", "e.V.", "e.V. iL", "eK", "GemVerb", "GVV", "IV", "Kr", "KV", "L", "LV",
				"NaKV", "NaP", "Pfl", "Seq", "TV", "V", "VEB", "VglV", "VIV", "VVaG", "ZV" })
			if (rf.equals(rechtsform))
				zwischen = true;
		return zwischen;
	}

	/**
	 * Schmeißt den Parser an.
	 * 
	 * @return die ausgelesene BaumbachBeteiligtenListen, streitwert und
	 *         Aktenzeichen als {@link VerfahrensDatenContainer}
	 * 
	 * @throws IOException
	 * @throws SAXException
	 */
	public VerfahrensDatenContainer parseXjustiz() throws IOException, SAXException {
		xmlReader.parse(xmlDatei);
		return new VerfahrensDatenContainer(klaeger, drittwiderbeklagte, beklagte, gegenstandswert, aktenzeichen,
				new XmlTransferHandler().speicherDatum());
	}

	/**
	 * Die Methode wird vom Parser aufgerufen, wenn das XML-Dokument beginnt
	 * 
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#startDocument()
	 */
	@Override
	public void startDocument() throws SAXException {
		betZaehler = 0;
		clearRecent();
		betOffen = false;
		rolleOffen = false;
		beteiligterElemente = new ArrayList<String>();
		rolleElemente = new ArrayList<String>();
		klaeger = new BaumbachBeteiligtenListe();
		drittwiderbeklagte = new BaumbachBeteiligtenListe();
		beklagte = new BaumbachBeteiligtenListe();
		gegenstandswert = null;// -1;
		aktenzeichen = null;
		rechtsform = null;
	}

	/**
	 * Die Methode wird vom Parser aufgerufen, wenn das Ende des Dokuments erreicht
	 * ist.
	 * 
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#endDocument()
	 */
	@Override
	public void endDocument() throws SAXException {
		String str = "";
		if (betOffen) {
			str = str + "Eine Beteiligung war noch offen! ";
		}
		str += String.format("Anzahl der Beteiligungen: %d", betZaehler);
		Kostenrechner.getLogger().info(str);
		if (klaeger != null)
			Collections.sort(klaeger);
		if (drittwiderbeklagte != null)
			Collections.sort(drittwiderbeklagte);
		if (beklagte != null)
			Collections.sort(beklagte);
	}

	/**
	 * Die Methode wird vom Parser bei jedem neuen XML-Element aufgerufen. Bei
	 * XJustiz interessiert mich das <code>&lt;Beteiligung&gt;</code> Element. Dann
	 * wird das Flag <code>betOffen</code> gesetzt, um die Ohren aufzuhalten für die
	 * Unterelemente <code>&ltRolle&gt;</code> (hierin steht der Beteiligtentyp) und
	 * <code>&lt;Beteiligter&gt;</code> (hierin steht das Geschlecht).
	 * 
	 * @param uri       the Namespace URI, or the empty string if the element has no
	 *                  Namespace URI or if Namespace processing is not being
	 *                  performed
	 * @param localName the local name (without prefix), or the empty string if
	 *                  Namespace processing is not being performed
	 * @param qName     the qualified name (with prefix), or the empty string if
	 *                  qualified names are not available
	 * @param atts      the attributes attached to the element. If there are no
	 *                  attributes, it shall be an empty Attributes object. The
	 *                  value of this object after startElement returns is undefined
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#startElement(java.lang.String,
	 *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	@Override
	public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
		// stringBuffer bei jedem Element leeren
		stringBuffer = new StringBuffer();
		if (BETEILIGUNG_TAG.equals(localName)) {
			betOffen = true;
			betZaehler++;
			if (locator != null)
				System.out.printf("Beteiligung Nr. %d beginnt bei Zeile %d, Spalte %d.%n", betZaehler,
						locator.getLineNumber(), locator.getColumnNumber() - localName.length() - 2);
		} else if (betOffen) {
			if (ROLLE_TAG.equals(localName)) {
				rolleOffen = true;
				rolleElemente = new ArrayList<String>();
			}
			if (rolleOffen && rolleElemente != null) {
				rolleElemente.add(localName);
			}
			if (!rolleOffen && BETEILIGTER_TAG.equals(localName)) {
				betrOffen = true;
				beteiligterElemente = new ArrayList<String>();
				beteiligterElemente.add(localName);
			}
			if (betrOffen && beteiligterElemente != null) {
				beteiligterElemente.add(localName);
			}
		}
		if (INSTANZDATEN_TAG.equals(localName)) {
			instOffen = true;
			if (instanzElemente == null)
				instanzElemente = new ArrayList<String>();
			else
				instanzElemente.clear();
		} else if (instOffen && instanzElemente != null) {
			instanzElemente.add(localName);
		}
	}

	/**
	 * Die Methode wird bei einem Abschluss-Element aufgerufen. Laut Java Tutorial
	 * soll hier das character-Handling geschehen - aber wie?
	 * 
	 * @param uri       the Namespace URI, or the empty string if the element has no
	 *                  Namespace URI or if Namespace processing is not being
	 *                  performed
	 * @param localName the local name (without prefix), or the empty string if
	 *                  Namespace processing is not being performed
	 * @param qName     the qualified XML name (with prefix), or the empty string if
	 *                  qualified names are not available
	 * @throws SAXException any SAX exception, possibly wrapping another exception
	 * 
	 * @see org.xml.sax.ContentHandler#endElement(java.lang.String,
	 *      java.lang.String, java.lang.String)
	 */
	@Override
	public void endElement(String uri, String localName, String qName) throws SAXException {
		String content = stringBuffer.toString().trim();
		if (BETEILIGUNG_TAG.equals(localName)) {
			betOffen = false;
			rolleOffen = false;
			betrOffen = false;
			rolleElemente.clear();
			beteiligterElemente.clear();
			BaumbachBeteiligter bb = baueBeteiligten();
			if (bb != null) {
				// Einer der BaumbachBeteiligtenlisten hinzufügen
				switch (bb.getTyp()) {
				case KLAEGER:
					klaeger.add(bb);
					System.out.println("Kläger/in gefunden!");
					break;
				case DRITTWIDERBEKLAGTE:
					drittwiderbeklagte.add(bb);
					System.out.println("Drittwiderbeklagte/n gefunden!");
					break;
				case BEKLAGTE:
				default:
					beklagte.add(bb);
					System.out.println("Beklagte/n gefunden!");
					break;
				}
			}
			clearRecent();
			if (locator != null) {
				String str = String.format("Beteiligung Nr. %d endet bei Zeile %d, Spalte %d.%n", betZaehler,
						locator.getLineNumber(), locator.getColumnNumber());
				Kostenrechner.getLogger().info(str);
			}
		}
		if (ROLLE_TAG.equals(localName)) {
			rolleOffen = false;
			rolleElemente.clear();
		}
		if (BETEILIGTER_TAG.equals(localName)) {
			betrOffen = false;
			beteiligterElemente.clear();
		}
		if (ZAHL_TAG.equals(localName)) {
			if (instOffen && instanzElemente != null && instanzElemente.size() > 1) {
				int size = instanzElemente.size();
				if (GEGENSTANDSWERT_TAG.equals(instanzElemente.get(size - 2)))
					Kostenrechner.getLogger().info("Gegenstandswert: " + content);
				try {
					gegenstandswert = Euro.ofEuros(Double.parseDouble(content));
				} catch (NumberFormatException e) {
					Kostenrechner.getLogger().log(Level.WARNING,
							"Gegenstandswert konnte nicht konvertiert werden: " + e.getLocalizedMessage(), e);
				}
			}
		}
		if (AKTENZEICHEN_TAG.equals(localName)) {
			if (instOffen) {
				if (aktenzeichen == null) {
					aktenzeichen = content;
				} else {
					aktenzeichen += (" bzw. " + content);
				}
				Kostenrechner.getLogger().info("Aktenzeichen: " + aktenzeichen);
			}
		}
		if (INSTANZDATEN_TAG.equals(localName)) {
			instOffen = false;
			instanzElemente.clear();
		}
		if (NR_TAG.equals(localName)) {
			// Laufende Nummer auslesen und gucken, dass sie zur richtigen
			// Rollenbezeichnung kommt
			if (betOffen && rolleOffen) {
				int parseInt;
				try {
					parseInt = Integer.parseInt(content);
				} catch (NumberFormatException e) {
					Kostenrechner.getLogger().log(Level.WARNING, e.getLocalizedMessage());
					parseInt = nrRollen.size();
				}
				if (nrRollen.size() >= aktRollen.size() && nrRollen.get(nrRollen.size() - 1) == null)
					nrRollen.set(nrRollen.size() - 1, parseInt);
				else
					nrRollen.add(parseInt);
			}
		}
		// Hier den stringBuffer auslesen, wenn <content>
		// String content = stringBuffer.toString().trim();
		if (CONTENT_TAG.equals(localName)) {
			if (betOffen && rolleElemente != null && rolleElemente.size() > 1) {
				int size = rolleElemente.size();
				if (ROLLENBEZEICHNUNG_TAG.equals(rolleElemente.get(size - 2))) {
					if (content != null && content.length() > 0) {
						aktRollen.add(content);
						if (nrRollen.size() < aktRollen.size())
							nrRollen.add(null);
					}
				}
			}
			if (betrOffen && beteiligterElemente != null && beteiligterElemente.size() > 1) {
				int size = beteiligterElemente.size();
				if (GESCHLECHT_TAG.equals(beteiligterElemente.get(size - 2))) {
					if (content != null && content.length() > 0) {
						aktGenus = content;
					}
				}
				if (RECHTSFORM_TAG.equals(beteiligterElemente.get(size - 2))) {
					if (content != null && content.length() > 0) {
						rechtsform = content;
					}
				}
			}
		}
	}

	/**
	 * Die Methode characters wird vom Parser aufgerufen, wenn er lustig ist. Der
	 * Parser übergibt dabei alles von einzelnen Zeichen bis zu mehreren Tausend auf
	 * einmal, wie er lustig ist. Das Java Tutorial von Oracle empfiehlt daher,
	 * alles in einem StringBuffer zu sammeln und und diesen bei einem endElement zu
	 * verarbeiten.
	 * 
	 * @param ch     the characters from the XML document
	 * @param start  the start position in the array
	 * @param length the number of characters to read from the array
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#characters(char[], int, int)
	 */
	@Override
	public void characters(char[] ch, int start, int length) throws SAXException {
		// Um den Inhalt kümmern - so ist aber blöd, da lässt er die Elemente
		// selbst weg...
		if (stringBuffer == null)
			stringBuffer = new StringBuffer(String.copyValueOf(ch, start, length));
		else
			stringBuffer.append(ch, start, length);
	}

	/**
	 * Die Methode erstellt einen BaumbachBeteiligten aus den Daten, die beim Ende
	 * einer jeden Beteiligung vorliegen. Dies nur, sofern es sich um einen der von
	 * Beteiligter unterstützten Beteiligungsformen KLAEGER, DRITTWIDERBEKLAGTE(r)
	 * oder BEKLAGTE(r) handelt.
	 * 
	 * BeteiligtenTyp heraus finden: Wenn KLAEGER_CONTENT enthalten --> KLAEGER wenn
	 * noch WIDERBEKLAGTE_CONTENT enthalten --> anWiderklageBeteiligt = true wenn
	 * WIDERBEKLAGTE_CONTENT oder DRITTWIDERBEKLAGTE_CONTENT ohne KLAEGER_CONTENT
	 * --> dann DRITTWIDERBEKLAGTE Wenn BEKLAGTE_CONTENT enthalten --> BEKLAGTE wenn
	 * auch etwas WIDERKLAEGER_CONTENT oder DRITTWIDERKLAEGER_CONTENT enthalten -->
	 * anWiderklageBeteiligt = true
	 * 
	 * @return einen BaumbachBeteiligten oder null, wenn es einen anderen
	 *         Beteiligtentyp gab.
	 * 
	 */
	BaumbachBeteiligter baueBeteiligten() {
		BaumbachBeteiligter bb = null;
		// int -1;
		BeteiligtenTyp typ = null;
		// int -1;
		GenusNumerus genusNumerus = null;
		int lfdNr = -1;
		boolean anWiderklageBeteiligt = false;
		String zwischen = "";
		if (aktRollen.contains(DRITTWIDERBEKLAGTE_CONTENT) || aktRollen.contains(WIDERBEKLAGTE_CONTENT)
				|| aktRollen.contains(DRITTWIDERKLAEGER_CONTENT) || aktRollen.contains(WIDERKLAEGER_CONTENT))
			anWiderklageBeteiligt = true;
		if (aktRollen.contains(KLAEGER_CONTENT)) {
			typ = BeteiligtenTyp.KLAEGER;
			lfdNr = nrRollen.get(aktRollen.indexOf(KLAEGER_CONTENT));
		} else if (aktRollen.contains(BEKLAGTER_CONTENT)) {
			typ = BeteiligtenTyp.BEKLAGTE;
			lfdNr = nrRollen.get(aktRollen.indexOf(BEKLAGTER_CONTENT));
		} else if (aktRollen.contains(DRITTWIDERBEKLAGTE_CONTENT)) {
			typ = BeteiligtenTyp.DRITTWIDERBEKLAGTE;
			lfdNr = nrRollen.get(aktRollen.indexOf(DRITTWIDERBEKLAGTE_CONTENT));
		} else if (!aktRollen.contains(KLAEGER_CONTENT) && aktRollen.contains(WIDERBEKLAGTE_CONTENT)) {
			typ = BeteiligtenTyp.DRITTWIDERBEKLAGTE;
			lfdNr = nrRollen.get(aktRollen.indexOf(WIDERBEKLAGTE_CONTENT));
		}
		for (String st : aktRollen)
			zwischen += st + ", ";
		Kostenrechner.getLogger().info("Rolle(n): " + zwischen);
		// Jetzt das Geschlecht heraus finden
		// System.out.print("Geschlecht: ");
		if (MAENNLICH_CONTENT.equals(aktGenus)) {
			Kostenrechner.getLogger().info("Geschlecht: männlich.");
			genusNumerus = GenusNumerus.MAENNLICH_SINGULAR;
		} else if (WEIBLICH_CONTENT.equals(aktGenus)) {
			Kostenrechner.getLogger().info("Geschlecht: weiblich.");
			genusNumerus = GenusNumerus.WEIBLICH_SINGULAR;
		} else if (rechtsform != null && istRechtsformGrammatikalischMaeenlich(rechtsform)) {
			// Wenn das Tag nicht gefunden wird, auch auf Rechtsform gucken
			Kostenrechner.getLogger().info("Geschlecht: männlich (Organisation).");
			genusNumerus = GenusNumerus.MAENNLICH_SINGULAR;
		} else {
			Kostenrechner.getLogger().info("Geschlecht: unbekannt?");
			// Ohne "Geschlecht" ist es wohl eine juristische Person; die sind
			// meist grammatikalisch weiblich (AG, GmbH)
			genusNumerus = GenusNumerus.WEIBLICH_SINGULAR;
		}
		// Nur wenn der typ nicht < 0 ist, einen Beteiligten anlegen
		Euro streitwert;// double
		if (typ != null) {// >= 0
			Kostenrechner.getLogger()
					.info(String.format("Typ: %d, genusNumerus: %d, lfdNr: %d, anWiderklageBeteiligt: %s, ", typ,
							genusNumerus, lfdNr, anWiderklageBeteiligt));
			// Einen Streitwert bekommt nur, gegen wen sich auch etwas richtet
			if (typ == BeteiligtenTyp.BEKLAGTE || typ == BeteiligtenTyp.DRITTWIDERBEKLAGTE
					|| (typ == BeteiligtenTyp.KLAEGER && anWiderklageBeteiligt)) {
				streitwert = gegenstandswert;
				Kostenrechner.getLogger().info(String.format("Streitwert: %,.2f EUR.%n", gegenstandswert));
			} else {
				streitwert = Euro.ZERO_CENTS;// 0.0;
			}
			// BaumbachBeteiligten bauen
			bb = new BaumbachBeteiligter(typ, genusNumerus, streitwert, Euro.ZERO_CENTS,
					anWiderklageBeteiligt);
			bb.setLfdNr(lfdNr);
		}
		return bb;
	}

	/**
	 * Der Parser benutzt diese Methode, um dem ContentHandler einen
	 * {@link org.xml.sax.Locator Locator} mitzuteilen. Diesen speichere ich mir.
	 * Dann kann ich bei Ereignissen immer gucken, an welcher Stelle (Zeile, Spalte)
	 * der XML-Datei der Parser gerade ist.
	 * 
	 * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
	 */
	@Override
	public void setDocumentLocator(Locator locator) {
		Kostenrechner.getLogger().info("Folgender Locator wird gesetzt: " + locator.toString());
		this.locator = locator;
	}

	/**
	 * Die Methode tut nix. Sie soll wohl eigentlich dazu dienen, damit
	 * Element-Ereignisse feststellen können, in welchem Namensraum sich das Element
	 * befindet.
	 * 
	 * @param prefix the Namespace prefix being declared. An empty string is used
	 *               for the default element namespace, which has no prefix.
	 * @param uri    the Namespace URI the prefix is mapped to
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String,
	 *      java.lang.String)
	 */
	@Override
	public void startPrefixMapping(String prefix, String uri) throws SAXException {
		// do nothing
	}

	/**
	 * Die Methode tut nix.
	 * 
	 * @param prefix
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#endPrefixMapping(java.lang.String)
	 */
	@Override
	public void endPrefixMapping(String prefix) throws SAXException {
		// do nothing

	}

	/**
	 * Die Methode wird von validierenden Parsern aufgerufen, wenn Weißraum
	 * (whitespace) auftaucht, der nach dem Schema ignoriert werden darf. Hier tut
	 * sie nichts.
	 * 
	 * @param ch     the characters from the XML document
	 * @param start  the start position in the array
	 * @param length the number of characters to read from the array
	 * @throws SAXException
	 * 
	 * @see org.xml.sax.ContentHandler#ignorableWhitespace(char[], int, int)
	 */
	@Override
	public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
		// do nothing

	}

	/**
	 * Die Methode wird vom Parser aufgerufen, um eine "processing instruction"
	 * mitzuteilen, also die Tags mit dem Fragezeichen, etwa <?XML 1.0 >... Das kann
	 * der Parser aber ohnehin handhaben, wie er lustig ist. Also tut die Methode
	 * nichts.
	 * 
	 * @param target target the processing instruction target
	 * @param data   the processing instruction data, or null if none was supplied.
	 *               The data does not include any whitespace separating it from the
	 *               target
	 * @throws SAXException any SAX exception, possibly wrapping another exception
	 * 
	 * @see org.xml.sax.ContentHandler#processingInstruction(java.lang.String,
	 *      java.lang.String)
	 */
	@Override
	public void processingInstruction(String target, String data) throws SAXException {
		// do nothing

	}

	/**
	 * Die Methode tut nichts. The Parser will invoke this method each time the
	 * entity is skipped.
	 * 
	 * @param name the name of the skipped entity. If it is a parameter entity, the
	 *             name will begin with '%', and if it is the external DTD subset,
	 *             it will be the string "[dtd]"
	 * @throws SAXException any SAX exception, possibly wrapping another exception
	 * 
	 * @see org.xml.sax.ContentHandler#skippedEntity(java.lang.String)
	 */
	@Override
	public void skippedEntity(String name) throws SAXException {
		// do nothing

	}

}
