/*
 * GebuehrenBerechnungPruefer.java
 * eu.gronos.kostenrechner.controller.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.Component;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JTable;

import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAuflistung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.model.gebuehren.GebuehrenTableModel;
import eu.gronos.kostenrechner.util.gebuehren.GebuehrenBerechnungPruefer;

/**
 * Ein {@link ParsendUndBauend} für {@link GebuehrenBerechnung}
 *
 * @author Peter Schuster (setrok)
 * @date 30.08.2019
 *
 */
public class GebuehrenBerechnungParser implements ParsendUndBauend<GebuehrenBerechnung> {

	private GebuehrenBerechnung rueckgabewert;
	private JFormattedTextField textfield;
	private JTable table;
	/**
	 * wenn <code>true</code>, dann prüft der {@link ParsendUndBauend} noch weitere
	 * Plausibilitäten
	 */
	private final boolean streng;

	/**
	 * Konstruktor, der nur die Standard-Plausibilitäten prüft.
	 * 
	 */
	public GebuehrenBerechnungParser() {
		this(false);
	}

	/**
	 * Konstruktor.
	 * 
	 * @param streng boolean wenn <code>true</code>, dann prüft der
	 *               {@link ParsendUndBauend} noch weitere Plausibilitäten
	 */
	public GebuehrenBerechnungParser(boolean streng) {
		super();
		this.streng = streng;
	}

	/**
	 * 
	 * @param components braucht ein {@link JFormattedTextField} und eine
	 *                   {@link JTable}
	 * @return diesen {@link GebuehrenBerechnungParser}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#leseComponents(Component[])
	 */
	@Override
	public ParsendUndBauend<GebuehrenBerechnung> leseComponents(Component... components) {
		if (components != null) {
			for (Component comp : components) {
				JComponent component = (JComponent) comp;
				if ("21005".equals(component.getName())) {
					textfield = (JFormattedTextField) component;
				} else if ("22011".equals(component.getName())) {
					table = (JTable) component;
				}
			}
		}
		return this;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, nachdem er Felder
	 * ausgelesen hat.
	 * 
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * @throws NumberFormatException wenn der {@link NumberExtractor} bei einem
	 *                               {@link JFormattedTextField} eine solche
	 *                               Exception wirft.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() throws NumberFormatException, NullPointerException {
		// TODO nur strenge Variante ruft parseEingabe(rueckgabewert) auf
		try {
			if (table == null)
				throw new NullPointerException();
			rueckgabewert = leseFelder();
			if (streng)
				return parseEingabe(rueckgabewert);
		} catch (NumberFormatException | NullPointerException ne) {
			FehlerHelper.zeigeFehler(ne.getLocalizedMessage(), ne);
			return false;
		}
		return true;
	}

	/**
	 * Die Methode überprüft den {@link #getRueckgabewert()}, ohne Felder
	 * auszulesen, indem er ihr übergeben wird.
	 * 
	 * @param rueckgabewert die zu prüfende {@link UnterContainerKlasse} vom Typ
	 *                      {@link GebuehrenBerechnung}
	 * @return <code>true</code>, wenn die Plausibilitäten zufrieden sind
	 * @throws NullPointerException     wenn values null ist
	 * @throws IllegalArgumentException wenn streitwert kleiner 0 ist
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#parseEingabe(java.lang.Object)
	 */
	@Override
	public boolean parseEingabe(GebuehrenBerechnung rueckgabewert)
			throws NumberFormatException, NullPointerException, IllegalArgumentException {
		this.rueckgabewert = rueckgabewert;
		return new GebuehrenBerechnungPruefer().pruefeEingabe(rueckgabewert);
	}

	/**
	 * Die Methode dient dazu, die zusammen gepackten Programmzustände in einem
	 * {@link VerfahrensDatenContainer} zurückzugeben
	 * 
	 * @return den Untercontainer von
	 *         {@link VerfahrensDatenContainer#gebuehrenBerechnung}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.ParsendUndBauend#getRueckgabewert()
	 */
	@Override
	public GebuehrenBerechnung getRueckgabewert() {
		return rueckgabewert;
	}

	/**
	 * Die Methode liest die übergebenen Felder aus und setzt daraus die
	 * {@link UnterContainerKlasse} zusammen, die dann {@link #parseEingabe()}
	 * prüfen kann.
	 * 
	 * @return eine {@link GebuehrenBerechnung}
	 * @throws NumberFormatException wenn der {@link NumberExtractor} bei einem
	 *                               {@link JFormattedTextField} eine solche
	 *                               Exception wirft.
	 * @throws NullPointerException  nur zur Sicherheit...
	 * 
	 */
	private GebuehrenBerechnung leseFelder() throws NumberFormatException, NullPointerException {
		GebuehrenBerechnung rueckgabewert = new GebuehrenBerechnung();
		rueckgabewert.streitwert = Euro.ZERO_CENTS;// 0L;
		rueckgabewert.gebuehren = new GebuehrenAuflistung();
		final NumberExtractor extractor = new NumberExtractor(textfield,
				textfield.getAccessibleContext().getAccessibleName());
		rueckgabewert.streitwert = Euro.ofEuros(extractor.extractDouble());//(extractor.extractLong());
		System.out.println("Gelesener Streitwert: " + rueckgabewert.streitwert);
		GebuehrenTableModel model = (GebuehrenTableModel) table.getModel();
		rueckgabewert.gebuehren = model.getAllValues();
		return rueckgabewert;
	}

}
