/**
 * GebuehrenTransferable.java
 * eu.gronos.kostenrechner.controller.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.controller.gebuehren;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;

import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;

/**
 * Transferable für {@link GebuehrenSatzTatbestand GebührenTatbeständen}
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 04.08.2014
 */
public class GebuehrenTransferable implements Transferable {

	final private GebuehrenTatbestand gt;
	/**
	 * Der DataFlavor mit dem Namen GEBUEHREN_FLAVOR speichert einen
	 * {@link java.awt.datatransfer.DataFlavor DataFlavor} zum Umgang mit
	 * {@link GebuehrenSatzTatbestand GebührenTatbeständen}. Dazu nutzt er
	 * {@link DataFlavor.javaJVMLocalObjectMimeType}, um die Übergabe als nur für
	 * programminterne zu kennzeichnen.
	 * 
	 * @see java.awt.datatransfer.DataFlavor#javaJVMLocalObjectMimeType
	 */
	static final DataFlavor GEBUEHREN_FLAVOR = getDataFlavor();

	private static DataFlavor getDataFlavor() {
		try {
			return new DataFlavor(DataFlavor.javaJVMLocalObjectMimeType
					+ ";class=eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand");
		} catch (ClassNotFoundException e) {
			FehlerHelper.zeigeFehler(
					String.format("Klasse nicht gefunden: GebuehrenTatbestand (%s)", e.getLocalizedMessage()), e);
			return null;
		}
	}

	/**
	 * Erzeugt einen neuen Transferable für den übergebenen
	 * {@link GebuehrenSatzTatbestand}.
	 * 
	 * @param gt der zu übertragende GebührenTatbestand
	 * 
	 */
	public GebuehrenTransferable(GebuehrenTatbestand gt) {
		super();
		this.gt = gt;
	}

	/**
	 * Returns an object which represents the data to be transferred. The class of
	 * the object returned is defined by the representation class of the flavor.
	 * 
	 * @param flavor the requested flavor for the data
	 * @return einen {@link GebuehrenSatzTatbestand}
	 * @throws UnsupportedFlavorException if the requested data flavor is not
	 *                                    supported.
	 * @throws IOException                if the data is no longer available in the
	 *                                    requested flavor.
	 * 
	 * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
	 */
	@Override
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
		if (isDataFlavorSupported(flavor))
			return getGt();
		else
			throw new UnsupportedFlavorException(flavor);
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be
	 * provided in. The array should be ordered according to preference for
	 * providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return an array of data flavors in which this data can be transferred
	 * 
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 */
	@Override
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { GEBUEHREN_FLAVOR };
	}

	/**
	 * Returns whether or not the specified data flavor is supported for this
	 * object.
	 * 
	 * @param flavor the requested flavor for the data
	 * @return boolean indicating whether or not the data flavor is supported
	 * 
	 * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
	 */
	@Override
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		return GEBUEHREN_FLAVOR.equals(flavor);
	}

	/**
	 * @return the gt
	 */
	private GebuehrenTatbestand getGt() {
		return gt;
	}

}