package eu.gronos.kostenrechner.data.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.AKKUSATIV;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.GENITIV;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.NOMINATIV;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlIDREF;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.data.forderungen.Forderung;
import eu.gronos.kostenrechner.data.forderungen.KlageForderung;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.EuroAdapter;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.ProzessVerhaeltnis;
import eu.gronos.kostenrechner.interfaces.AntragErfolgElement;
import eu.gronos.kostenrechner.interfaces.EnumString;
import eu.gronos.kostenrechner.interfaces.ParteiBeziehung;
import eu.gronos.kostenrechner.logic.TenorToken;

/**
 * Angriffe zur Ermittlung der Kostenquote bei Baumbachscher Formel. Enthält
 * praktisch die bei {@link BaumbachBeteiligter}n über {@link Beteiligter}
 * hinausgehenenden Felder.
 *
 * TODO Später mit {@link ProzessVerhaeltnis} und
 * {@link HauptsacheVerhaeltnis} in Einklang bringen, auch mit {@link Forderung}
 * und {@link KlageForderung}
 * 
 * {@link Beteiligter}
 * 
 * @author Peter Schuster (setrok)
 * @date 09.10.2021
 *
 */
public class Angriff implements Comparable<Angriff>, ParteiBeziehung<List<Beteiligter>>, AntragErfolgElement {
    /**
     * Ein {@link Enum}, mit dem sich {@link #KLAGE} von {@link #WIDERKLAGE}
     * unterscheiden lässt.
     *
     * @author Peter Schuster (setrok)
     * @date 01.11.2021
     *
     */
    public enum AngriffArt implements EnumString<AngriffArt> {
        KLAGE("Klage"), WIDERKLAGE("Widerklage"), UNBESTIMMT("Klage-/Widerklage");

        private final String string;

        private AngriffArt(String string) {
            this.string = string;
        }

        /**
         * Die Methode gibt die {@link AngriffArt} als {@link String} aus, also Klage,
         * Widerklage oder bei Unbestimmtheit Klage-/Widerklage
         * 
         * @return einen {@link String} wie beschrieben
         * 
         * @see java.lang.Enum#toString()
         */
        @Override
        public String toString() {
            return string;
        }

        /**
         * Sucht die passende {@link AngriffArt} für den {@link Beteiligter#getTyp()}
         * 
         * @param typ {@link Beteiligter#getTyp()} des {@link Angriff#getAngreifer()}s
         * @return die zum Typ passende {@link AngriffArt}, also {@link #KLAGE} oder
         *         {@link #WIDERKLAGE}; {@link #UNBESTIMMT}, wenn irgendein anderer
         *         <code>int</code>.
         */
        public static AngriffArt forBeteiligterTyp(BeteiligtenTyp typ) {
            switch (typ) {
            case KLAEGER:
                return AngriffArt.KLAGE;
            case BEKLAGTE:
                return AngriffArt.WIDERKLAGE;
            default:
                return AngriffArt.UNBESTIMMT;
            }
        }
    }
	
	
	private List<Beteiligter> angreifer;

	private List<Beteiligter> gegner;

	private Euro antrag;

	private Euro erfolg;

	private boolean gesamtSchuldnerisch;

	public Angriff() {
		this.angreifer = new ArrayList<>();
		this.gegner = new ArrayList<>();
	}

	/**
	 * Die Methode formatiert einen {@link Angriff} als {@link String}, z.B. &quot;
	 * Kläger gegen [Beklagte-1 Beklagte-2] auf 0,00 EUR (Erfolg: 300,00 EUR).&quot;
	 * 
	 * 
	 * @return einen {@link String} im Format
	 *         &quot;<code>&lt;Angreifer> gegen &lt;Gegner> auf &lt;Streitwert> EUR (Erfolg: &lt;Erfolg> EUR)</code>
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder(formatTitle());
		;
		builder.append(formatErfolg());

		return builder.toString();
	}

	/**
	 * Die Methode formatiert den Titel eines {@link Angriff} als {@link String},
	 * z.B. &quot; Kläger gegen [Beklagte-1 Beklagte-2] auf 0,00 EUR&quot;
	 * 
	 * @return eine {@link CharSequence} im Format
	 *         &quot;<code>&lt;Angreifer> gegen &lt;Gegner> auf &lt;Streitwert> EUR</code>
	 */
	private CharSequence formatTitle() {
		StringBuilder builder = new StringBuilder(formatAngreifer());
		builder.append(TenorToken.GEGEN.toString() + TenorToken.LEER);
		builder.append(formatGegner());
		builder.append(formatAntrag());
		return builder;
	}

	/**
	 * 
	 * @return gibt {@link #angreifer} als {@link List<Beteiligter>} zurück. Lieber
	 *         List statt Set, denn Set hat keinen Getter...
	 * 
	 * @see eu.gronos.kostenrechnerjunits.ParteiBeziehung#getAngreifer()
	 */
	@Override
	// TODO oder Set<Beteiligter>? List<Beteiligter>?
	// TODO muss in JAXB ein @XmlIdRef sein - geht das auch mit Collections?
	@XmlElementWrapper(name = "angreiferlein")
	@XmlIDREF
	@XmlElement
	public List<Beteiligter> getAngreifer() {
		return angreifer;
	}

	/**
	 * 
	 * @param angreifer d. {@link #angreifer}, d. gesetzt werden soll als
	 *                  {@link List<Beteiligter>}.
	 * 
	 * @see eu.gronos.kostenrechnerjunits.ParteiBeziehung#setAngreifer(java.util.List)
	 */
	@Override
	public void setAngreifer(List<Beteiligter> angreifer) {
		this.angreifer.clear();
		this.angreifer.addAll(angreifer);
	}

	/**
	 * Die Methode formatiert einen {@link Angriff#getAngreifer()} als
	 * {@link String}, z.B. &quot; Kläger&quot;
	 * 
	 * @return eine {@link CharSequence} im Format &quot; Kläger-1&quot;
	 */
	private CharSequence formatAngreifer() {
		StringBuilder builder = new StringBuilder();
		for (Beteiligter bt : getAngreifer()) {
			builder.append(bt.kurzBezeichner(GENITIV));
			if (bt.getLfdNr() > 0) {
				builder.append('-');
				builder.append(bt.getLfdNr());
			}
			builder.append(TenorToken.LEER);
		}
		return builder;
	}

	/**
	 * 
	 * @return gibt {@link #gegner} als {@link List<Beteiligter>} zurück. Lieber
	 *         List statt Set, denn Set hat keinen Getter...
	 * 
	 * @see eu.gronos.kostenrechnerjunits.ParteiBeziehung#getGegner()
	 */
	@Override
	// TODO oder Set<Beteiligter>? List<Beteiligter>?
	// TODO muss in JAXB ein @XmlIdRef sein - geht das auch mit Collections?
	@XmlElementWrapper(name = "gegnerlein")
	@XmlIDREF
	@XmlElement
	public List<Beteiligter> getGegner() {
		return gegner;
	}

	/**
	 * 
	 * @param gegner d. {@link #gegner}, d. gesetzt werden soll als
	 *               {@link List<Beteiligter>}.
	 * 
	 * @see eu.gronos.kostenrechnerjunits.ParteiBeziehung#setGegner(java.util.List)
	 */
	@Override
	public void setGegner(List<Beteiligter> gegner) {
		this.gegner.clear();
		gegner.addAll(gegner);
	}

	/**
	 * Die Methode formatiert die {@link Angriff#getGegner()}
	 * 
	 * @return eine {@link CharSequence} im Fomat &quot; gegen &lt;Gegner>&quot;
	 */
	private CharSequence formatGegner() {

		StringBuilder builder = new StringBuilder();

		if (getGegner().size() > 1)
			builder.append('[');
		for (Beteiligter bt : getGegner()) {
			builder.append(bt.kurzBezeichner(AKKUSATIV));
			if (bt.getLfdNr() > 0) {
				builder.append('-');
				builder.append(bt.getLfdNr());
			}
			builder.append(TenorToken.LEER);
		}
		if (getGegner().size() > 1) {
			builder.deleteCharAt(builder.lastIndexOf(TenorToken.LEER.toString()));
			builder.append("]@");
			builder.append(Integer.toHexString(getGegner().hashCode()));
			builder.append(TenorToken.LEER);
		}
		builder.append(formatGesamtschuldnerisch());

		return builder;
	}

	/**
	 * Die Methode formatiert ob {@link #isGesamtSchuldnerisch()}
	 * 
	 * @return &quot; gesamtschuldnerisch &quot;, falls
	 *         {@link #isGesamtSchuldnerisch()}, sonst &quot; &quot;
	 */
	private String formatGesamtschuldnerisch() {
		if (isGesamtSchuldnerisch()) {
			return TenorToken.LEER.toString() + TenorToken.GESAMTSCHULDNERISCH + TenorToken.LEER;
		}
		return TenorToken.LEER.toString();
	}

	@Override
	// Oder inanspruchnahme? wert?
	@XmlAttribute(name = "antrag")
	@XmlJavaTypeAdapter(EuroAdapter.class)
	public Euro getAntrag() {
		return this.antrag;
	}

	@Override
	public void setAntrag(Euro antrag) {
		this.antrag = antrag;
	}

	/**
	 * Die Methode formatiert den Streitwert
	 * 
	 * @return eine {@link CharSequence} im Format &quot;auf &lt;Streitwert>
	 *         EUR&quot;
	 */
	private CharSequence formatAntrag() {
		StringBuilder builder = new StringBuilder();
		if (getAntrag().getCents() > 0L) {
			builder.append(TenorToken.AUF.toString() + TenorToken.LEER);
			builder.append(getAntrag().toString());
			builder.append(TenorToken.LEER.toString() + TenorToken.EURO + TenorToken.LEER);
		}
		return builder;
	}

	@Override
	// Oder verurteilung? obsiegen?
	@XmlAttribute(name = "erfolg")
	@XmlJavaTypeAdapter(EuroAdapter.class)
	public Euro getErfolg() {
		return this.erfolg;
	}

	@Override
	public void setErfolg(Euro erfolg) {
		this.erfolg = erfolg;
	}

	/**
	 * Die Methode formatiert den Erfolg
	 * 
	 * @return eine {@link CharSequence} im Format
	 *         &quot;<code> (Erfolg: &lt;Erfolg> EUR)</code>
	 */
	private CharSequence formatErfolg() {
		StringBuilder builder = new StringBuilder("(Erfolg:" + TenorToken.LEER);
		builder.append(getErfolg().toString());
		builder.append(TenorToken.LEER.toString() + TenorToken.EURO + ").");
		return builder;
	}

	/**
	 * @return gibt {@link #gesamtSchuldnerisch} als {@link boolean} zurück.
	 */
	@XmlAttribute
	public boolean isGesamtSchuldnerisch() {
		return gesamtSchuldnerisch;
	}

	/**
	 * @param gesamtSchuldnerisch d. {@link #gesamtSchuldnerisch}, d. gesetzt werden
	 *                            soll als {@link boolean}.
	 */
	public void setGesamtSchuldnerisch(boolean gesamtSchuldnerisch) {
		this.gesamtSchuldnerisch = gesamtSchuldnerisch;
	}

	/**
	 * Die Methode vergleicht anhand von {@link #getAngreifer()}, dann nach
	 * {@link #getGegner()} (mehrere vor einzelnen)
	 * 
	 * @param o das andere Objekt
	 * @return a negative integer, zero, or a positive integer as this object is
	 *         less than, equal to, or greater than the specified object.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Angriff o) {
		// TODO null-Vergleich
		if (angreifer.get(0).compareTo(o.angreifer.get(0)) != 0)
			return angreifer.get(0).compareTo(o.angreifer.get(0));

		// TODO null-Vergleich
		if (gegner.size() > 1 || o.gegner.size() > 1) {
			return o.gegner.size() - gegner.size();
		} else if (gegner.get(0).compareTo(gegner.get(0)) != 0) {
			return gegner.get(0).compareTo(gegner.get(0));
		}

		return 0;
	}

	/**
	 * Manchmal braucht man halt sowohl {@link #getAngreifer()} als auch
	 * {@link #getGegner()}
	 * 
	 * @return eine {@link List} aus {@link Beteiligter} mit {@link #getAngreifer()}
	 *         und {@link #getGegner()}
	 */
	public List<Beteiligter> joinBeteiligte() {
		List<Beteiligter> beteiligte = new ArrayList<>(getAngreifer());
		beteiligte.addAll(getGegner());
		return beteiligte;
	}

	/**
	 * Die Methode benennt den Angriff gegen den Beteiligten mit dem
	 * <code>index</code>, indem es die {@link #parteiBezeichner(int, int)} und den
	 * Streitwert ausgibt.
	 * 
	 * @param angriff    der {@link Angriff}, der benannt werden soll
	 * @param beteiligte die Gesamt-{@link List}e aller {@link Beteiligter}n, auch
	 *                   die nicht in den Bezeichner aufzunehmenden, für eine
	 *                   Prüfung, ob sie eindeutig sind, für
	 *                   {@link MehrfachBeteiligter#istEinzigerSeinerArt(Beteiligter, List)}
	 * @return ein {@link String}
	 */
	public static String benenneAngriff(Angriff angriff, List<Beteiligter> beteiligte) {
		StringBuilder builder = new StringBuilder();
		builder.append(
				MehrfachBeteiligter.kurzBezeichnerListe(beteiligte, angriff.getAngreifer(), NOMINATIV));
		builder.append(TenorToken.VS.toString());
		builder.append(MehrfachBeteiligter.kurzBezeichnerListe(beteiligte, angriff.getGegner(), AKKUSATIV));
		builder.append(TenorToken.LEER.toString() + TenorToken.AUF + TenorToken.LEER);
		builder.append(angriff.getAntrag().toString());
		builder.append(TenorToken.LEER.toString() + TenorToken.EURO);
		return builder.toString();
	}

	/**
	 * Die Methode nimmt aus einer {@link BaumbachGesamtschuldnerschaft} alle
	 * {@link BaumbachBeteiligter}n
	 * 
	 * @param gegner die (wider-)beklagte {@link BaumbachGesamtschuldnerschaft}
	 * @return eine {@link Collection} aus {@link Beteiligter} mit den Einzelgegnern
	 */
	private static Collection<? extends Beteiligter> gesamtschuldnerBeteiligte(BaumbachGesamtschuldnerschaft gegner) {
		List<Beteiligter> liste = new ArrayList<>();
		for (int index : gegner.getAufzaehlung()) {
			liste.add(copyBeteiligter(gegner.getBaumbachBeteiligtenListe().get(index)));
		}
		return liste;
	}

	/**
	 * Die Methode erstellt einen neuen Beteiligten mit
	 * {@link Beteiligter#Beteiligter(Beteiligter)} und setzt zusätzlich noch die
	 * {@link Beteiligter#getLfdNr()}
	 * 
	 * @param beteiligter die Kopierquelle als {@link Beteiligter}
	 * @return neuen {@link Beteiligter}n
	 */
	private static Beteiligter copyBeteiligter(final Beteiligter beteiligter) {
		final Beteiligter copy = new Beteiligter(beteiligter);
		copy.setLfdNr(beteiligter.getLfdNr());
		return copy;
	}

	/**
	 * Die Methode baut einen einzelnen {@link Angriff} aus zwei
	 * {@link BaumbachBeteiligter}n: dem Angreifer (Kläger/Widerkläger) und dem
	 * Gegner (Beklagter/Widerbeklagter.
	 * 
	 * @param angreifer Angreifer (Kläger/Widerkläger) als
	 *                  {@link BaumbachBeteiligter}
	 * @param gegner    Gegner (Beklagter/Widerbeklagter) als
	 *                  {@link BaumbachBeteiligter}
	 * @return den {@link Angriff}
	 */
	static Angriff from(final BaumbachBeteiligter angreifer, final BaumbachBeteiligter gegner) {
		Angriff angriff = new Angriff();

		final Beteiligter angreiferBeteiligter = Angriff.copyBeteiligter(angreifer);
		angriff.getAngreifer().add(angreiferBeteiligter);

		if (gegner instanceof BaumbachGesamtschuldnerschaft) {
			final List<Beteiligter> liste = angriff.getGegner();
			liste.addAll(Angriff.gesamtschuldnerBeteiligte((BaumbachGesamtschuldnerschaft) gegner));
			angriff.setGesamtSchuldnerisch(true);
		} else {
			final Beteiligter gegnerBeteiligter = Angriff.copyBeteiligter(gegner);
			angriff.getGegner().add(gegnerBeteiligter);
			angriff.setGesamtSchuldnerisch(false);
		}

		angriff.setAntrag(gegner.getAntrag());//(Euro.ofEuros(gegner.getAntrag()));
		angriff.setErfolg(gegner.getErfolg());//(Euro.ofEuros(gegner.getErfolg()));

		return angriff;
	}

}
