package eu.gronos.kostenrechner.data.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.BEKLAGTE;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.KLAEGER;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import eu.gronos.kostenrechner.data.baumbach.Angriff.AngriffArt;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.AntragErfolgElement;
import eu.gronos.kostenrechner.interfaces.ParteiBeziehung;
import eu.gronos.kostenrechner.util.baumbach.BeteiligteInAngreiferGegnerReihenfolge;

/**
 * Eine {@link ArrayList} aus {@link Angriff}en, die sich auch die
 * {@link Beteiligter}n als {@link List} merkt und mit
 * {@link AngriffListe#getBeteiligte()} zurückgibt.
 * 
 * 
 * Kann außerdem BaumbachBeteiligte (BaumbachBeteiligtenListe) in
 * List&lt;Angriff> umwandeln
 *
 * @author Peter Schuster (setrok)
 * @date 12.10.2021
 *
 */
public class AngriffListe extends ArrayList<Angriff> {

	private static final long serialVersionUID = -2334090899427375988L;
	private final List<Beteiligter> beteiligte;

	public AngriffListe() {
		super();
		beteiligte = new ArrayList<>();
	}

	@Override
	public boolean add(Angriff angriff) {
		addBeteiligteFrom(angriff);
		return super.add(angriff);
	}

	@Override
	public boolean addAll(Collection<? extends Angriff> angriffe) {
		final int oldSize = size();
		addAllBeteiligte(angriffe);
		return size() != oldSize;
	}

	/**
	 * Die Methode fügt alle {@link Beteiligter}n einer {@link Collection} aus
	 * {@link Angriff}en der {@link #beteiligte}-{@link List} hinzu, also aus den
	 * jeweiligen {@link Angriff#getAngreifer()} und {@link Angriff#getGegner()}
	 * 
	 * @param angriffe eine {@link Collection} aus {@link Angriff}en
	 */
	private void addAllBeteiligte(Collection<? extends Angriff> angriffe) {
		for (Angriff angriff : angriffe) {
			addBeteiligteFrom(angriff);
			this.add(angriff);
		}
	}

	@Override
	public void add(int index, Angriff angriff) {
		addBeteiligteFrom(angriff);
		super.add(index, angriff);
	}

	/**
	 * @return gibt {@link #beteiligte} als {@link List<Beteiligter>} zurück.
	 */
	public List<Beteiligter> getBeteiligte() {
		return beteiligte;
	}

	/**
	 * Die Methode führt die eigentliche Konvertierung einer
	 * {@link BaumbachBeteiligtenListe} in eine {@link Collection} aus
	 * {@link Angriff} aus.
	 * 
	 * @param beteiligte {@link BaumbachBeteiligtenListe}
	 * @return {@link Collection} aus {@link Angriff}
	 */
	public static AngriffListe from(BaumbachBeteiligtenListe beteiligte) {
		AngriffListe angriffe = new AngriffListe();

		/*
		 * Anhand der Gegner der Angriffe durchlaufen, innerhalb der Schleife den
		 * zugehörigen Angreifer finden
		 */
		for (BaumbachBeteiligter gegner : beteiligte) {
			final BaumbachBeteiligter angreifer = findeAngreifer(beteiligte, gegner);
			/*
			 * Wenn es keinen Angreifer zum Beteiligten der Schleife gibt (null), dann ist
			 * der Beteiligte kein Gegner und muss übersprungen werden
			 */
			if (angreifer == null)
				continue;
			final Angriff angriff = Angriff.from(angreifer, gegner);
			angriffe.add(angriff);
		}

		angriffe.addAll(angriffeUmklammern(angriffe));
		return angriffe;
	}

	/**
	 * Die Methode fügt die {@link Beteiligter}n aus dem Angriff
	 * ({@link Angriff#getAngreifer()} und {@link Angriff#getGegner()}) der
	 * {@link List} {@link #getBeteiligte()} hinzu und sortiert diese dann neu.
	 * 
	 * @param angriff der hinzugefügte {@link Angriff}
	 */
	private void addBeteiligteFrom(Angriff angriff) {
		List<Beteiligter> liste = angriff.joinBeteiligte();
		for (Beteiligter beteiligter : liste) {
			if (!this.beteiligte.contains(beteiligter)) {
				this.beteiligte.add(beteiligter);
			}
		}
		Collections.sort(this.beteiligte, new BeteiligteInAngreiferGegnerReihenfolge());
	}

	/**
	 * Die Methode dient dazu, zu prüfen, ob der Erfolg insgesamt nicht über den
	 * Streitwert geht und ermittelt dafür den gesamten Erfolg der {@code gruppe},
	 * ob der {@code gegner} in irgendeiner der {@link Angriff#getGegner()}-Listen
	 * enthalten ist.
	 * 
	 * @param gruppe    eine Angriff-Gruppe als {@link List} von {@link Angriff}
	 * @param angreifer der Angreifer ({@link Angriff#getAngreifer()}) eines
	 *                  einzelnen {@link Angriff}s als {@link List} von
	 *                  {@link Beteiligter}
	 * @param gegner    der Gegner ({@link Angriff#getGegner()}) eines einzelnen
	 *                  {@link Angriff}s als {@link List} von {@link Beteiligter}
	 * @return die Summe aller {@link Angriff#getErfolg()} als {@link Euro}
	 */
	public static Euro findeGanzenErfolg(List<Angriff> gruppe, List<Beteiligter> angreifer, List<Beteiligter> gegner) {
		Euro erfolg = gruppe.stream()
				// wenn derselbe Angreifer und alle Gegner in Liste enthalten
				.filter(angriff -> angriff.getGegner().containsAll(gegner) && angriff.getAngreifer().equals(angreifer))
				// den Erfolg ziehen
				.map(Angriff::getErfolg)
				// Erfolg zusammenrechnen
				.reduce(Euro.ZERO_CENTS, (bisher, neu) -> bisher.add(neu));

		return erfolg;
	}

	/**
	 * Die Methode zählt den Wert aller Anträge der {@link List} von
	 * {@link Angriff}en zusammen
	 * 
	 * @param gruppe eine Antragsgruppe als {@link List} von {@link Angriff}en
	 * @return die Summe als long
	 */
	public static Euro addiereAntrag(List<Angriff> gruppe) {
		Euro gesamt = gruppe.stream()//
				.map(Angriff::getAntrag)//
				.reduce(Euro.ZERO_CENTS, (bisher, neu) -> bisher.add(neu));
		return gesamt;
	}

	/**
	 * Die Methode zählt den Erfolg aller Angriffe der {@link List} von
	 * {@link Angriff}en zusammen
	 * 
	 * @param gruppe eine Antragsgruppe als {@link List} von {@link Angriff}en
	 * @return die Summe als {@link Euro}
	 */
	public static Euro addiereErfolg(List<Angriff> gruppe) {
		Euro gesamt = gruppe.stream()//
				.map(Angriff::getErfolg)//
				.reduce(Euro.ZERO_CENTS, (bisher, neu) -> bisher.add(neu));
		return gesamt;
	}

	/**
	 * Die Methode prüft, ob der {@link Angriff} eine "Klage" oder "Widerklage" ist
	 * 
	 * Die Methode findet heraus, ob eine gruppierte {@link List} von
	 * {@link Angriff}en nur Widerklagen enthält
	 * 
	 * @param gruppe eine gruppierte {@link List} von {@link Angriff}en
	 * 
	 * @return {@link AngriffArt#KLAGE}, wenn in allen {@link Angriff}en die
	 *         {@link Angriff#getAngreifer()} aus {@link Beteiligter#KLAEGER}n,
	 *         {@link AngriffArt#WIDERKLAGE}, wenn sie nur aus
	 *         {@link Beteiligter#BEKLAGTE}n besteht, sonst
	 *         {@link AngriffArt#UNBESTIMMT}.
	 */
	public static AngriffArt klageOderWiderklage(List<Angriff> gruppe) {
		BeteiligtenTyp typ = einheitlicherAngreifer(gruppe);
		return AngriffArt.forBeteiligterTyp(typ);
	}

	/**
	 * Die Methode prüft, ob die gruppierte {@link List} von {@link Angriff}en einen
	 * einheitlichen Typ von Angreifern hat
	 *
	 * @param gruppe eine gruppierte {@link List} von {@link Angriff}en
	 * 
	 * @return den einheitlichen {@link Beteiligter#getTyp()} oder <code>-1</code>,
	 *         sofern nicht vorhanden
	 */
	private static BeteiligtenTyp einheitlicherAngreifer(List<Angriff> gruppe) {
		//int typ = -1;
		BeteiligtenTyp typ = null;
		for (ParteiBeziehung<List<Beteiligter>> angriff : gruppe) {
			for (Beteiligter angreifer : angriff.getAngreifer()) {
				if (typ == null) {//-1
					typ = angreifer.getTyp();
				} else if (typ != angreifer.getTyp()) {
					return null;//-1;
				} // sonst ist alles ok
			}
		}
		return typ;
	}

	/**
	 * Die Methode ermittelt alle {@link Beteiligter}n aus einer {@link List} von
	 * {@link Angriff}en, also den jeweilige Angreifer und Gegner
	 * 
	 * @param angriffe die zu durchsuchende {@link List} von {@link Angriff}en
	 * @return eine {@link List} von {@link Beteiligter}n
	 */
	public static List<Beteiligter> alleBeteiligteAus(List<Angriff> angriffe) {
		// TODO lieber als Stream und mit flatMap?
		if (angriffe instanceof AngriffListe) {
			return ((AngriffListe) angriffe).getBeteiligte();
		}
		final AngriffListe liste = new AngriffListe();
		liste.addAll(angriffe);
		return liste.getBeteiligte();
	}

	/**
	 * Die Methode findet einen einheitlichen Streitwert des
	 * {@link Angriff#getAntrag()}s in der {@link List} der {@link Angriff}e
	 * 
	 * @param gruppe Angriffsgruppe als {@link List} aus {@link Angriff}en
	 * @return einen gemeinsamen Streitwert oder {@code null}, sofern es keinen
	 *         gibt.
	 */
	public static Euro findeEinheitlichenAntrag(List<Angriff> gruppe) {
		long einheitlicherStreitwertCent = 0L;
		for (AntragErfolgElement angriff : gruppe) {
			// Wenn zuvor noch kein Streitwert gesetzt wurde (oder nur 0L), dann
			// einfach den
			// nächsten nehmen.
			if (einheitlicherStreitwertCent == 0L) {
				einheitlicherStreitwertCent = angriff.getAntrag().getCents();
				continue;
			} else if (angriff.getAntrag().getCents() == 0L) {
				// 0L-Werte sind egal
				continue;
			} else if (angriff.getAntrag().getCents() != einheitlicherStreitwertCent) {
				// Wenn nicht 0L aber anders als vorher, dann nicht einheitlich, also
				// -1 als
				// Fehlanzeige zurückgeben
				return null;
			}
		}
		return Euro.ofCents(einheitlicherStreitwertCent);
	}

	/**
	 * Die Methode fächert eine {@link List} von {@link Angriff}en in Gruppen auf,
	 * um später eine Begründungstabelle bauen zu können. Die {@link List}en sind
	 * gruppiert nach {@link Angriff#getAngreifer()}n gegen alle ihre
	 * {@link Angriff#getGegner()}.
	 * 
	 * @param angriffe eine {@link List} von {@link Angriff}en
	 * @return eine {@link List} aus Unter- {@link List}en von {@link Angriff}en
	 */
	public static List<List<Angriff>> gruppiereAngriffe(List<Angriff> angriffe) {
		List<List<Angriff>> liste = new ArrayList<>();
		List<Integer> ueberspringbare = new ArrayList<>();

		for (int i = 0; i < angriffe.size(); i++) {
			// schon behandelte Angriffe aussparen
			if (ueberspringbare.contains(Integer.valueOf(i)))
				continue;

			final List<Angriff> gruppe = new AngriffListe();
			final ParteiBeziehung<List<Beteiligter>> gruppenAngriff = angriffe.get(i);

			for (int j = i; j < angriffe.size(); j++) {
				Angriff angriff = angriffe.get(j);
				if (j == i) {
					gruppe.add(angriff);
				} else if (angriff.getAngreifer().equals(gruppenAngriff.getAngreifer())
						&& gruppenAngriff.getGegner().containsAll(angriff.getGegner())) {
					gruppe.add(angriff);
					ueberspringbare.add(Integer.valueOf(j));
				}
			}

			liste.add(gruppe);
		}

		return liste;
	}

	/**
	 * Die Methode baut genug alle Gegner umfassende Angriffe für Klage und
	 * Widerklage, damit {@link AngriffListe#gruppiereAngriffe(List)} für die
	 * Begründung funktioniert.
	 * 
	 * @param angriffe eine {@link List} aus {@link Angriff}en
	 * @return eine {@link List} aus den zusätzlich benötigten {@link Angriff}en
	 */
	private static Collection<? extends Angriff> angriffeUmklammern(List<Angriff> angriffe) {
		List<Angriff> liste = new ArrayList<>();
		List<Beteiligter> angreiferlein = new ArrayList<>();

		// alle Angreifer sammeln
		for (ParteiBeziehung<List<Beteiligter>> angreiferAngriff : angriffe) {
			for (Beteiligter angreifer : angreiferAngriff.getAngreifer()) {
				if (!angreiferlein.contains(angreifer))
					angreiferlein.add(angreifer);
			}
		}

		for (Beteiligter angreifer : angreiferlein) {
			// neuen Angriff mit dem Angreifer und allen seinen Gegnern bauen (Sw 0L,
			// erfolg 0L, nicht gesamtschuldnerisch)
			final Angriff klammer = new Angriff();
			klammer.getAngreifer().add(angreifer);
			klammer.setAntrag(Euro.ZERO_CENTS);
			klammer.setErfolg(Euro.ZERO_CENTS);
			klammer.setGesamtSchuldnerisch(false);
			// alle Gegner dieses Angreifers sammeln und dem Angriff hinzufügen
			for (ParteiBeziehung<List<Beteiligter>> angreiferAngriff : angriffe) {
				if (angreiferAngriff.getAngreifer().contains(angreifer)) {
					for (Beteiligter gegner : angreiferAngriff.getGegner()) {
						if (!klammer.getGegner().contains(gegner)) {
							klammer.getGegner().add(gegner);
						}
					}
				}
			}
			// gucken, ob es einen Angriff dieses Angreifers schon gibt, der alle
			// Gegner enthält
			boolean vorhanden = false;
			vorhandenSchleife: for (ParteiBeziehung<List<Beteiligter>> angreiferAngriff : angriffe) {
				if (angreiferAngriff.getAngreifer().contains(angreifer)
						&& angreiferAngriff.getGegner().containsAll(klammer.getGegner())) {
					vorhanden = true;
					break vorhandenSchleife;
				}
			}
			// falls nicht vorhanden: zur Liste nehmen
			if (!vorhanden)
				liste.add(klammer);
		}

		return liste;
	}

	/**
	 * Die Methode sucht den widerklagenden Beklagten aus der
	 * {@link BaumbachBeteiligtenListe}
	 * 
	 * @param beteiligte die zu durchsuchende {@link BaumbachBeteiligtenListe}
	 * @return den Widerkläger als {@link BaumbachBeteiligter}
	 */
	private static BaumbachBeteiligter findeWiderklaeger(BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
		BaumbachBeteiligter angreifer;
		final int[] bereich = baumbachBeteiligtenListe.minUndMax(BEKLAGTE);
		for (int i = bereich[0]; i <= bereich[1]; i++) {
			if (baumbachBeteiligtenListe.get(i).isAnWiderklageBeteiligt()) {
				angreifer = baumbachBeteiligtenListe.get(i);
				return angreifer;
			}
		}
		return null;
	}

	/**
	 * Die Methode sucht den Kläger aus der {@link BaumbachBeteiligtenListe}
	 * 
	 * @param beteiligte die zu durchsuchende {@link BaumbachBeteiligtenListe}
	 * @return der Kläger als {@link BaumbachBeteiligter}
	 */
	private static BaumbachBeteiligter findeKlaeger(BaumbachBeteiligtenListe beteiligte) {
		if (beteiligte.getTypFor(0) == KLAEGER) {
			return beteiligte.get(0);
		}
		return null;
	}

	/**
	 * Die Methode sucht den {@link Angriff#getAngreifer()} zu einem gegebenen
	 * {@link Angriff#getGegner()} aus einer {@link BaumbachBeteiligtenListe} über
	 * {@link AngriffListe#findeKlaeger(BaumbachBeteiligtenListe)} und
	 * {@link AngriffListe#findeWiderklaeger(BaumbachBeteiligtenListe)}
	 * 
	 * @param beteiligte die zu durchsuchende {@link BaumbachBeteiligtenListe}
	 * @param gegner     ein {@link BaumbachBeteiligter}, der entweder Beklagte*r
	 *                   oder widerbeklagter Kläger/Drittwiderbeklagter ist
	 * @return den Angreifer als als {@link BaumbachBeteiligter}
	 */
	private static BaumbachBeteiligter findeAngreifer(BaumbachBeteiligtenListe beteiligte, BaumbachBeteiligter gegner) {
		if (gegner.getTyp() == BEKLAGTE) {
			return AngriffListe.findeKlaeger(beteiligte);
		} else if (gegner.isAnWiderklageBeteiligt()) {
			return AngriffListe.findeWiderklaeger(beteiligte);
		}
		return null;
	}

}
