/**
 * BaumbachBeteiligter.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.baumbach;

import java.io.Serializable;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.EuroAdapter;
import eu.gronos.kostenrechner.interfaces.AntragErfolgElement;
import eu.gronos.kostenrechner.interfaces.Hinzufuegbar;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.*;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.*;

/**
 * Eine Klasse, abgeleitet von Beteiligter, die zusätzlich den Streitwert, das
 * Unterliegen und die Beteiligung an einer Widerklage speichert.
 * 
 * @author Peter Schuster (setrok)
 * @date: 27.04.2014
 * 
 * @todo TODO equals(), hash()
 * 
 */
public class BaumbachBeteiligter extends Beteiligter implements Serializable, Hinzufuegbar, AntragErfolgElement {
	public static final String UND_WK_WBK = "und %s";
	public static final String PARTEI_UND_WK_WBK = "%s und %s";
	public final static String[] WIDERBEKLAGTEN_BEUGUNG = new String[] { "Widerbeklagter", "Widerbeklagte" };
	public final static String[] WIDERKLAEGER_BEUGUNG = new String[] { "Widerkläger", "Widerklägerin" };

	private static final long serialVersionUID = -3484780882204159173L;
	/**
	 * Der {@link Euro} mit dem Namen streitwert speichert den Streitwert der gegen
	 * (!) den Beteiligten geltend gemachten Forderung(en), nicht der von ihm
	 * geltend gemachten.
	 */
	private Euro antrag;
	/**
	 * Der {@link Euro} mit dem Namen unterliegen speichert das Unterliegen des
	 * jeweiligen Beteiligten hinsichtlich der gegen ihn (!) geltend gemachten
	 * Forderung, bei einem Beklagten also hinsichtlich der Hauptklage, bei einem
	 * Kläger/Drittwiderbeklagten hinsichtlich der Widerklage. Nicht gemeint ist das
	 * Unterliegen hinsichtlich der selbst von dem Beteiligten gegen andere geltend
	 * gemachten Forderungen.
	 */
	private Euro erfolg;
	// final
	/**
	 * Der boolean mit dem Namen anWiderklageBeteiligt speichert, ob der
	 * BaumbachBeteiligte an der Widerklage beteiligt ist (der Beklagte also
	 * zugleich Widerkläger, ein Kläger zugleich Widerbeklagter ist).
	 */
	private boolean anWiderklageBeteiligt;

	/**
	 * Konstruktor:
	 * 
	 * @param typ                   eine der Konstanten Beteiligter.KLAEGER,
	 *                              Beteiligter.DRITTWIDERBEKLAGTE,
	 *                              Beteiligter.BEKLAGTE
	 * @param genusNumerus          eine der Konstanten MAENNLICH oder WEIBLICH
	 * @param streitwert2           Streitwert bzgl. des Angriffs gegen die
	 *                              jeweilige Partei
	 * @param unterliegen           das Unterliegen des Beteiligten bzgl. des
	 *                              Angriffs gegen die jeweilige Partei (aus Sicht
	 *                              der Partei)
	 * @param anWiderklageBeteiligt ist die jeweilige Partei an der Widerklage
	 *                              beteiligt?
	 */
//	@Deprecated
//	public BaumbachBeteiligter(int typ, int genusNumerus, Euro streitwert, double unterliegen,
//			boolean anWiderklageBeteiligt) {
//		super(typ, genusNumerus);
//		this.streitwert = streitwert.doubleValue();
//		this.unterliegen = unterliegen;
//		this.anWiderklageBeteiligt = anWiderklageBeteiligt;
//	}

	/**
	 * Konstruktor:
	 * 
	 * @param typ                   {@link Enum} {@link Beteiligter.BeteiligtenTyp}
	 * @param genusNumerus          eine der Konstanten MAENNLICH oder WEIBLICH
	 * @param streitwert2           Streitwert bzgl. des Angriffs gegen die
	 *                              jeweilige Partei
	 * @param unterliegen           das Unterliegen des Beteiligten bzgl. des
	 *                              Angriffs gegen die jeweilige Partei (aus Sicht
	 *                              der Partei)
	 * @param anWiderklageBeteiligt ist die jeweilige Partei an der Widerklage
	 *                              beteiligt?
	 */
	public BaumbachBeteiligter(BeteiligtenTyp typ, GenusNumerus genusNumerus, Euro streitwert, Euro unterliegen,
			boolean anWiderklageBeteiligt) {
		super(typ, genusNumerus);
		this.antrag = streitwert;// .doubleValue();
		this.erfolg = unterliegen;
		this.anWiderklageBeteiligt = anWiderklageBeteiligt;
	}

	/**
	 * Konstruktor, der statt typ und genusNumerus eine Instanz von Beteiligter
	 * nimmt und ausliest. Dieser Konstruktor richtet eine Instanz mit denselben
	 * Werten für typ und genusNumerus wie der übergegebene Beteiligte ein (soll die
	 * Arbeit bei abgeleiteten Klassen und mit CellEditor erleichtern).
	 * 
	 * @param beteiligter           ein Beteiligter, von dem typ und genusNumerus
	 *                              übernommen werden.
	 * @param streitwert            Streitwert bzgl. des Angriffs gegen die
	 *                              jeweilige Partei
	 * @param unterliegen           das Unterliegen des Beteiligten bzgl. des
	 *                              Angriffs gegen die jeweilige Partei (aus Sicht
	 *                              der Partei)
	 * @param anWiderklageBeteiligt ist die jeweilige Partei an der Widerklage
	 *                              beteiligt?
	 */
	public BaumbachBeteiligter(Beteiligter beteiligter, Euro streitwert, Euro unterliegen,
			boolean anWiderklageBeteiligt) {
		super(beteiligter);
		this.antrag = streitwert;
		this.erfolg = unterliegen;
		this.anWiderklageBeteiligt = anWiderklageBeteiligt;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public BaumbachBeteiligter() {
		super();
	}

	/**
	 * @return streitwert gibt den streitwert als double zurück, also den Streitwert
	 *         bzgl. des Angriffs gegen die jeweilige Partei.
	 */
	@XmlAttribute(name = "streitwert")
	@XmlJavaTypeAdapter(EuroAdapter.class)
	public Euro getAntrag() {
		return antrag;
	}

	/**
	 * @param streitwert the streitwert to set
	 */
	public void setAntrag(Euro antrag) {
		/*
		 * Bei nicht-widerbeklagten Klägern keine Änderung zulassen TODO Muss woanders
		 * geprüft werden, sonst Fehler! if (getTyp() == KLAEGER &&
		 * !isAnWiderklageBeteiligt()) return; if (streitwert < 0) return;
		 */
		this.antrag = antrag;
	}

	/**
	 * @return unterliegen gibt den unterliegen GEGEN diesen Beteiligten als double
	 *         zurück, also das Unterliegen des Beteiligten bzgl. des Angriffs gegen
	 *         die jeweilige Partei (aus Sicht der Partei)
	 */
	@XmlAttribute(name = "unterliegen")
	@XmlJavaTypeAdapter(EuroAdapter.class)
	public Euro getErfolg() {
		return erfolg;
	}

	/**
	 * @param unterliegen d. unterliegen, d. gesetzt werden soll
	 */
	public void setErfolg(Euro unterliegen) {
		this.erfolg = unterliegen;
	}

	/**
	 * @return anWiderklageBeteiligt gibt den anWiderklageBeteiligt als boolean
	 *         zurück, also ob die jeweilige Partei an der Widerklage beteiligt ist.
	 */
	@XmlAttribute(name = "widerklage")
	public boolean isAnWiderklageBeteiligt() {
		return anWiderklageBeteiligt;
	}

	/**
	 * @param anWiderklageBeteiligt the anWiderklageBeteiligt to set
	 */
	public void setAnWiderklageBeteiligt(boolean anWiderklageBeteiligt) {
		// beteiligtenTypAsInt
		if (getTyp() == DRITTWIDERBEKLAGTE)
			this.anWiderklageBeteiligt = true;
		else
			this.anWiderklageBeteiligt = anWiderklageBeteiligt;
	}

	/**
	 * @return gibt true zurück, wenn es sich um einen BaumbachGesamtschuldner
	 *         handelt (hier also immer false)
	 */
	public boolean isGesamtschuldnerschaft() {
		return false;
	}

	@Override
	public String getId() {
		if (this.getTyp() == null)// ||
			return null;
		// hier immer die interne Nummer, für die Eindeutigkeit
		return "" + getTyp().kurz() + getTempNr();
	}

	/**
	 * Die Methode erzeugt eine String-Repräsentation des Beteiligten durch Aufruf
	 * der Methode {@link parteiBezeichner(int)} im Nominativ mit zusätzlicher
	 * laufender Nummer.
	 * 
	 * @return einen Parteibezeichner
	 * 
	 * @see eu.gronos.kostenrechner.data.tenordaten.Beteiligter#toString()
	 */
	public String langBezeichnung() {
		return parteiBezeichner(NOMINATIV, this.getLfdNr(), false);
	}

	/**
	 * Die Methode erzeugt eine {@link String}-Repräsentation des Beteiligten, die
	 * knapp alle Werte der Instanz aufführt. Nach Umbau in <code>0.6.1</code>
	 * eignet sich das Ergebnis nicht mehr zur Verarbeitung in einem Tenortext,
	 * dafür besser zur Ausgabe in Logbüchern oder für Testverfahren. Zur
	 * Textausgabe bitte {@link #langBezeichnung()} nehmen.
	 * 
	 * @return einen String, der die Werte aus {@link Beteiligter#toString()} sowie
	 *         {@link #getAntrag()}, {@link #getErfolg()} und
	 *         {@link #isAnWiderklageBeteiligt()} aufnimmt.
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder(
				super.toString().replace("Beteiligter ", "BaumbachBeteiligter ").replace("]", ", "));
		builder.append("streitwert=");
		builder.append(antrag);
		builder.append(", unterliegen=");
		builder.append(erfolg);
		builder.append(", anWiderklageBeteiligt=");
		builder.append(anWiderklageBeteiligt);
		builder.append("]");
		return builder.toString();
	}

}
