/*
 * Streitgenossen.java
 * eu.gronos.kostenrechner.model.tenordaten (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.baumbach;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlElements;

import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;

/**
 * {@link UnterContainerKlasse} von {@link VerfahrensDatenContainer} für die
 * Registerkarte Streitgenossen
 *
 * @author Peter Schuster (setrok)
 * @date 27.12.2014
 *
 */
public class Streitgenossen implements UnterContainerKlasse {

	public static final int TAB_ID = 1;

	/**
	 * widerklage speichert als {@link boolean}, ob eine Widerklage vorliegt.
	 */
	@XmlAttribute(name = "widerklage")
	public boolean widerklage;

	/**
	 * wirtschaftlichIdentisch speichert als {@link boolean}, ob Klage und
	 * Widerklage wirtschaftlich identisch im Sinne von § 45 Abs. 1 S. 3 GKG sind.
	 */
	@XmlAttribute(name = "p45i3GKG")
	public boolean wirtschaftlichIdentisch;

	/**
	 * klaeger speichert den Kläger als {@link BaumbachBeteiligtenListe}.
	 */
	@XmlElement(name = "klaeger")
	public BaumbachBeteiligtenListe klaeger = new BaumbachBeteiligtenListe();

	/**
	 * drittwiderbeklagte speichert die Drittwiderbeklagten und die
	 * Widerklagten-Gesamtschuldnerschaften als {@link BaumbachBeteiligtenListe}.
	 */
	@XmlElements({ @XmlElement(name = "drittwiderbeklagter", type = BaumbachBeteiligter.class),
			@XmlElement(name = "widerbeklagtenGesamtschuldnerschaft", type = BaumbachGesamtschuldnerschaft.class) })
	@XmlElementWrapper(name = "drittwiderbeklagtenListe")
	public BaumbachBeteiligtenListe drittwiderbeklagte = new BaumbachBeteiligtenListe();

	/**
	 * beklagte speichert die Beklagten und die Beklagten-Gesamtschuldnerscahften
	 * als {@link BaumbachBeteiligtenListe}.
	 */
	@XmlElements({ @XmlElement(name = "beklagter", type = BaumbachBeteiligter.class),
			@XmlElement(name = "beklagtenGesamtschuldnerschaft", type = BaumbachGesamtschuldnerschaft.class) })
	@XmlElementWrapper(name = "beklagtenListe")
	public BaumbachBeteiligtenListe beklagte = new BaumbachBeteiligtenListe();

	/**
	 * @return <code>true</code>, wenn es eine nicht-leere Klägerliste gibt.
	 */
	public boolean gibtsKlaeger() {
		return klaeger != null && !klaeger.isEmpty();
	}

	/**
	 * @return <code>true</code>, wenn es eine nicht-leere Beklagten-Liste gibt
	 */
	public boolean gibtsBeklagte() {
		return beklagte != null && !beklagte.isEmpty();
	}

	/**
	 * @return <code>true</code>, wenn es eine nicht-leere Drittwiderbeklagten-Liste
	 *         gibt
	 */
	public boolean gibtsDrittwiderbeklagte() {
		return drittwiderbeklagte != null && !drittwiderbeklagte.isEmpty();
	}

	/**
	 * 
	 * @return <code>true</code>, wenn {@link #klaeger}, {@link #beklagte} und
	 *         {@link #drittwiderbeklagte} leer sind sowie {@link #widerklage} nicht
	 *         gesetzt ist. <code>false</code>, wenn auch nur eine der Tabellen
	 *         Werte enthält oder {@link #widerklage} gesetzt ist.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.UnterContainerKlasse#isEmpty()
	 */
	@Override
	public boolean isEmpty() {
		return !widerklage//
				&& (klaeger == null || klaeger.isEmpty())//
				&& (beklagte == null || beklagte.isEmpty())//
				&& (drittwiderbeklagte == null || drittwiderbeklagte.isEmpty());
	}
}