/**
 * AufrechnungsForderung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.forderungen;

import java.util.Optional;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.OptionalEuroAdapter;

/**
 * Eine Klasse für Hilfsaufrechnungen
 * 
 * @author Peter Felix Schuster (setrok)
 * @date: 31.07.2014
 */
public class AufrechnungsForderung extends Forderung {

	private static final long serialVersionUID = 4494763842223765977L;
	private static final String BESCHREIBUNG = "Hilfsaufrechnung";
	private Optional<Euro> effektiverWert = Optional.empty();// double effektiverWert = -1;
	private Optional<Euro> effektiverErfolg = Optional.empty();// double effektiverErfolg = -1;

	/**
	 * Erzeugt eine neue Instanz einer AufrechnungsForderung (d.h.
	 * Hilfsaufrechnung). Die Felder effektiverWert und effektiverErfolg werden
	 * zunächst mit -1 initialisiert.
	 * 
	 * @param gegenforderung ob es sich um eine Forderung des Klägers gegen den
	 *                       Beklagten (false) oder eine Forderung des Beklagten
	 *                       gegen den Kläger handelt
	 * @param wert           die Höhe der Forderung, also ihren Wert.
	 * @param erfolg         inwieweit die Forderung Erfolg hatte.
	 * @throws IllegalArgumentException wenn der Erfolg höher als der Wert ist
	 */
	public AufrechnungsForderung(boolean gegenforderung, Euro wert, Euro erfolg) throws IllegalArgumentException {
		super(gegenforderung, wert, erfolg);
	}

	/**
	 * Der Standardkonstruktor erzeugt eine neue Instanz einer AufrechnungsForderung
	 * (d.h. Hilfsaufrechnung). Die Felder effektiverWert und effektiverErfolg
	 * werden zunächst mit -1 initialisiert. Gegenforderung wird als Standard auf
	 * true gesetzt.
	 * 
	 * @param wert   die Höhe der Forderung, also ihren Wert.
	 * @param erfolg inwieweit die Forderung Erfolg hatte.
	 * @throws IllegalArgumentException wenn der Erfolg höher als der Wert ist
	 */
	public AufrechnungsForderung(Euro wert, Euro erfolg) throws IllegalArgumentException {
		this(true, wert, erfolg);
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public AufrechnungsForderung() {
		super();
		super.setGegenforderung(true);
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append(", effektiverWert=");
		builder.append(effektiverWert);
		builder.append(", effektiverErfolg=");
		builder.append(effektiverErfolg);

		StringBuilder superBuilder = new StringBuilder(
				super.toString().replace("Forderung ", "AufrechnungsForderung "));
		superBuilder.insert(superBuilder.lastIndexOf("]"), builder);
		return superBuilder.toString();
	}

	/**
	 * Gibt eine Beschreibung der Forderungsart zurück
	 * 
	 * @return "Hilfsaufrechnung"
	 * 
	 * @see eu.gronos.kostenrechner.data.forderungen.Forderung#getBeschreibung()
	 */
	@Override
	public String getBeschreibung() {
		return BESCHREIBUNG;
	}

	/**
	 * @return den Wert der Aufrechnungsforderung, soweit er sich mit der
	 *         verbliebenen Klageforderung deckt ("Schnittmenge" = effektiver Wert).
	 */
	@XmlAttribute(name = "effektivWert")
	@XmlJavaTypeAdapter(OptionalEuroAdapter.class)
	public Optional<Euro> getEffektiverWert() {
		return effektiverWert;
	}
	// @XmlTransient

	/**
	 * @param euro the effektiverWert to set
	 */
	public void setEffektiverWert(Euro euro) {
		if (euro == null || euro.lessThan(Euro.ZERO_CENTS))
			this.effektiverWert = Optional.empty();
		else
			this.effektiverWert = Optional.of(euro);
	}

	/**
	 * Ein Setter, der direkt den {@link Optional} nimmt.
	 * 
	 * @param effektiverWert the effektiverWert to set
	 */
	public void setEffektiverWert(Optional<Euro> effektiverWert) {
		this.effektiverWert = effektiverWert;
	}

	/**
	 * Diese Methode dient dazu, den effektiven Wert zurückzusetzen. Wenn seit der
	 * letzten Berechnung Werte verändert wurden, kommt es sonst zu Fehlern.
	 * 
	 */
	public void setzeEffektivenWertZurueck() {
		setEffektiverWert(Optional.empty());// (-1);
	}

	/**
	 * @return den Erfolg der Aufrechnungsforderung, soweit er sich mit der
	 *         verbliebenen Klageforderung deckt ("Schnittmenge" = effektiver
	 *         Erfolg).
	 */
	@XmlAttribute(name = "effErfolg")
	@XmlJavaTypeAdapter(OptionalEuroAdapter.class)
	public Optional<Euro> getEffektiverErfolg() {
		return effektiverErfolg;
	}

	/**
	 * @param euro the effektiverErfolg to set
	 */
	public void setEffektiverErfolg(Euro euro) {
		if (euro == null || euro.lessThan(Euro.ZERO_CENTS))
			this.effektiverErfolg = Optional.empty();
		else
			this.effektiverErfolg = Optional.of(euro);
	}

	/**
	 * Ein Setter, der direkt den {@link Optional} nimmt.
	 * 
	 * @param effektiverErfolg the effektiverErfolg to set
	 */
	public void setEffektiverErfolg(Optional<Euro> effektiverErfolg) {
		this.effektiverErfolg = effektiverErfolg;
	}

	/**
	 * Die Methode dient dazu, den effektiven Erfolg zurück zu setzen. Wenn seit der
	 * letzten Berechnung Werte verändert wurden, kommt es sonst zu Fehlern.
	 * 
	 */
	public void setzeEffektivenErfolgZurueck() {
		setEffektiverErfolg(Optional.empty());// (-1);
	}
}
