/**
 * Forderung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.forderungen;

import java.io.Serializable;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.EuroAdapter;
import eu.gronos.kostenrechner.interfaces.AntragErfolgElement;
import eu.gronos.kostenrechner.interfaces.Hinzufuegbar;
import eu.gronos.kostenrechner.view.forderungen.ForderungenRenderer;

/**
 * Grundklasse für Forderungen.
 * 
 * @author Peter Felix Schuster (setrok)
 * @date: 30.07.2014
 *
 */
@XmlType(propOrder = { "antrag", "erfolg" })//"wert", "erfolg"
public abstract class Forderung implements Serializable, AntragErfolgElement, Hinzufuegbar {
	private static final long serialVersionUID = 5194917413163124727L;
	private boolean gegenforderung;
	private Euro antrag;//double wert
	private Euro erfolg;//double

	/**
	 * Konstruktor:
	 * 
	 * @param gegenforderung ob es sich um eine Forderung des Klägers gegen den
	 *                       Beklagten (false) oder eine Forderung des Beklagten
	 *                       gegen den Kläger handelt
	 * @param antrag         die Höhe der Forderung, des Antrags, also ihren Wert.
	 * @param erfolg         inwieweit die Forderung Erfolg hatte.
	 */
	Forderung(boolean gegenforderung, Euro antrag, Euro erfolg) throws IllegalArgumentException {
		this.gegenforderung = gegenforderung;
		this.antrag = antrag;
		setErfolg(erfolg);
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 */
	public Forderung() {
		super();
	}

	/**
	 * @return <code>boolean</code>, ob es sich um eine Forderung des Klägers gegen
	 *         den Beklagten (<code>false</code>) oder eine Forderung des Beklagten
	 *         gegen den Kläger handelt (<code>true</code>).
	 */
	@XmlAttribute(name = "gegenforderung")
	public boolean isGegenforderung() {
		return gegenforderung;
	}

	/**
	 * @param gegenforderung the gegenforderung to set
	 */
	public void setGegenforderung(boolean gegenforderung) {
		this.gegenforderung = gegenforderung;
	}

	/**
	 * @return die Höhe der Forderung, also ihren Wert.
	 */
	@XmlAttribute(name = "wert")
    @XmlJavaTypeAdapter(EuroAdapter.class)
	public Euro getAntrag() {
		return antrag;
	}

	/**
	 * @param antrag the wert to set
	 */
	public void setAntrag(Euro antrag) {
		this.antrag = antrag;
	}

	/**
	 * @return inwieweit die Forderung Erfolg hatte.
	 */
	@XmlAttribute(name = "erfolg")
    @XmlJavaTypeAdapter(EuroAdapter.class)
	public Euro getErfolg() {
		return erfolg;
	}

	/**
	 * @param erfolg the erfolg to set
	 */
	public void setErfolg(Euro erfolg) {
		this.erfolg = erfolg;
	}

	/**
	 * Eine zu überschreibende Funktion für String-Ausgaben
	 * 
	 * @return eine interne Beschreibung der Forderung für ableitende Klassen.
	 */
	public abstract String getBeschreibung();

	
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("Forderung [");
		if (getBeschreibung() != null) {
			builder.append("getBeschreibung()=");
			builder.append(getBeschreibung());
			builder.append(", ");
		}
		builder.append("gegenforderung=");
		builder.append(gegenforderung);
		builder.append(", wert=");
		builder.append(antrag);
		builder.append(", erfolg=");
		builder.append(erfolg);
		
		builder.append("]");
		return builder.toString();
	}

	/**
	 * Die Methode ist eine spezielle {@link #toString()}-Methode für den
	 * {@link ForderungenRenderer}.
	 * 
	 * @return eine nutzerlesbare {@link String}-Repräsentation der
	 *         {@link Forderung}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Hinzufuegbar#langBezeichnung()
	 */
	public String langBezeichnung() {
		String isGegenforderung = gegenforderung ? "Gegenforderung" : "Forderung";
		return String.format("%s (%s) in Höhe von %s, davon %s begründet", getBeschreibung(), isGegenforderung, antrag,
				erfolg);
	}

}
