/**
 * AuslagenTatbestand.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import javax.xml.bind.JAXB;

import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenParser;
import eu.gronos.kostenrechner.data.tenordaten.Euro;

/**
 * Von GebuehrenTatbestand abgeleitete Klasse, für Auslagen und
 * streitwertunabhängige Gebühren.
 * 
 * @author Peter Schuster (setrok)
 * @date 06.05.2014
 * 
 */
public class AuslagenTatbestand extends PauschalTatbestand implements Cloneable {

	private static final long serialVersionUID = -7242122091272139243L;
	public static final String ART = "Auslagen";

	/**
	 * Dieser Konstruktor (ohne die Höhe der Auslagen) kann benutzt werden, wenn die
	 * Höhe variabel, also nicht immer dieselbe ist (etwa bei Kosten der
	 * Beweisaufnahme). Eine Abfrage der Auslagen liefert so lange -1.0, bis ein
	 * anderer Wert mit {@link #setBetrag(double)} festgesetzt wird.
	 * 
	 * @todo TODO -1.0 ist gefährlich, wenn mit {@link GebuehrenParser} geprüft
	 *       wird!
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand#Constructor(String,double,Class)
	 */
	public AuslagenTatbestand(String vorschrift, String bezeichnung,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		this(vorschrift, bezeichnung, null, gebuehrenKlasse);//-1.0
	}

	/**
	 * Dieser Konstruktor (mit Höhe der Auslagen) kann benutzt werden, wenn die Höhe
	 * der Auslage bei Initialisieren bereits feststeht, z.B. weil es eigentlich
	 * eine Gebühr mit fester (streitwertunabhängiger) Höhe ist, z.B. die
	 * Aktenversandpauschale.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param auslagen        die Höhe der Auslagen oder der streitwertunabhängigen
	 *                        Gebühr/Pauschale, oder ein negativer Wert (-1.0), wenn
	 *                        diese noch nicht feststeht und später gesetzt werden
	 *                        soll.
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 */
	public AuslagenTatbestand(String vorschrift, String bezeichnung, Euro auslagen,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super(vorschrift, bezeichnung, auslagen, gebuehrenKlasse);
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public AuslagenTatbestand() {
		super();
	}

	/**
	 * Die toString() Methode der Klasse AuslagenTatbestand liefert einen String mit
	 * der Bezeichnung des Gebührentatbestands und der Höhe der Auslagen in
	 * Klammern.
	 * 
	 * @return einen String nach dem Format "Text (Auslagen)", z.B.
	 *         "Verfahrensgebühr, Ziffer xxxx VVRVG (2.000,00 EUR)"
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return String.format("AuslagenTatbestand [vorschrift=%s, bezeichnung=%s, gebuehrenKlasse=%s, betrag=%s]",
				getVorschrift(), getBezeichnung(), getGebuehrenKlasse(), getBetrag());
	}

	/**
	 * Die Methode dient dazu, einen {@link Object#hashCode()} zu erstellen.
	 * 
	 * @return einen <code>int</code>, der sich aus {@link #getVorschrift()},
	 *         {@link #getBezeichnung()}, {@link #getGebuehrenKlasse()} und
	 *         {@link #getBetrag()} berechnet.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + getBetrag().hashCode();//(int) (temp ^ (temp >>> 32));
		return result;
	}

	/**
	 * Vergleicht nach den Kriterien {@link #getVorschrift()},
	 * {@link #getBezeichnung()}, {@link #getGebuehrenKlasse()} und
	 * {@link #getBetrag()}
	 * 
	 * @param obj das zu vergleichende {@link Object}
	 * @return <code>true</code>, wenn die Kriterien übereinstimmen, sonst
	 *         <code>false</code>
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (!(obj instanceof AuslagenTatbestand)) {
			return false;
		}
		AuslagenTatbestand other = (AuslagenTatbestand) obj;
 //		if (Double.doubleToLongBits(getBetrag()) != Double.doubleToLongBits(other.getBetrag())) {
		if (!getBetrag().equals(other.getBetrag())) {
			return false;
		}
		return true;
	}

	@Override
	public AuslagenTatbestand clone() {
		return new AuslagenTatbestand(getVorschrift(), getBezeichnung(), getBetrag(), getGebuehrenKlasse());
	}

}
