/**
 * GebuehrenErhoehungsTatbestand.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear;

/**
 * Eine Klasse, um Mehrvertretungsgebühren abzubilden, also Gebühren, die die
 * Verfahrensgebühr um einen bestimmten Satz erhöhen. Und zwar umso mehr, je
 * mehr Leute (anzahl) vertreten werden, aber maximal bis zu einer bestimmten
 * Höhe.
 *
 * @author Peter Schuster (setrok)
 * @date 25.08.2014
 *
 */
public class GebuehrenErhoehungsTatbestand extends GebuehrenSatzTatbestand
		implements Cloneable, GebuehrenMitAnzahlLinear<Double> {

	private static final long serialVersionUID = -2255864991661667203L;
	private double jeweils;
	private double maxSatz;
	public static final String ART = "Mehrvertretungsgebühr";
	private int anzahl;
	private String einheit;
	/**
	 * berechnet: der {@link boolean} wird auf <code>true</code> gesetzt, wenn
	 * {@link #getBetrag()} einmal {@link #berechne()}t wurde, wieder auf
	 * <code>false</code> gesetzt, wenn sich {@link #getAnzahl()},
	 * {@link #getJeweils()} oder {@link #getMaxSatz()} ändern.
	 */
	private boolean berechnet;

	/**
	 * Der Standardkonstruktor erlaubt, alle Felder bei der Initialisierung schon zu
	 * setzen.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param jeweils         der Satz, der für jeden der in {@link #anzahl}
	 *                        genannten Personen jeweils gilt
	 * @param maxSatz         der Satz, der maximal anzusetzen ist
	 * @param anzahl          die Anzahl der Personen (oder sonstigen Einheiten).
	 * @param einheit         die Bezeichnung der Einheit, also z.B.
	 *                        &quot;mehrvertretene Personen&quot;
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 */
	public GebuehrenErhoehungsTatbestand(String vorschrift, String bezeichnung, double jeweils, double maxSatz,
			int anzahl, String einheit, Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super(vorschrift, bezeichnung, -1.0, gebuehrenKlasse);
		this.jeweils = jeweils;
		this.maxSatz = maxSatz;
		this.anzahl = anzahl;
		this.einheit = einheit;
		berechnet = false;
	}

	/**
	 * Ein vereinfachter Konstruktor, der die {@link #anzahl} bei Initialisierung
	 * auf -1 setzt, damit sie nachträglich noch gesetzt werden kann.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param jeweils         der Satz, der für jeden der in {@link #anzahl}
	 *                        genannten Personen jeweils gilt
	 * @param maxSatz         der Satz, der maximal anzusetzen ist
	 * @param einheit         die Bezeichnung der Einheit, also z.B.
	 *                        &quot;mehrvertretene Personen&quot;
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 */
	public GebuehrenErhoehungsTatbestand(String vorschrift, String bezeichnung, double jeweils, double maxSatz,
			String einheit, Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		this(vorschrift, bezeichnung, jeweils, maxSatz, -1, einheit, gebuehrenKlasse);
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenErhoehungsTatbestand() {
		super();
	}

	/**
	 * anzahl speichert die Anzahl der Einheiten, z.B. der zusätzlich Vertretenen,
	 * als int
	 * 
	 * @return gibt {@link #anzahl} als int zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#getAnzahl()
	 */
	@Override
	@XmlAttribute(name = "anzahl")
	public int getAnzahl() {
		return anzahl;
	}

	/**
	 * @param anzahl d. {@link #anzahl}, d. gesetzt werden soll als int
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#setAnzahl(int)
	 */
	@Override
	public void setAnzahl(int anzahl) {
		berechnet = false;
		this.anzahl = anzahl;
	}

	/**
	 * jeweils speichert, wie hoch der Gebührensatz je zusätzlicher Einheit steigt,
	 * als double
	 * 
	 * @return gibt {@link #jeweils} als double zurück.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#getJeweils()
	 */
	@Override
	@XmlAttribute(name = "je")
	public Double getJeweils() {
		return new Double(jeweils);
	}

	/**
	 * @param jeweils d. {@link #jeweils}, d. gesetzt werden soll als double
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#setJeweils(Object)
	 */
	@Override
	public void setJeweils(Double jeweils) {
		berechnet = false;
		this.jeweils = jeweils.doubleValue();
	}

	/**
	 * maxSatz speichert, wie hoch der Satz höchstens sein darf, als double
	 * 
	 * @return gibt {@link #maxSatz} als double zurück.
	 */
	@XmlAttribute(name = "max")
	public double getMaxSatz() {
		return maxSatz;
	}

	/**
	 * @param maxSatz d. {@link #maxSatz}, d. gesetzt werden soll als double
	 */
	public void setMaxSatz(double maxSatz) {
		berechnet = false;
		this.maxSatz = maxSatz;
	}

	/**
	 * Die Methode beherbergt die "innere Logik" der GebührenErhöhung.
	 * 
	 * @return grundsätzlich {@link #jeweils} * {@link #anzahl}; wenn
	 *         {@link #maxSatz} erreicht oder überschritten wird, dieser; wenn
	 *         {@link #anzahl} noch nicht gesetzt wurde, dann wird 0.0
	 *         zurückgegeben.
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand#getSatz()
	 */
	@Override
	public double getSatz() {
		if (!berechnet)
			berechne();
		return super.getSatz();
	}

	/**
	 * Die Methode baut einen {@link Object#hashCode()}
	 * 
	 * @return einen <code>int</code>, der sich aus
	 *         {@link GebuehrenSatzTatbestand#hashCode()} sowie
	 *         {@link #getAnzahl()}, {@link #getJeweils()} und {@link #getMaxSatz()}
	 *         berechnet. berechnet.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + anzahl;
		long temp;
		temp = Double.doubleToLongBits(jeweils);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(maxSatz);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/**
	 * Die Methode vergleicht anhand von {@link #getAnzahl()}, {@link #getJeweils()}
	 * und {@link #getMaxSatz()} sowie
	 * {@link GebuehrenSatzTatbestand#equals(Object)} die Gleichheit
	 * 
	 * @param obj das zu vergleichende {@link Object}
	 * @return <code>true</code>, wenn die Kriterien übereinstimmen, sonst
	 *         <code>false</code>
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!super.equals(obj)) {
			return false;
		}
		if (!(obj instanceof GebuehrenErhoehungsTatbestand)) {
			return false;
		}
		GebuehrenErhoehungsTatbestand other = (GebuehrenErhoehungsTatbestand) obj;
		if (anzahl != other.anzahl) {
			return false;
		}
		if (Double.doubleToLongBits(jeweils) != Double.doubleToLongBits(other.jeweils)) {
			return false;
		}
		if (Double.doubleToLongBits(maxSatz) != Double.doubleToLongBits(other.maxSatz)) {
			return false;
		}
		return true;
	}

	/**
	 * Die toString() Methode der Klasse GebuehrenErhoehungsTatbestand liefert einen
	 * String mit der Bezeichnung des Gebührentatbestand und dem Satz in Klammern.
	 * 
	 * @return einen {@link String}
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand#toString()
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return String.format(
				"GebuehrenErhoehungsTatbestand [vorschrift=%s, bezeichnung=%s, gebuehrenKlasse=%s, satz=%s, jeweils=%s, maxSatz=%s, anzahl=%s, einheit=%s]",
				getVorschrift(), getBezeichnung(), getGebuehrenKlasse(), getSatz(), jeweils, maxSatz, anzahl, einheit);
	}

	@Override
	public GebuehrenErhoehungsTatbestand clone() {
		GebuehrenErhoehungsTatbestand other = new GebuehrenErhoehungsTatbestand(getVorschrift(), getBezeichnung(),
				getJeweils(), getMaxSatz(), anzahl, getEinheit(), getGebuehrenKlasse());
		other.setSatz(getSatz());
		return other;
	}

	/**
	 * @return gibt {@link #einheit} als {@link String} zurück, also die Bezeichnung
	 *         der Einheit, also z.B. &quot;mehrvertretene Personen&quot;
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#getEinheit()
	 */
	@Override
	@XmlAttribute(name = "einheit")
	public String getEinheit() {
		return einheit;
	}

	/**
	 * @param einheit d. {@link #einheit}, d. gesetzt werden soll als
	 *                {@link String}, also die Bezeichnung der Einheit, also z.B.
	 *                &quot;mehrvertretene Personen&quot;
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.GebuehrenMitAnzahlLinear#setEinheit(java.lang.String)
	 */
	@Override
	public void setEinheit(String einheit) {
		this.einheit = einheit;
	}

	private double berechne() {
		double satz = (double) getAnzahl() * getJeweils();
		if (getAnzahl() < 0) {
			satz = 0.0;
			setSatz(satz);
		} else if (satz < getMaxSatz()) {
			satz = (double) getAnzahl() * getJeweils();
			setSatz(satz);
		} else
			setSatz(getMaxSatz());
		berechnet = true;
		return satz;
	}

}
