/**
 * GebuehrenRechner.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;

/**
 * Die Klasse dient als Oberklasse für eine Gebührenberechnung von
 * Gebührenordnungen, die sich nach Streit- bzw. Gegenstandstandswert richten.
 * 
 * Um nunmehr eine Speicherung als XML zu ermöglichen, stellt diese Klasse nur
 * noch die Methoden der Berechnung zur Verfügung. Die eigentlichen Daten
 * speichert eine {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; mit
 * {@link GebuehrenGrundZeile} als erster und weiteren {@link GebuehrenZeile
 * GebührenZeilen}, die als Parameterobjekt genutzt wird.
 * 
 * Maßgeblich für Gebührenordnung, die sich nach Streit- bzw.
 * Gegenstandstandswert richten, ist die Wertgebührenvorschrift der jeweiligen
 * Gebührenordnung (z.B. § 13 RVG, § 34 GKG). Gebührenordnungen sind so
 * aufgebaut, dass die 1,0-Gebühr bis zum einem bestimmten untersten
 * Gegenstands-/Streitwert (<code>sprung</code>) einen bestimmten Sockelbetrag (
 * <code>hoehe</code>) beträgt &mdash; dafür gibt's eine
 * {@link GebuehrenGrundZeile}. Diese Gebühr erhöht sich mit jeder weiteren
 * {@link GebuehrenZeile} dann Gegenstandswert bis <code>grenze</code> Euro, für
 * jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
 * <code>hoehe</code> Euro.
 * 
 * Darf nicht mehr <code>abstract</code> sein, damit sie mit {@link JAXB}
 * initialisiert werden kann.
 * 
 * @author Peter Schuster (setrok)
 * @date 01.01.2015
 *
 */
@XmlRootElement(name = "Wertgebuehren", namespace = "http://www.kostentenor.de/gebuehren")
public class GebuehrenTabelle {
	private ArrayList<GebuehrenZeile> tabelle = new ArrayList<GebuehrenZeile>();

	private GebuehrenVerzeichnis verzeichnis;

	/**
	 * Der Konstruktor der abstrakten Oberklasse nimmt die zur Berechnung nötige
	 * {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; entgegen
	 * 
	 * @param tabelle eine {@link ArrayList}&lt;{@link GebuehrenZeile}&gt;
	 * 
	 *                private GebuehrenTabelle(ArrayList<GebuehrenZeile> tabelle) {
	 *                setTabelle(tabelle); }
	 */

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenTabelle() {
		super();
		verzeichnis = new GebuehrenVerzeichnis();
	}

	/**
	 * Die {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; enthält alle Werte der
	 * jeweiligen Gebührenordnung, nämlich als Sockel vom Typ
	 * {@link GebuehrenGrundZeile} den Gegenstands-/Streitwert (
	 * <code>grundSprung</code>) bis zu einen bestimmten Sockelbetrag (
	 * <code>grundHoehe</code>) sowie alle weiteren {@link GebuehrenZeile
	 * GebuehrenZeilen}.
	 * 
	 * @return gibt eine {@link ArrayList}&lt; {@link GebuehrenZeile}&gt; zurück.
	 */
	@XmlElements({ @XmlElement(name = "betraegtDieGebuehr", type = GebuehrenGrundZeile.class),
			@XmlElement(name = "gebuehrErhoehtSichBei", type = GebuehrenZeile.class) })
	public ArrayList<GebuehrenZeile> getTabelle() {
		return tabelle;
	}

	/**
	 * @param tabelle d. {@link #tabelle}, d. gesetzt werden soll als
	 *                {@link ArrayList}&lt;{@link GebuehrenZeile}&gt;
	 * @throws IllegalArgumentException wenn die Reihenfolge nicht eingehalten ist.
	 */
	public void setTabelle(ArrayList<GebuehrenZeile> tabelle) throws IllegalArgumentException {
		if (istReihenfolgeEingehalten(tabelle))
			this.tabelle = tabelle;
		else
			throw new IllegalArgumentException(
					"Die erste Zeile muss eine GebuehrenGrundZeile sein. Keine der weiteren Zeilen darf eine GebuehrenGrundZeile sein.");
	}

	/**
	 * @return gibt {@link #verzeichnis} als {@link GebuehrenVerzeichnis} zurück.
	 */
	@XmlJavaTypeAdapter(GebuehrenMapAdapter.class)
	public GebuehrenVerzeichnis getVerzeichnis() {
		return verzeichnis;
	}

	/**
	 * @param verzeichnis d. {@link #verzeichnis}, d. gesetzt werden soll als
	 *                    {@link GebuehrenVerzeichnis}.
	 */
	public void setVerzeichnis(GebuehrenVerzeichnis verzeichnis) {
		this.verzeichnis = verzeichnis;
	}

	/**
	 * Gibt eine 1,0 Gebühr nach dem übergebenen <code>streitwert</code> zurück.
	 * 
	 * - Startwert nehmen
	 * 
	 * - Beginn äußere Schleife: für jede Zeile aus (Grenze, Sprung, Höhe)
	 * 
	 * - innere Schleife: addiere solange Zeile.Sprung auf die Zähler-Variable und
	 * Zeile.Höhe auf die Zwischen-Variable, bis der nächste Schritt Zeile.Grenze
	 * oder den Streitwert überschreiten würde. Ende innere Schleife
	 * 
	 * - Abbruch, wenn Zähler + Zeile.Höhe > Streitwert
	 * 
	 * @param streitwert Der Streitwert als Ganzzahl (long).
	 * @return Gibt eine 1,0 Gebühr nach dem übergebenen <code>streitwert</code>
	 *         zurück (als double).
	 */
	public Euro errechneGebuehr(Euro streitwert) {
		Euro zwischenGebuehr = Euro.ofEuros(0.0);
		Euro zwischenStreitWert = Euro.ofEuros(0.0);
		/* Beginn äußere Schleife: für jede Zeile aus (Grenze, Sprung, Höhe) */
		for (GebuehrenZeile zeile : getTabelle()) {
			/* Startwert nehmen (35 EUR) */
			if (zeile instanceof GebuehrenGrundZeile) {
				GebuehrenGrundZeile grund = (GebuehrenGrundZeile) zeile;
				zwischenGebuehr = grund.getHoehe();
				zwischenStreitWert = grund.getSprung();
			}
			/*
			 * innere Schleife: addiere solange Zeile.Sprung auf die Zähler-Variable und
			 * Zeile.Höhe auf die Zwischen-Variable, bis der nächste Schritt Zeile.Grenze
			 * oder den Streitwert überschreiten würde. Ende innere Schleife
			 */
			else
				while (//
				zwischenStreitWert.compareTo(zeile.getGrenze()) <= 0//
						&& zwischenStreitWert.add(zeile.getHoehe()).compareTo(zeile.getGrenze()) <= 0//
						&& zwischenStreitWert.add(zeile.getHoehe()).compareTo(streitwert) <= 0//
				) {
					zwischenGebuehr = zwischenGebuehr.add(zeile.getHoehe());
					zwischenStreitWert = zwischenStreitWert.add(zeile.getSprung());
				}
			/*
			 * Abbruch, wenn Zähler + Zeile.Höhe >= Streitwert, also nicht kleiner als
			 * Streitwert
			 */
			if (zwischenStreitWert.add(zeile.getHoehe()).compareTo(streitwert) >= 0) {
				break;
			}
		}
		return zwischenGebuehr;
	}

	/**
	 * Gibt eine 1,0 * <code>faktor</code> Gebühr nach dem übergebenen
	 * <code>streitwert</code> zurück.
	 * 
	 * @param streitwert der Streitwert als Ganzzahl (long)
	 * @param faktor     der Faktor nach Kostenverzeichnis bzw. Gebührenverzeichnis
	 *                   (z.B. 1,2 oder 3,0)
	 * @return gibt die entsprechende Gebühr als double zurück.
	 */
	public Euro errechneGebuehr(Euro streitwert, double faktor) {
//		System.out.println("Streitwert: " + streitwert + ", faktor: " + faktor);
		return errechneGebuehr(streitwert).multiply(faktor);
	}

	/**
	 * Die Methode baut eine ArrayList&lt;Long&gt; mit allen Streitwerten, die einen
	 * Sprung darstellen.
	 * 
	 * @return die erstellte ArrayList&lt;Long&gt;
	 */
	public ArrayList<Euro> errechneStreitwertListe() {
		ArrayList<Euro> liste = new ArrayList<Euro>();
		Euro streitwert = Euro.ofCents(0L);
		// war hier falsch: liste.add(streitwert);
		for (GebuehrenZeile zeile : getTabelle()) {
			/* Startwert nehmen (35 EUR) */
			if (zeile instanceof GebuehrenGrundZeile) {
				GebuehrenGrundZeile grund = (GebuehrenGrundZeile) zeile;
				streitwert = grund.getSprung();// grund.getSprung();
				// Die erste Zeile muss schon rein:
				liste.add(streitwert);
			} else
				while (streitwert.compareTo(zeile.getGrenze()) <= 0
						&& streitwert.add(zeile.getHoehe()).compareTo(zeile.getGrenze()) <= 0) {
					streitwert = streitwert.add(zeile.getSprung());
					liste.add(streitwert);
				}
		}
		System.out.print("Streitwerte: ");
		for (HtmlRtfFormattierend euro : liste)
			System.out.print(euro + " ");
		return liste;
	}

	/**
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} über
	 * {@link JAXB#unmarshal(InputStream, Class)} aus einem {@link InputStream} zu
	 * erlangen und aus dieser dann die {@link #getTabelle() Tabelle} zu entnehmen.
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} mit den GebuehrenZeilen
	 * der jeweiligen Wertgebührenordnung (z.B. § 13 RVG, § 34 GKG) zu
	 * initialisieren.
	 * 
	 * Diese sind so aufgebaut, dass die 1,0-Gebühr bis zum einem bestimmten
	 * untersten Gegenstands-/Streitwert einen bestimmten Sockelbetrag beträgt (das
	 * wird mit dem Konstruktor initialisiert,
	 * {@link eu.gronos.kostenrechner.GebuehrenRechnerAlt#GebuehrenTabelle(long,long, Calendar,Calendar)}
	 * ). Diese Gebühr erhöht sich dann Gegenstandswert bis <code>grenze</code>
	 * Euro, für jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
	 * <code>hoehe</code> Euro
	 * 
	 * @param stream ein {@link InputStream} von einer XML-Datei für {@link JAXB}
	 * @return eine {@link GebuehrenTabelle}, aus der man mit {@link #getTabelle()}
	 *         eine gefüllte {@link ArrayList}&lt;{@link GebuehrenZeile}&gt; und mit
	 *         {@link #getVerzeichnis()} die {@link GebuehrenTatbestand}e
	 * 
	 * @throws NullPointerException wenn <code>stream</code> == <code>null</code>
	 *                              ist.
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenZeile#GebuehrenZeile(long,long,long)
	 */
	public GebuehrenTabelle readFromStream(InputStream stream) throws NullPointerException {
		if (stream == null)
			throw new NullPointerException("Der InputStream für die angegebene Resource darf nicht null sein.");
		final GebuehrenTabelle tab = JAXB.unmarshal(stream, GebuehrenTabelle.class);
		return tab;
	}

	/**
	 * Die Methode initialisiert er mittels
	 * {@link #readTabelleFromStream(InputStream)} und
	 * {@link #setTabelle(ArrayList)} mittels XML-Datei und {@link JAXB} die Werte
	 * der Tabelle.
	 * 
	 * @param stream ein {@link InputStream} von einer XML-Datei für {@link JAXB}
	 * @throws NullPointerException
	 * @throws IllegalArgumentException wenn die Reihenfolge bei
	 *                                  {@link #setTabelle(ArrayList)} nicht
	 *                                  eingehalten ist
	 */
	protected void setFromStream(InputStream stream) throws NullPointerException, IllegalArgumentException {
		final GebuehrenTabelle tab = readFromStream(stream);
		setTabelle(tab.getTabelle());
		setVerzeichnis(tab.getVerzeichnis());
	}

	/**
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} über
	 * {@link JAXB#unmarshal(InputStream, Class)} aus einem {@link InputStream} zu
	 * erlangen und aus dieser dann die {@link #getTabelle() Tabelle} zu entnehmen.
	 * Die Methode dient dazu, eine {@link GebuehrenTabelle} mit den GebuehrenZeilen
	 * der jeweiligen Wertgebührenordnung (z.B. § 13 RVG, § 34 GKG) zu
	 * initialisieren.
	 * 
	 * Diese sind so aufgebaut, dass die 1,0-Gebühr bis zum einem bestimmten
	 * untersten Gegenstands-/Streitwert einen bestimmten Sockelbetrag beträgt (das
	 * wird mit dem Konstruktor initialisiert,
	 * {@link eu.gronos.kostenrechner.GebuehrenRechnerAlt#GebuehrenTabelle(long,long, Calendar,Calendar)}
	 * ). Diese Gebühr erhöht sich dann Gegenstandswert bis <code>grenze</code>
	 * Euro, für jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
	 * <code>hoehe</code> Euro
	 *
	 * 
	 * @param stream ein {@link InputStream} von einer XML-Datei für {@link JAXB}
	 * @return eine gefüllte {@link ArrayList}&lt;{@link GebuehrenZeile}&gt;
	 *
	 * @throws NullPointerException wenn <code>stream</code> == <code>null</code>
	 *                              ist.
	 * 
	 * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenZeile#GebuehrenZeile(long,long,long)
	 *
	 */
	protected ArrayList<GebuehrenZeile> readTabelleFromStream(InputStream stream) throws NullPointerException {
		final GebuehrenTabelle tab = readFromStream(stream);
		return tab.getTabelle();
	}

	/**
	 * Die Methode dient dazu, zu prüfen, ob die erste Zeile eine
	 * {@link GebuehrenGrundZeile} ist, wie es sein muss. Keine der weiteren Zeilen
	 * darf eine {@link GebuehrenGrundZeile} sein.
	 * 
	 * @param tabelle die zu prüfende {@link ArrayList}&lt;{@link GebuehrenZeile}
	 *                &gt;
	 * @return <code>true</code>, wenn eingehalten, sonst <code>false</code>. Auch
	 *         dann <code>false</code>, wenn die {@link ArrayList}&lt;
	 *         {@link GebuehrenZeile}&gt; <code>null</code> oder zu klein ist.
	 */
	private boolean istReihenfolgeEingehalten(ArrayList<GebuehrenZeile> tabelle) {
		if (tabelle == null || tabelle.size() < 2)
			return false;
		/* Die erste Zeile muss eine GebuehrenGrundZeile sein */
		if (!(tabelle.get(0) instanceof GebuehrenGrundZeile))
			return false;
		/* Keine der weiteren Zeilen darf eine GebuehrenGrundZeile sein */
		for (int index = 1; index < tabelle.size(); index++)
			if (tabelle.get(index) instanceof GebuehrenGrundZeile)
				return false;
		return true;
	}

}
