/**
 * AbstractGebuehrenTatbestand.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import java.io.Serializable;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlType;

import eu.gronos.kostenrechner.interfaces.Hinzufuegbar;
import eu.gronos.kostenrechner.logic.gebuehren.GebuehrenBerechnungsAufstellung;
import eu.gronos.kostenrechner.logic.gebuehren.MehrkostenMethode;
import eu.gronos.kostenrechner.logic.gebuehren.QuotenMethode;

/**
 * Eine Oberklasse, um GebührenTatbestände zu speichern, entweder aus dem VV RVG
 * oder dem KV GKG. Als Oberklasse für alle Gebührentatbestände nur mit
 * beschreibung und Verweis auf die Klasse der Gebührenordnung. Abgeleitete
 * Klassen können den eigentlichen Inhalt wie Gebührensatz, Auslagenhöhe oder
 * Mehrwertsteuersatz bereitstellen.
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 21.08.2014
 */
@XmlType(propOrder = { "vorschrift", "bezeichnung" })
public abstract class GebuehrenTatbestand implements Serializable, Hinzufuegbar {

	private static final String ZIFF = "Ziff.";
	private static final String KV_GKG = "KV GKG";
	private static final String VV_RVG = "VV RVG";
	private static final long serialVersionUID = -7148753363484353653L;
	private String vorschrift;
	private String bezeichnung;
	private Class<? extends GebuehrenTabelle> gebuehrenKlasse;

	/**
	 * Erstellt eine Instanz der Klasse mit den angegebenen Werten.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * @throws ClassNotFoundException
	 */
	public GebuehrenTatbestand(String vorschrift, String bezeichnung,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		super();
		this.vorschrift = vorschrift;
		this.bezeichnung = bezeichnung;
		this.gebuehrenKlasse = gebuehrenKlasse;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public GebuehrenTatbestand() {
		super();
	}

	/**
	 * Die Methode gibt gesetzliche Vorschrift (ohne die Gesetzesangabe selbst)
	 * zurück, nach der sich der GebuehrenTatbestand richtet, etwa "1007 VV"
	 * 
	 * @return gibt {@link #vorschrift} als {@link String} zurück.
	 */
	@XmlAttribute(name = "vorschrift")
	public String getVorschrift() {
		if (vorschrift == null) {
			repairVorschrift();
		}
		return vorschrift;
	}

	/**
	 * @param vorschrift d. {@link #vorschrift}, d. gesetzt werden soll als
	 *                   {@link String}.
	 */
	public void setVorschrift(String vorschrift) {
		this.vorschrift = vorschrift;
	}

	/**
	 * Die Methode gibt die Bezeichnung des Tatbestands als String zurück. Dieser
	 * sollte den jeweiligen Tatbestand einigermaßen eindeutig beschreiben.
	 * 
	 * @return Die Bezeichnung des Gebührentatbestands nach dem jeweiligen
	 *         Verzeichnis als String, z.B. "Nr. 1100 KV"
	 */
	@XmlAttribute(name = "bezeichnung")
	public String getBezeichnung() {
		return bezeichnung;
	}

	/**
	 * @param bezeichnung d. {@link #bezeichnung}, d. gesetzt werden soll als String
	 */
	public void setBezeichnung(String bezeichnung) {
		this.bezeichnung = bezeichnung;
	}

	/**
	 * @return gebuehrenKlasse gibt zurück, auf welche Gebührentabelle sich der
	 *         Tatbestand bezieht, also eine Klasse, die GebuehrenTabelle erweitert.
	 */
	@XmlAttribute(name = "gesetz")
	public Class<? extends GebuehrenTabelle> getGebuehrenKlasse() {
		return gebuehrenKlasse;
	}

	/**
	 * @param gebuehrenKlasse d. {@link #gebuehrenKlasse}, d. gesetzt werden soll
	 *                        als Class<? extends GebuehrenRechner>
	 */
	public void setGebuehrenKlasse(Class<? extends GebuehrenTabelle> gebuehrenKlasse) {
		this.gebuehrenKlasse = gebuehrenKlasse;
		if (this.gebuehrenKlasse == null) {
			repairClass();
		}
	}

	/**
	 * Die Methode dient dazu, einen {@link Object#hashCode()} zu erstellen.
	 * 
	 * @return einen <code>int</code>, der sich aus {@link #getVorschrift()},
	 *         {@link #getBezeichnung()} und {@link #getGebuehrenKlasse()}
	 *         berechnet.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((bezeichnung == null) ? 0 : bezeichnung.hashCode());
		result = prime * result + ((gebuehrenKlasse == null) ? 0 : gebuehrenKlasse.hashCode());
		result = prime * result + ((vorschrift == null) ? 0 : vorschrift.hashCode());
		return result;
	}

	/**
	 * Vergleicht nach den Kriterien {@link #getVorschrift()},
	 * {@link #getBezeichnung()} und {@link #getGebuehrenKlasse()}
	 * 
	 * @param obj das zu vergleichende {@link Object}
	 * @return <code>true</code>, wenn die Kriterien übereinstimmen, sonst
	 *         <code>false</code>
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof GebuehrenTatbestand)) {
			return false;
		}
		GebuehrenTatbestand other = (GebuehrenTatbestand) obj;
		if (bezeichnung == null) {
			if (other.bezeichnung != null) {
				return false;
			}
		} else if (!bezeichnung.equals(other.bezeichnung)) {
			return false;
		}
		if (gebuehrenKlasse == null) {
			if (other.gebuehrenKlasse != null) {
				return false;
			}
		} else if (!gebuehrenKlasse.equals(other.gebuehrenKlasse)) {
			return false;
		}
		if (vorschrift == null) {
			if (other.vorschrift != null) {
				return false;
			}
		} else if (!vorschrift.equals(other.vorschrift)) {
			return false;
		}
		return true;
	}

	/**
	 * Die Methode ist eine spezielle {@link #toString()}-Methode für die
	 * Begründungstabelle aus {@link QuotenMethode}, {@link MehrkostenMethode} und
	 * {@link GebuehrenBerechnungsAufstellung}
	 * 
	 * @return ein formatierter {@link String} mit {@link #getBezeichnung()} und
	 *         {@link #getVorschrift()}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Hinzufuegbar#langBezeichnung()
	 */
	public String langBezeichnung() {
		String gesetz = "";
		if (gebuehrenKlasse == AnwaltsGebuehrenTabelle.class) {
			gesetz = " RVG";
		} else if (gebuehrenKlasse == GerichtsGebuehrenTabelle.class) {
			gesetz = " GKG";
		}
		return String.format("%s (%s%s)", getBezeichnung(), getVorschrift(), gesetz);
	}

	/**
	 * 
	 * @return eine formatierte Zeichenkette mit der {@link #getBezeichnung()} und
	 *         {@link #getGebuehrenKlasse()}.
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return String.format("GebuehrenTatbestand [vorschrift=%s, bezeichnung=%s, gebuehrenKlasse=%s]", vorschrift,
				bezeichnung, gebuehrenKlasse);
	}

	/**
	 * Die Methode dient dazu, die Class zu reparieren, wenn null, weil XML von
	 * alter Version
	 */
	private void repairClass() {
		if (bezeichnung != null) {
			if (bezeichnung.contains(VV_RVG)) {
				this.gebuehrenKlasse = AnwaltsGebuehrenTabelle.class;
			} else if (bezeichnung.contains(KV_GKG)) {
				this.gebuehrenKlasse = GerichtsGebuehrenTabelle.class;
			}
		} else if (vorschrift != null) {
			if (vorschrift.contains("VV")) {
				this.gebuehrenKlasse = AnwaltsGebuehrenTabelle.class;
			} else if (vorschrift.contains("KV")) {
				this.gebuehrenKlasse = GerichtsGebuehrenTabelle.class;
			}
		}
	}

	/**
	 * Die Methode dient dazu, die Vorschrift zu reparieren, wenn null, weil XML von
	 * alter Version. Dabei zieht sie sich die Vorschrift aus der Bezeichnung.
	 */
	private void repairVorschrift() {
		if (vorschrift == null || vorschrift.isEmpty()) {
			if (bezeichnung != null && (bezeichnung.contains(KV_GKG) || bezeichnung.contains(VV_RVG))
					&& bezeichnung.contains(ZIFF)) {
				String substring = bezeichnung.substring(bezeichnung.indexOf(ZIFF) + ZIFF.length()).trim();
				if (substring.contains(" (")) {
					substring = substring.substring(0, substring.indexOf(" ("));
				}
				this.vorschrift = substring;
			}
		}
	}

}