/*
 * GebuehrenVerzeichnis.java
 * eu.gronos.kostenrechner.model.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.xml.bind.annotation.XmlAttribute;

/**
 * Von {@link TreeMap} abgeleitete Klasse zu den Gebührenverzeichnissen in
 * {@link GebuehrenTabelle}, also das KostenVerzeichnis in
 * {@link GerichtsGebuehrenTabelle} und VerguetungsVerzeichnis in
 * {@link AnwaltsGebuehrenTabelle}.
 * 
 * Enthält {@link #getAll(String...)} zum Abruf mehrerer
 * {@link GebuehrenTatbestand}e auf einmal und {@link #filter(String)} zum
 * Herausfiltern der {@link GebuehrenTatbestand}e nach einem Suchwort.
 *
 * @author Peter Schuster (setrok)
 * @date 29.03.2020
 *
 */
public class GebuehrenVerzeichnis extends TreeMap<String, GebuehrenTatbestand> {

	private static final long serialVersionUID = -934035989553745462L;
	private String title = new String();

	/**
	 * Die Methode ist ein modifizierter {@link #get(int)}-Getter
	 * 
	 * @param vorschriften Auflistung der
	 *                     {@link GebuehrenTatbestand#getVorschrift()}en, kann
	 *                     Mehrfachnennungen enthalten.
	 * @return alle Elemente, die {@link GebuehrenTatbestand#getVorschrift()}
	 *         enthalten
	 * @throws IllegalArgumentException, wenn eine der vorschriften nicht gefunden
	 *                                   wird
	 */
	public GebuehrenAuflistung getAll(String... vorschriften) throws IllegalArgumentException {
		return Stream.of(vorschriften).//
				map(key -> this.getNonNull(key)).//
				collect(collector());
	}

	/**
	 * Die Methode fügt nur die Elemente aus dem Verzeichnis hinzu, die den
	 * Suchbegriff {@link String} enthalten. Groß-/Kleinschreibung wird ignoriert.
	 * 
	 * @param term ein {@link String} mit dem Suchbegriff
	 * @return {@link GebuehrenAuflistung}
	 */
	public GebuehrenAuflistung filter(String term) {
		return entrySet().//
				stream().//
				filter(map -> map.getValue().toString().toLowerCase().contains(term.toLowerCase())).//
				map(Map.Entry::getValue).//
				collect(collector());
	}

	/**
	 * @return gibt {@link #title} als {@link String} zurück.
	 */
	@XmlAttribute(name = "title")
	public String getTitle() {
		return title;
	}

	/**
	 * @param title d. {@link #title}, d. gesetzt werden soll als {@link String}.
	 */
	public void setTitle(String title) {
		this.title = title;
	}

	/**
	 * Die Methode soll verhindern, dass {@link #getAll(String...)} eine
	 * {@link List} mit <code>null</code>-Werten zurückgibt. Dazu wirft es eine
	 * {@link IllegalArgumentException}, wenn {@link #get(Object)} nicht fündig
	 * wird.
	 * 
	 * @param vorschrift einen {@link String} mit dem Key
	 * @return den {@link GebuehrenTatbestand}
	 * @throws IllegalArgumentException wenn es zu dem Key keinen
	 *                                  {@link GebuehrenTatbestand} gibt.
	 */
	private GebuehrenTatbestand getNonNull(String vorschrift) throws IllegalArgumentException {
		GebuehrenTatbestand gt = get(vorschrift);
		if (gt == null)
			throw new IllegalArgumentException(
					String.format("Gebührentatbestand nicht im Verzeichnis: %s!", vorschrift));
		return gt;
	}

	/**
	 * Die Methode liefert einen {@link Collector} zur Umwandlung in eine
	 * {@link GebuehrenAuflistung}
	 * 
	 * @return den {@link Collector}
	 */
	private Collector<GebuehrenTatbestand, ?, GebuehrenAuflistung> collector() {
		return Collectors.toCollection(GebuehrenAuflistung::new);
	}
}
