/**
 * GebuehrenZeile.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import java.io.Serializable;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.EuroAdapter;

/**
 * Die Klasse repräsentiert eine Zeile der Wertgebührenvorschrift der jeweiligen
 * Gebührenordnung (z.B. § 13 RVG, § 34 GKG). Diese sind so aufgebaut, dass die
 * 1,0-Gebühr bis zum einem bestimmten untersten Gegenstands-/Streitwert einen
 * bestimmten Sockelbetrag beträgt. Diese Gebühr erhöht sich dann
 * Gegenstandswert bis <code>grenze</code> Euro, für jeden angefangenen Betrag
 * von weiteren <code>sprung</code> Euro, um <code>hoehe</code> Euro
 * 
 * @author Peter Schuster (setrok)
 * @date 12.01.2014
 * 
 * @TODO TODO Es müsste auch noch eine Art Objectid (bzw. HashTag) als Referenz
 *       auf die erzeugende Klasse (GebuehrenTabelle) übergeben werden.
 */
@XmlType(propOrder = { "grenze", "sprung", "hoehe" })
public class GebuehrenZeile implements Comparable<GebuehrenZeile>, Serializable {
    private static final long serialVersionUID = 4891531889403046069L;
    private Euro grenze;// war long
    private Euro sprung;// war long
    private Euro hoehe;// long

    /**
     * Der Konstruktor erzeugt eine Zeile der Wertgebührenvorschrift der jeweiligen
     * Gebührenordnung (z.B. § 13 RVG, § 34 GKG).
     * 
     * Diese Gebühr erhöht sich dann Gegenstands-/Streitwert bis <code>grenze</code>
     * Euro, für jeden angefangenen Betrag von weiteren <code>sprung</code> Euro, um
     * <code>hoehe</code> Euro.
     * 
     * @param grenze bis zu welcher Grenze erhöht sich die Gebühr
     * @param sprung für welchen Betrag erhöht sich die Gebühr jeweils
     * @param hoehe  um wieviel Euro erhöht sich die Gebühr
     * 
     * @see eu.gronos.kostenrechner.data.gebuehren.GebuehrenZeile
     */
    public GebuehrenZeile(Euro grenze, Euro sprung, Euro hoehe) {
        this.grenze = grenze;
        this.sprung = sprung;
        this.hoehe = hoehe;
    }

    /**
     * Konstruktor ohne Parameter für {@link JAXB}
     * 
     */
    public GebuehrenZeile() {
        super();
    }


	/**
     * @return gibt zurück, bis zu welcher Grenze sich die Gebühr erhöht (als long)
     */
    @XmlAttribute(name = "GegenstandswertBisEUR")
    @XmlJavaTypeAdapter(EuroAdapter.class)
    public Euro getGrenze() {// long
        return grenze;
    }

    /**
     * @return gibt zurück, für welchen Betrag sich die Gebühr jeweils erhöht
     */
    @XmlAttribute(name = "fuerJedenAngefangenenBetragVonWeiterenEUR")
    @XmlJavaTypeAdapter(EuroAdapter.class)
    public Euro getSprung() {// long
        return sprung;
    }

    /**
     * @return gibt zurück, um wieviel Euro sich die Gebühr erhöht.
     */
    @XmlAttribute(name = "umEUR")
    @XmlJavaTypeAdapter(EuroAdapter.class)
    public Euro getHoehe() {// long
        return hoehe;
    }

    /**
     * @param grenze d. {@link #grenze}, d. gesetzt werden soll als long
     */
    public void setGrenze(Euro grenze) {
        this.grenze = grenze;
    }

    /**
     * @param sprung d. {@link #sprung}, d. gesetzt werden soll als long
     */
    public void setSprung(Euro sprung) {
        this.sprung = sprung;
    }

    /**
     * @param hoehe d. {@link #hoehe}, d. gesetzt werden soll als long
     */
    public void setHoehe(Euro hoehe) {
        this.hoehe = hoehe;
    }

    /**
     * Vergleicht die Instanz der GebuehrenZeile mit einer anderen GebuehrenZeile
     * anhand des Felds {@link #getGrenze() Grenze}.
     * 
     * @param that eine andere GebuehrenZeile
     * @return a negative integer, zero, or a positive integer as this object is
     *         less than, equal to, or greater than the specified object.
     * 
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    @Override
    public int compareTo(GebuehrenZeile that) {
        if (this.equals(that))
            return 0;
//        return new Long(getGrenze()).compareTo(new Long(that.getGrenze()));
        return getGrenze().compareTo(that.getGrenze());
    }

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((grenze == null) ? 0 : grenze.hashCode());
		result = prime * result + ((hoehe == null) ? 0 : hoehe.hashCode());
		result = prime * result + ((sprung == null) ? 0 : sprung.hashCode());
		return result;
	}



    
    
//    @Override
//    public int hashCode() {
//        final int prime = 31;
//        int result = 1;
//        result = prime * result + (int) (grenze ^ (grenze >>> 32));
//        long temp;
//        temp = Double.doubleToLongBits(hoehe.doubleValue());
//        result = prime * result + (int) (temp ^ (temp >>> 32));
//        result = prime * result + (int) (sprung ^ (sprung >>> 32));
//        return result;
//    }
//

//  @Override
//  public int hashCode() {
//      final int prime = 31;
//      int result = 1;
//      result = prime * result + (int) (grenze ^ (grenze >>> 32));
//      return result;
//  }

    /**
     * Vergleicht diese {@link GebuehrenZeile} mit einer anderen anhand des Felds
     * 
     * {@link #getGrenze() Grenze}
     * 
     * @param obj eine andere GebuehrenZeile oder jedes {@link Object}
     * @return true, wenn identisch oder wenn {@link GebuehrenZeile} mit derselben
     *         Grenze.
     * @see java.lang.Object#equals(java.lang.Object)
     */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!(obj instanceof GebuehrenZeile)) {
			return false;
		}
		GebuehrenZeile other = (GebuehrenZeile) obj;
		if (grenze == null) {
			if (other.grenze != null) {
				return false;
			}
		} else if (!grenze.equals(other.grenze)) {
			return false;
		}
		if (hoehe == null) {
			if (other.hoehe != null) {
				return false;
			}
		} else if (!hoehe.equals(other.hoehe)) {
			return false;
		}
		if (sprung == null) {
			if (other.sprung != null) {
				return false;
			}
		} else if (!sprung.equals(other.sprung)) {
			return false;
		}
		return true;
	}
//    @Override
//    public boolean equals(Object obj) {
//        if (this == obj) {
//            return true;
//        }
//        if (!(obj instanceof GebuehrenZeile)) {
//            return false;
//        }
//        GebuehrenZeile other = (GebuehrenZeile) obj;
//        if (grenze != other.grenze) {
//            return false;
//        }
////        if (Double.doubleToLongBits(hoehe) != Double.doubleToLongBits(other.hoehe)) {
////            return false;
////        }
////        if (sprung != other.sprung) {
////            return false;
////        }
//        return true;
//    }

//  @Override
//  public boolean equals(Object obj) {
//      if (this == obj)
//          return true;
//      if (obj == null)
//          return false;
//      if (getClass() != obj.getClass())
//          return false;
//      GebuehrenZeile that = (GebuehrenZeile) obj;
//      if (grenze != that.grenze)
//          return false;
//      return true;
//  }
}
