/**
 * MehrwertsteuerTatbestand.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.gebuehren;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;

import eu.gronos.kostenrechner.data.tenordaten.Euro;

/**
 * Von GebuehrenTatbestand abgeleitete Klasse zur Berechnung der Mehrwert auf
 * RA-Gebühren.
 *
 * @author Peter Schuster (setrok)
 * @date 24.08.2014
 *
 */
public class MehrwertsteuerTatbestand extends GebuehrenTatbestand {

	private static final long serialVersionUID = -8164105788751809714L;
	public static final String ART = "Umsatzsteuer";
	/**
	 * steuerSatz speichert den Mehrwertsteuersatz als double zwischen 0 und 1, z.B.
	 * 0.19 für 19% Mehrwertsteuer..
	 */
	private double steuerSatz;
	/**
	 * isoliert speichert, ob sich die Steuer alle Gebuehren &quot;reserviert&quot;
	 * als boolean.
	 */
	private boolean isoliert;

	/**
	 * Erstellt eine Instanz der Klasse mit den angegebenen Werten.
	 * {@link #isoliert} wird auf <code>false</code> gesetzt.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param steuerSatz      Der Mehrwertsteuersatz als double zwischen 0 und 1,
	 *                        z.B. 0.19 für 19% Mehrwertsteuer.
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 */
	MehrwertsteuerTatbestand(String vorschrift, String bezeichnung, double steuerSatz,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) throws IllegalArgumentException {
		this(vorschrift, bezeichnung, steuerSatz, false, gebuehrenKlasse);
	}

	/**
	 * Erstellt eine Instanz der Klasse mit den angegebenen Werten.
	 * 
	 * @param vorschrift      Die gesetzliche Vorschrift, nach der sich der
	 *                        GebuehrenTatbestand richtet.
	 * @param bezeichnung     Die Bezeichnung des Gebührentatbestands nach dem
	 *                        jeweiligen Verzeichnis als String, z.B. "Nr. 1100 KV"
	 * @param steuerSatz      Der Mehrwertsteuersatz als double >0 und <1, z.B. 0.19
	 *                        für 19% Mehrwertsteuer.
	 * @param isoliert        ob sich die Steuer alle Gebuehren
	 *                        &quot;reserviert&quot; als boolean (false, wenn die
	 *                        Steuer für alle RVG-Gebühren auf einmal berechnet
	 *                        werden soll; true, wenn dies in Etappen geschieht).
	 * @param gebuehrenKlasse Klasse der Gebühr, also den Verweis auf die Klasse
	 *                        eines Ablegers von GebuehrenTabelle, also entweder
	 *                        GerichtsGebuehrenTabelle.class oder
	 *                        AnwaltsGebuehrenTabelle.class
	 * @throws IllegalArgumentException wenn der Steuersatz nicht zwischen 0 und 1
	 *                                  (jeweils exklusive) ist.
	 */
	MehrwertsteuerTatbestand(String vorschrift, String bezeichnung, double steuerSatz, boolean isoliert,
			Class<? extends GebuehrenTabelle> gebuehrenKlasse) throws IllegalArgumentException {
		super(vorschrift, bezeichnung, gebuehrenKlasse);
		if (steuerSatz >= 0.0 && steuerSatz < 1.0)
			this.steuerSatz = steuerSatz;
		else
			throw new IllegalArgumentException("Der steuerSatz muss >= 0.0 und < 1.0 sein.");
		this.isoliert = isoliert;
	}

	/**
	 * Konstruktor ohne Parameter für {@link JAXB}
	 * 
	 */
	public MehrwertsteuerTatbestand() {
		super();
	}

	/**
	 * @return gibt den Mehrwertsteuersatz als double zwischen 0 und 1 zurück, z.B.
	 *         0.19 für 19% Mehrwertsteuer
	 */
	@XmlAttribute(name = "satz")
	public double getSteuerSatz() {
		return steuerSatz;
	}

	/**
	 * @param steuerSatz der Mehrwertsteuersatz als double zwischen 0 und 1, z.B.
	 *                   0.19 für 19% Mehrwertsteuer
	 * @throws IllegalArgumentException
	 */
	public void setSteuerSatz(double steuerSatz) throws IllegalArgumentException {
		if (steuerSatz >= 0.0 && steuerSatz < 1.0)
			this.steuerSatz = steuerSatz;
		else
			throw new IllegalArgumentException("Der steuerSatz muss >= 0.0 und < 1.0 sein.");
	}

	/**
	 * @return gibt isoliert als boolean zurück, ob sich die Steuer alle Gebuehren
	 *         &quot;reserviert&quot
	 * @see #isoliert
	 */
	@XmlAttribute(name = "reset")
	public boolean isIsoliert() {
		return isoliert;
	}

	/**
	 * Die Methode setzt den {@link #isIsoliert()}
	 * 
	 * @param isoliert isoliert als boolean, ob sich die Steuer alle Gebuehren
	 *                 &quot;reserviert&quot
	 */
	public void setIsoliert(boolean isoliert) {
		this.isoliert = isoliert;
	}

	/**
	 * Die toString() Methode liefert einen String mit der Bezeichnung des
	 * Gebührentatbestands und dem Steuersatz in Prozent in Klammern. Zudem sagt er,
	 * ob auf alle Gebühren oder nur einen Teil die Steuer berechnet wird.
	 * 
	 * @return einen String nach dem Format "Text (Steuersatz)", z.B.
	 *         "Verfahrensgebühr, Ziffer xxxx VVRVG (19%; für alle Gebühren)"
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final String strIsoliert = isIsoliert() ? "; mit anschließendem Zurücksetzen der Summe" : "";
		return String.format("GebuehrenTatbestand [vorschrift=%s, bezeichnung=%s, gebuehrenKlasse=%s, steuerSatz=%s%s]",
				getVorschrift(), getBezeichnung(), getGebuehrenKlasse(), steuerSatz, strIsoliert);
	}

	/**
	 * Die Methode dient dazu, die Steuer zu einer Summe von RVG-Gebühren zu
	 * berechnen.
	 * 
	 * @param summe die Summe
	 * @return die Steuer auf <code>summe</code> unter Berücksichtigung des
	 *         {@link #steuerSatz steuerSatzes}
	 */
	public Euro errechneSteuer(Euro summe) {
		return summe.multiply(getSteuerSatz());
	}

}
