package eu.gronos.kostenrechner.data.tenordaten;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.stream.Collectors;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlRootElement;

import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;

/**
 * Die Klasse soll {@link DoubleDataRows} ersetzen und weitere Datentypen als
 * Element erlauben, die von {@link Number} abgeleitet sind. Von
 * {@link ArrayList} abgeleitete Klasse, die notwendig ist, weil sich in
 * {@link JAXB} keine mehrdimenensionalen Listen ohne eigene Klasse exportieren
 * lassen lassen.
 *
 * Von {@link ArrayList} abgeleitete Klasse, die notwendig ist, weil sich in
 * {@link JAXB} keine mehrdimenensionalen Listen ohne eigene Klasse exportieren
 * lassen lassen.
 *
 * TODO {@link DoubleDataRows#getZeile()} und
 * {@link DoubleDataRows#setZeile(String)} habe ich nicht übernommen, da sich
 * für die Umwandlung in XML vielleicht bessere Lösungen bieten.
 *
 * @author Peter Schuster (setrok)
 * @date 31.12.2021
 *
 */
@XmlRootElement(name = "zeile")
public class BegruendungsZahlenZeile extends ArrayList<Number> {

	private static final long serialVersionUID = 6397831147537226777L;

	public BegruendungsZahlenZeile() {
		super();
	}

	public BegruendungsZahlenZeile(ArrayList<? extends Number> dv) {
		super(dv.size());
		super.addAll(dv);
	}

	public BegruendungsZahlenZeile(int size) {
		super(size);
	}

	/**
	 * Die Methode stellt die Zeile als Hintereinanderreihung von {@link Number}s
	 * dar.
	 * 
	 * @return alle Werte einer Zeile, als zweistellige {@link Number}s, also
	 *         {@link Double}, {@link Euro} oder {@link Fraction} mit Leerzeichen
	 *         getrennt
	 */
	@XmlAttribute(name = "spalten")
	public String getZeile() {
		return toString();
	}

	/**
	 * Die Methode trennt die Zeile, als zweistellige {@link Double}s, mit
	 * Leerzeichen getrennt, in eine {@link List} von {@link Double}s auf.
	 * 
	 * @param zeile alle Werte einer Zeile, als zweistellige {@link Double}s, mit
	 *              Leerzeichen getrennt, als {@link String}
	 */
	public void setZeile(String zeile) {
		if (zeile == null || zeile.isEmpty())
			return;
		Arrays.stream(zeile.trim().split("\\s+"))//
				.filter(s -> s != null && !s.trim().isEmpty())//
				.map(String::trim)//
				.map(this::spalteToNumber)//
				.forEach(this::add);
	}

	@Override
	public String toString() {
		return super.//
		stream().//
				map(this::numberToSpalte).//
				reduce("", (bisher, dazu) -> (bisher + (!bisher.isEmpty() ? ",\t" : "\t") + dazu));
	}

	/**
	 * Die Methode wandelt die ganze Tabellenzeile, bestehend aus einem String aus
	 * in eine Tab-separierte Textzeile als {@link String} um
	 * 
	 * @return
	 */
	public String tabSeparatedValues() {
		return super//
		.stream()//
				.map(this::numberToSpalte)//
				.collect(Collectors.joining("\t"))//
				+ "\n";
	}

	/**
	 * Konvertiert für {@link #toString()} die einzelne Zelle in {@link String}
	 * 
	 * @param n die {@link Number}
	 * @return einen String-Wert dafür
	 */
	public String numberToSpalte(Number n) {
		if (n instanceof Euro) {
			return ((HtmlRtfFormattierend) n).toString() + "€";
		} else if (n instanceof Double) {
			return String.format("%,.2f ", n);
		}
		return n.toString();
	}

	private Number spalteToNumber(String s) {
		if (s.endsWith("€")) {
			return Euro.valueOf(s.substring(0, s.length() - 1).trim());
		} else if (s.contains("/")) {
			return Fraction.valueOf(s);
		} else {
			return Double.parseDouble(s.replace(",", "."));
		}
	}
}
