package eu.gronos.kostenrechner.data.tenordaten;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import eu.gronos.kostenrechner.interfaces.Calculable;
import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;

/**
 * Eine {@link Number}-Klasse für Euro-Währungen. Speichert intern den
 * Cent-Betrag als {@link #getCents()}, gibt den Euro-Wert über
 * {@link #longValue()} und {@link #doubleValue()} heraus.
 *
 * @author Peter Schuster (setrok)
 * @date 07.11.2021
 *
 */
public class Euro extends Number implements Comparable<Euro>, Calculable<Euro, Double>, HtmlRtfFormattierend {

	private static final String EURO_ZERO_CENT_FORMAT = "%,d,00";
	private static final String EURO_CENT_FORMAT = "%,d,%02d";
	private static final long serialVersionUID = -6454236365619266897L;
	private static final long CENTS_PER_EURO = 100L;
	private static final double CENTS_PER_EURO_DOUBLE = 100.0;
	private static final long MAX_CACHE_EUROS = 30_000_000L;
	private static final ConcurrentMap<Long, Euro> POOL = new ConcurrentHashMap<>();
	public static final Euro ZERO_CENTS = ofCents(0L);

	private final long cents;

	/**
	 * Der Konstruktor ist <code>private</code>
	 * 
	 * @param cents
	 */
	private Euro(long cents) {
		super();
		this.cents = cents;
	}

	/**
	 * Returns the Euro value as an {@code int}, which may involve rounding or
	 * truncation.
	 *
	 * @return the numeric value represented by this object after conversion to type
	 *         {@code int}.
	 * 
	 * @see java.lang.Number#intValue()
	 */
	@Override
	public int intValue() {
		return (int) (getCents() / CENTS_PER_EURO);
	}

	/**
	 * Returns the Euro value as a {@code long}, which may involve rounding or
	 * truncation.
	 *
	 * @return the numeric value represented by this object after conversion to type
	 *         {@code long}.
	 * 
	 * @see java.lang.Number#longValue()
	 */
	@Override
	public long longValue() {
		return getCents() / CENTS_PER_EURO;
	}

	/**
	 * Returns the euro value as a {@code float}, which may involve rounding.
	 *
	 * @return the numeric value represented by this object after conversion to type
	 *         {@code float}.
	 * 
	 * @see java.lang.Number#floatValue()
	 */
	@Override
	public float floatValue() {
		return (float) doubleValue();
	}

	/**
	 * Returns the euro value as a {@code double}, which may involve rounding.
	 *
	 * @return the numeric value represented by this object after conversion to type
	 *         {@code double}.
	 * 
	 * @see java.lang.Number#doubleValue()
	 */
	@Override
	public double doubleValue() {
		return getCents() / CENTS_PER_EURO_DOUBLE;
	}

	/**
	 * Die Methode erzeugt einen {@link Euro} aus einem Euro-Betrag als
	 * {@code double}
	 * 
	 * @param euros der Euro-Betrag als {@code double}
	 * @return einen neuen {@link Euro}
	 */
	public static Euro ofEuros(double euros) {
		return ofCents((long) (euros * CENTS_PER_EURO_DOUBLE));
	}

	/**
	 * Die Methode erzeugt einen {@link Euro} aus einem Centbetrag als {@code long}
	 * Ganze Eurobeträge werden zwischengespeichert bis {@value #MAX_CACHE_EUROS}
	 * 
	 * @param cents Cent-Betrag als {@code long}
	 * @return einen neuen {@link Euro}
	 */
	public static Euro ofCents(final long cents) {
		/*
		 * Ganze euros >= 0 werden zwischengespeichert bis MAX_CACHE_EUROS
		 */
		if (cents >= 0 && cents % CENTS_PER_EURO == 0) {
			final long euros = cents / CENTS_PER_EURO;
			if (euros <= MAX_CACHE_EUROS)
				return POOL.computeIfAbsent(euros, e -> new Euro(cents));
		}
		return new Euro(cents);

	}

	/**
	 * Die Methode erzeugt einen {@link Euro} aus einem Centbetrag als {@code int}
	 * und errechnet Cent- aus Eurobeträgen
	 * 
	 * @param euros als {@code int}
	 * @return einen neuen {@link Euro}
	 */
	static Euro ofEuros(int euros) {
		return ofCents(euros * CENTS_PER_EURO);
	}

	/**
	 * Die Methode erzeugt einen {@link Euro} aus einem Euro-Betrag als {@code long}
	 * 
	 * @param euros der Euro-Betrag als {@code long}
	 * @return einen neuen {@link Euro}
	 */
	public static Euro ofEuros(long euros) {
		return ofCents(euros * CENTS_PER_EURO);
	}

	/**
	 * @param euros ganze Euro
	 * @param cents {@link #remainingCents()} - Zusätzliche Cent-Beträge im Bereich
	 *              0-99
	 * @return eine {@link Euro} Instanz
	 * @throws IllegalArgumentException wenn cents <0 oder >=
	 *                                  {@value #CENTS_PER_EURO}
	 */
	public static Euro of(long euros, int cents) {
		if (cents < 0 || cents >= CENTS_PER_EURO) {
			throw new IllegalArgumentException("Cents müssen im Bereich 0-99 cent liegen, waren aber: " + cents);
		}
		return ofCents(euros * CENTS_PER_EURO + cents);
	}

	/**
	 * Die Methode soll einen Euro-Betrag aus einem {@link String} lesen
	 * 
	 * @param value der zu konvertierende {@link String} mit einem Euro-Betrag
	 * @return ein neuer {@link Euro}
	 */
	static Euro valueOf(String value) {
		value = value.trim();
		final String komma = ",";
		if (value.contains(komma)) {
			int pos = value.indexOf(komma);
			final Long euros = Long.valueOf(value.substring(0, pos));
			int end = value.length();
			long multi = 1L;
			if ((end - pos - 1) > 2)
				end = pos + 3;
			else if ((end - pos - 1) == 1)
				multi = 10L;
			else if ((end - pos - 1) < 1)
				return new Euro(Euro.CENTS_PER_EURO * euros);
			final Long cents = Long.valueOf(value.substring(pos + 1, end));
			return new Euro((Euro.CENTS_PER_EURO * euros) + (multi * cents));
		} else
			return ofEuros(Double.valueOf(value));
	}

	/**
	 * @return gibt {@link #cents} als {@code long} zurück.
	 */
	public long getCents() {
		return cents;
	}

	/**
	 * Die Methode erzeugt einen {@link Object#hashCode()} für den {@link Euro}
	 * anhand des {@link #getCents()} Betrags.
	 * 
	 * @return einen {@link Object#hashCode()}
	 * 
	 * @see java.lang.Object#hashCode()
	 */
//	@XmlID
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (int) (getCents() ^ (getCents() >>> 32));
		return result;
	}

	/**
	 * Die Methode vergleicht den {@link Euro} mit einem {@link Object}, sofern
	 * {@link Euro} anhand von {@link #getCents()}
	 * 
	 * @param obj zu vergleichendes {@link Object}
	 * @return {@code true}, sofern auch ein {@link Euro} mit identischem
	 *         {@link #getCents()} Betrag.
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj instanceof Euro) {
			return getCents() == ((Euro) obj).getCents();
		}
		return false;
	}

	/**
	 * Die Methode vergleicht den {@link Euro} mit einem anderen {@link Euro} anhand
	 * von {@link #getCents()}
	 * 
	 * @param anotherEuro der andere {@link Euro}
	 * @return {@code 0}, wenn identischer {@link #getCents()}, {@code -1}, wenn der
	 *         andere {@link Euro} weniger {@link #getCents()} hat, sonst {@code 1}
	 * 
	 * @see java.lang.Long#compareTo(Long)
	 */
	@Override
	public int compareTo(Euro anotherEuro) {
		return (this.getCents() < anotherEuro.getCents()) ? -1 : ((this.getCents() == anotherEuro.getCents()) ? 0 : 1);
	}

	/**
	 * Die Methode dient zum Addieren von einem Cent-Betrag auf {@link Euro}s
	 * 
	 * @param cents ein {@link #getCents()}-Betrag als {@code long}
	 * @return {@code this + cents}
	 * 
	 * @see java.math.BigInteger#add(java.math.BigInteger)
	 */
	public HtmlRtfFormattierend add(long cents) {
		return ofCents(getCents() + cents);
	}

	/**
	 * Die Methode addiert zwei Euro-Beträge
	 * 
	 * @param euro ein weiterer {@link Euro}
	 * @return {@code this +} {@link Euro}
	 * 
	 * @see java.math.BigInteger#add(java.math.BigInteger)
	 */
	@Override
	public Euro add(Euro euro) {
		return ofCents(getCents() + euro.getCents());
	}

	/**
	 * Die Methode formatiert den gespeicherten {@link #getCents()} zu Euro als
	 * {@link String}
	 * 
	 * @return Euro-Betrag als {@link String} mit zwei Nachkommastellen
	 */
	@Override
	public String toString() {
		return String.format(EURO_CENT_FORMAT, longValue(), remainingCents());
	}

	/**
	 * @return {@link #toString()} mit geschütztem Leerzeichen und €
	 */
	@Override
	public String toRtfString() {
		return toString() + "\\~€";// geschütztes Leerzeichen
	}

	/**
	 * @return {@link #toString()} mit geschütztem Leerzeichen und €
	 */
	@Override
	public String toHtmlString() {
		return toString() + "\u00A0€";
	}

	/**
	 * Die Methode ermittelt die Nachkommastellen als {@code long}
	 * 
	 * @return die {@link #getCents()}, die nach dem Komma bleiben als {@code long}
	 */
	private long remainingCents() {
		return getCents() % CENTS_PER_EURO;
	}

	/**
	 * Die Methode erzeugt einen neuen {@link Double} aus {@code this / divisor}
	 * 
	 * @param divisor der {@link Euro}-Betrag, durch den geteilt werden soll
	 * @return {@code this / divisor} als neuer {@link Double}
	 * 
	 * @see java.math.BigInteger#divide(java.math.BigInteger)
	 */
	@Override
	public Double divide(Euro divisor) {
		if (divisor == null || divisor.getCents() == 0L)
			return Double.NaN;
		return (double) getCents() / (double) divisor.getCents();
	}

	/**
	 * Die Methode erzeugt eine neue {@link Fraction} aus {@code this / divisor}
	 * 
	 * @param divisor der {@link Euro}-Betrag, durch den geteilt werden soll
	 * @return {@code this / divisor} als neue {@link Fraction}
	 * 
	 * @see java.math.BigInteger#divide(java.math.BigInteger)
	 */
	public Fraction divideAsFraction(Euro divisor) {
		return // new Fraction
		Fraction.valueOf(getCents(), divisor.getCents());
	}

	/**
	 * Die Methode erzeugt einen neuen {@link Euro} aus {@code this - subtrahend}
	 * 
	 * @param subtrahend der {@link Euro}-Betrag, der abgezogen werden soll
	 * @return {@code this - subtrahend}
	 * 
	 * @see java.math.BigInteger#subtract(java.math.BigInteger)
	 */
	@Override
	public Euro subtract(Euro subtrahend) {
		return ofCents(getCents() - subtrahend.getCents());
	}

	/**
	 * Die Methode erzeugt einen neuen {@link Euro} aus {@code this * factor}
	 * 
	 * @param factor ein {@code int}, mit dem {@code this} multipliziert werden soll
	 * @return {@code this * factor}
	 * 
	 * @see java.math.BigInteger#multiply(java.math.BigInteger)
	 */
	@Override
	public Euro multiply(Double factor) {
		return ofCents((long) (getCents() * factor.doubleValue()));
	}

	/**
	 * Die Methode erzeugt einen neuen {@link Euro} aus {@code this * factor} mit
	 * einer {@link Fraction} als Factor
	 * 
	 * @param factor eine {@link Fraction}, mit der multipliziert werden soll
	 * @return {@code this * factor} als neuer {@link Euro}
	 * 
	 * @see java.math.BigInteger#multiply(java.math.BigInteger)
	 */
	public Euro multiply(Fraction factor) {
		return ofCents(Fraction.valueOf(getCents(), 1L).multiply(factor).longValue());
	}

	/**
	 * Die Methode formatiert den übergebenen {@code long} zu Euro als
	 * {@link String}
	 * 
	 * @param euros ein {@code long} mit einem ganzzahligen Euro-Betrag
	 * @return den String im {@link #EURO_CENT_FORMAT}
	 */
	public static String formatRealEuros(final long euros) {
		return String.format(EURO_ZERO_CENT_FORMAT, euros);
	}

	/**
	 * Die Methode ermittelt den größeren von diesem und dem anderen
	 * {@link Euro}-Betrag
	 * 
	 * @param other der zu vergleichende {@link Euro}-Betrag
	 * @return {@code this} oder {@code other}
	 * 
	 * @see java.math.BigInteger#max(java.math.BigInteger)
	 */
	public Euro max(Euro other) {
		return getCents() > other.getCents() ? this : other;
	}

	/**
	 * Returns the greater of two {@link Euro} values.
	 *
	 * @param a an argument.
	 * @param b another argument.
	 * @return the larger of {@code a} and {@code b}.
	 */
	public static Euro max(Euro a, Euro b) {
		if (a == null || b == null)
			return null;
		return (a.getCents() >= b.getCents()) ? a : b;
	}

	/**
	 * Returns the smaller of two {@link Euro} values.
	 *
	 * @param a an argument.
	 * @param b another argument.
	 * @return the smaller of {@code a} and {@code b}.
	 */
	public static Euro min(Euro a, Euro b) {
		if (a == null || b == null)
			return null;
		return (a.getCents() <= b.getCents()) ? a : b;
	}

	@Override
	public boolean greaterThan(Euro other) {
		return compareTo(other) > 0;
	}

	@Override
	public boolean lessThan(Euro other) {
		return compareTo(other) < 0;
	}

	@Override
	public boolean greaterThanOrEqualTo(Euro other) {
		return compareTo(other) >= 0;
	}

	@Override
	public boolean lessThanOrEqualTo(Euro other) {
		return compareTo(other) <= 0;
	}

	/**
	 * Vor der Umstellung auf die Klasse {@link Euro} deutete der Wert -1 an vielen
	 * Stellen an, dass der Wert noch nicht gesetzt war, was durch {@code null}
	 * ersetzt wurde. Negative Werte sind trotzdem nicht auszuschließen.
	 * 
	 * @param euro eine {@link Euro} Instanz
	 * @return {@code true} wenn die {@code null} oder negativ ist
	 */
	public static boolean undefined(Euro euro) {
		return euro == null || euro.getCents() < 0L;
	}
}