/**
 * TenorDatenContainer.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.data.tenordaten;

import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlRootElement;

import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.baumbach.StreitgenossenAngriffe;
import eu.gronos.kostenrechner.data.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenBerechnung;
import eu.gronos.kostenrechner.data.gebuehren.StreitwertReduktionen;
import eu.gronos.kostenrechner.data.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.interfaces.Tenorierend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;

/**
 * Containerklasse, um alle Entscheidungstexte eines {@link Tenorierend} wie
 * auch dessen strukturierten Berechnungsdaten aufnehmen und für Debug- und
 * Support-Zwecke als XML speichern zu können
 * 
 * @author Peter Schuster (setrok)
 * @date 02.07.2018
 *
 */
@XmlRootElement(name = "SchustersStrukturierteTenordaten", namespace = "https://www.kostentenor.de/SchustersStrukturierteTenordaten")
public class TenorDatenContainer extends VerfahrensDatenContainer {

	@XmlElement
	public String berechnungsTyp;

	@XmlElements({ @XmlElement(name = "swrn", type = StreitwertReduktionen.class),
			@XmlElement(name = "streit", type = StreitgenossenAngriffe.class) })
	public UnterContainerKlasse erweiterteKlasse;

	/**
	 * {@link EntscheidungsListenElemente} für die Hauptsache im
	 * {@link TenorDatenContainer}
	 */
	@XmlElement(name = "hauptsache")
	public EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsacheEntscheidung;

	/**
	 * {@link EntscheidungsListenElemente} für die Kostenentscheidung im
	 * {@link TenorDatenContainer}
	 */
	@XmlElement(name = "kosten")
	public EntscheidungsListenElemente<KostenTragungsVerhaeltnis> kostenEntscheidung;

	/**
	 * {@link EntscheidungsListenElemente} zur Vollstreckbarkeit im
	 * {@link TenorDatenContainer}
	 */
	@XmlElement(name = "vorlaeufigVollstreckbar")
	public EntscheidungsListenElemente<VollstreckungsVerhaeltnis> vollstreckbarkeitsEntscheidung;

	@XmlElement(name = "streitwertEntscheidung")
	public StreitwertEntscheidungsElemente streitwertEntscheidung;

	/**
	 * {@link EntscheidungsElemente} für sonstige Entscheidungsbestandteile im
	 * {@link TenorDatenContainer}
	 */
	@XmlElement(name = "sonstiges")
	public EntscheidungsElemente sonstigeEntscheidung;

	@XmlElement(name = "begruendung")
	public BegruendungsElemente begruendung;

	/**
	 * Standard-Konstruktor ohne Parameter, für {@link JAXB}
	 * 
	 */
	public TenorDatenContainer() {
		super();
		dateikopf = new Dateikopf();
		hauptsacheEntscheidung = new EntscheidungsListenElemente<HauptsacheVerhaeltnis>();
		kostenEntscheidung = new EntscheidungsListenElemente<KostenTragungsVerhaeltnis>();
		vollstreckbarkeitsEntscheidung = new EntscheidungsListenElemente<VollstreckungsVerhaeltnis>();
		streitwertEntscheidung = new StreitwertEntscheidungsElemente();
		sonstigeEntscheidung = new EntscheidungsElemente();
		begruendung = new BegruendungsElemente();
	}

	/**
	 * Konstruktor, der einen vollständigen {@link VerfahrensDatenContainer}
	 * erweitert.
	 * 
	 * @param container ein {@link VerfahrensDatenContainer}
	 */
	public TenorDatenContainer(VerfahrensDatenContainer container) {
		this();
		this.allgemein = container.allgemein;
		this.teilklageruecknahme = container.teilklageruecknahme;
		this.streitgenossen = container.streitgenossen;
		this.staffelung = container.staffelung;
		this.gebuehrenBerechnung = container.gebuehrenBerechnung;
	}

	/**
	 * Konstruktor, der die {@link UnterContainerKlasse}n eines
	 * {@link VerfahrensDatenContainer} übernimmt.
	 * 
	 * @param unters mehrere {@link UnterContainerKlasse}n
	 */
	public TenorDatenContainer(UnterContainerKlasse... unters) {
		this();
		for (UnterContainerKlasse unter : unters) {
			if (unter instanceof Allgemein)
				this.allgemein = (Allgemein) unter;
			else if (unter instanceof Teilklageruecknahme)
				this.teilklageruecknahme = (Teilklageruecknahme) unter;
			else if (unter instanceof Streitgenossen)
				this.streitgenossen = (Streitgenossen) unter;
			else if (unter instanceof ForderungsStaffelung)
				this.staffelung = (ForderungsStaffelung) unter;
			else if (unter instanceof GebuehrenBerechnung)
				this.gebuehrenBerechnung = (GebuehrenBerechnung) unter;
		}
	}
}
