/**
 * RowsRemoveable.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.interfaces;

/**
 * Ein Interface für TableModels, damit Zeilen zurückgegeben, entfernt und
 * hinzugefügt werden können. Eignet sich für TableModels, die intern mit
 * ArrayList&lt;T&gt; arbeiten und jedes &lt;T&gt; eine Zeile verkörpert.
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 31.07.2014
 * @param <T>
 *            der Typ, der eine Zeile verkörpert
 */
public interface RowHandler<T> {

	/**
	 * Gibt eine Zeile als Parameterobjekt des Typs &lt;T&gt; zurück
	 * 
	 * @param rowIndex
	 *            der Index der Zeile
	 * @return das Parameterobjekt für die Zeile als &lt;T&gt;
	 * @see java.util.ArrayList#get(int)
	 */
	T getRow(int rowIndex) throws IndexOutOfBoundsException;

	/**
	 * Entfernt eine Zeile aus dem Model, indem der entsprechende Eintrag aus
	 * der ArrayList&lt;T&gt; gelöscht wird. Eine implementierende Klasse muss
	 * {@link javax.swing.table.AbstractTableModel#fireTableRowsDeleted(int, int)
	 * fireTableRowsDeleted} durchführen.
	 * 
	 * @param rowIndex
	 *            der Index der Zeile
	 * @throws IndexOutOfBoundsException
	 *             if the index is out of range (index < 0 || index >= size()
	 * @throws IllegalArgumentException
	 *             wenn das TableModel bei Entfernen der Zeile einen ungültigen
	 *             Zustand annehmen würde
	 * @see java.util.ArrayList#remove(int)
	 */
	void removeRow(int rowIndex) throws IndexOutOfBoundsException,
			IllegalArgumentException;

	/**
	 * Die Methode dient dazu, ein neues Element an die Werte anzuhängen, was
	 * eine ganze Zeile der Tabelle hinzufügt. Eine implementierende Klasse muss
	 * {@link javax.swing.table.AbstractTableModel#fireTableRowsInserted(int, int)
	 * fireTableRowsInserted} durchführen.
	 * 
	 * @param rowContent
	 *            der anzuhängende BaumbachBeteiligte
	 * @throws IllegalArgumentException
	 *             wenn das Hinzufügen des Elements das TableModel in einen
	 *             ungültigen Zustand versetzen würde
	 * @see java.util.ArrayList#add(Object)
	 */
	void addRow(T rowContent) throws IllegalArgumentException;
}
