/**
 * TenorVorbereitend.java
 * eu.gronos.kostenrechner.interfaces (Kostenrechner)
 */
package eu.gronos.kostenrechner.interfaces;

import eu.gronos.kostenrechner.data.tenordaten.BegruendungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;

/**
 * Das Interface schreibt Klassen vor, Methoden zu Einzeltenoren zu
 * implementieren. So kann ein Adapter daraus den String zu der Methode String
 * erzeugeTenor() bauen.
 * 
 * Eine implementierende Klasse sollte so funktionieren, dass sie einen
 * {@link VerfahrensDatenContainer} bzw. einen {@link UnterContainerKlasse}
 * daraus als Parameter bekommt und einen {@link TenorDatenContainer} erzeugt.
 * 
 * @author Peter Schuster (setrok)
 * @date 18.10.2018
 *
 */
public interface TenorVorbereitend {
	/**
	 * Die Methode soll einen Hauptsachetenor erzeugen.
	 * 
	 * @return ein komplettes {@link EntscheidungsListenElemente}-Objekt für
	 *         {@link HauptsacheVerhaeltnis}, darin den Text des Hauptsachetenors
	 *         als String mit abschließendem Zeilenumbruch,
	 */
	abstract EntscheidungsListenElemente<HauptsacheVerhaeltnis> erzeugeHauptsacheEntscheidung();

	/**
	 * Die Methode soll einen Kostentenor erzeugen.
	 * 
	 * @return ein komplettes {@link EntscheidungsListenElemente}-Objekt für
	 *         {@link KostenTragungsVerhaeltnis} zurückgeben, also vor allem den
	 *         Text des Kostentenors als String mit abschließendem Zeilenumbruch.
	 */
	abstract EntscheidungsListenElemente<KostenTragungsVerhaeltnis> erzeugeKostenEntscheidung();

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit
	 * 
	 * @return ein komplettes {@link EntscheidungsListenElemente} für
	 *         {@link VollstreckungsVerhaeltnis}, vor allem einen String mit dem
	 *         Tenortext mit abschließendem Zeilenumbruch.
	 */
	abstract EntscheidungsListenElemente<VollstreckungsVerhaeltnis> erzeugeVollstreckbarkeitsEntscheidung();

	/**
	 * Die Methode soll eine Streitwertfestsetzung erzeugen.
	 * 
	 * @return ein komplettes {@link StreitwertEntscheidungsElemente}, darin v.a.
	 *         den Text der Streitwertfestsetzung als String mit abschließendem
	 *         Zeilenumbruch.
	 */
	abstract StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung();

	/**
	 * Die Methode soll den Container für eine sonstige Entscheidung (etwa die
	 * Gebührenaufstellung) erzeugen. Er darf auch nur leeren Text enthalten. Wenn
	 * er Text enthält, dann mit einem Zeilenumbruch beendeten.
	 * 
	 * @return einen Untercontainer vom Typ {@link EntscheidungsElemente} (Sonstige
	 *         hat keinen Untertyp)
	 */
	abstract EntscheidungsElemente erzeugeSonstigeEntscheidung();

	/**
	 * Die Methode soll einen {@link TenorDatenContainer} zusammenbauen. Der besteht
	 * aus {@link EntscheidungsListenElemente} für {@link HauptsacheVerhaeltnis},
	 * {@link KostenTragungsVerhaeltnis} und {@link VollstreckungsVerhaeltnis},
	 * {@link StreitwertEntscheidungsElemente}, {@link EntscheidungsElemente} für
	 * Sonstiges und {@link BegruendungsElemente}.
	 * 
	 * @return den erstellten Container
	 */
	abstract TenorDatenContainer erzeugeContainer();

	/**
	 * @return Die Methode soll eine Beschreibung für die Art des Tenors, also des
	 *         Typs der Berechnung zurückgeben, damit andere Klassen etwa das
	 *         Dokument danach benennen können.
	 */
	abstract String getBerechnungsTyp();

	/**
	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
	 * 
	 * @return den Text Gründe/Entscheidungsgründe als String
	 */
	abstract String getGruende();

}
