package eu.gronos.kostenrechner.logic.baumbach;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.*;

import eu.gronos.kostenrechner.data.baumbach.Angriff;
import eu.gronos.kostenrechner.data.baumbach.Angriff.AngriffArt;
import eu.gronos.kostenrechner.data.baumbach.GerichtsKostenBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.ImUebrigen;
import eu.gronos.kostenrechner.data.baumbach.MehrfachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.StreitgenossenAngriffe;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.TenorToken;
import eu.gronos.kostenrechner.util.VerlusteBank;
import eu.gronos.kostenrechner.util.baumbach.BeteiligteInAngreiferGegnerReihenfolge;

/**
 * Die Klasse verfasst die Tenortexte für {@link BaumbachFormel}, nunmehr ein
 * {@link TenorTexter} für eine Berechnung nach Baumbach mit {@link Angriff}en
 *
 * @author Peter Schuster (setrok)
 * @date 7 Jun 2019
 *
 */
public class BaumbachTexter extends TenorTexter<StreitgenossenAngriffe> {

//    private List<Angriff> angriffe;
	private final StreitgenossenAngriffe streit;
	private List<List<Angriff>> gruppen;
	private List<ImUebrigen> imUebrigen;
	private Map<Beteiligter, VerlusteBank> bereinigteQuoten;
//	private final boolean alsBruch;

	/**
	 * @param streit   ein {@link StreitgenossenAngriffe}
	 * @param alsBruch {@code boolean}, ob Kostenquoten als Bruch dargestellt
	 *                 werden. Sonst werden sie als Prozent geschrieben.
	 */
	public BaumbachTexter(StreitgenossenAngriffe streit, boolean alsBruch) {
		super(alsBruch);
		this.streit = streit;
//		this.alsBruch = alsBruch;
	}

	@Override
	public String texteHauptsache(EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache) {
		final StringBuilder tenor = new StringBuilder();

		for (List<Angriff> gruppe : gruppen) {
			ImUebrigen iue = this.imUebrigen.get(gruppen.indexOf(gruppe));
			/*
			 * Ermitteln, an welche Stelle des Tenors die Abweisung im Übrigen gehört.
			 * Jeweils an die Stelle, an der der letzte Beklagte (bzw. Widerbeklagte)
			 * erwähnt wird; das sagt mir die Gruppe der Angriffe
			 */
			for (Angriff angriff : gruppe) {
				/* die Prozessverhältnisse sind schon in der richtigen Reihenfolge */
				hauptSacheSchleife: for (HauptsacheVerhaeltnis verhaeltnis : hauptsache.prozessverhaeltnisse) {
					if (!equals(verhaeltnis, angriff))
						continue hauptSacheSchleife;
					final Beteiligter glaeubiger = verhaeltnis.getGlaeubiger();
					final Beteiligter schuldner = verhaeltnis.getSchuldner();
					if (iue.getAngriffArt() == AngriffArt.WIDERKLAGE) {
						tenor.append(TenorToken.AUF_DIE_WIDERKLAGE.toString() + TenorToken.LEER);
						tenor.append(super.verbessereGesamtschuldner(//
								super.getWirdWerden(schuldner.isPlural()) //
										+ TenorToken.LEER//
										+ schuldner.parteiBezeichner(NOMINATIV, schuldner.getLfdNr(),
												MehrfachBeteiligter.istEinzigerSeinerArt(schuldner,
														this.streit.beteiligte)) //
						));
					} else {
						tenor.append(// super.// verbessereGesamtschuldner(//
								satzBeginn(schuldner.parteiBezeichner(NOMINATIV, schuldner.getLfdNr(),
										MehrfachBeteiligter.istEinzigerSeinerArt(schuldner, this.streit.beteiligte))) //
										+ TenorToken.LEER//
										+ super.getWirdWerden(schuldner.isPlural()) //
//								)
						);
					}
					/*
					 * Nach darüber hinaus gucken, um es an der richtigen Stelle in den Tenor zu
					 * stecken.
					 */
					if (verhaeltnis.isDarueberHinaus()) {
						tenor.append(TenorToken.LEER);
						tenor.append(TenorToken.DARUEBER_HINAUS);
					}
					tenor.append(TenorToken.LEER.toString() + TenorToken.VERURTEILT + TenorToken.KOMMA_LEER //
					/*
					 * Angriff statt HauptsacheVerhaeltnis weil genauer (mit Cent)
					 */
							+ angriff.getErfolg().toString() //
							+ TenorToken.LEER + TenorToken.EURO//
							+ TenorToken.LEER + TenorToken.AN + TenorToken.LEER //
							+ glaeubiger.parteiBezeichner(AKKUSATIV, glaeubiger.getLfdNr(),
									MehrfachBeteiligter.istEinzigerSeinerArt(glaeubiger, this.streit.beteiligte))//
							+ TenorToken.LEER + TenorToken.ZU_ZAHLEN + TenorToken.PUNKT_ABSATZ);

				}
			}
			final String abweisungsString = iue.toAbweisungsString();//
			tenor.append(abweisungsString);
		}
		return verbessereGesamtschuldner(tenor.toString());
	}

	/**
	 * Die Methode gibt der Reihe nach alle {@link VerlusteBank}en der {@link Map}
	 * aus {@link Beteiligter}, {@link VerlusteBank} aus.
	 * 
	 * Die Methode gibt alle Verluste einer {@link VerlusteBank} aus
	 * 
	 * @param titel    Teil der Überschrift, die sagt, um wessen
	 *                 Kostenerstattungsansprüche es sich handelt
	 * 
	 * @param angriffe eine {@link List} aus {@link Angriff}en
	 * 
	 * @param quoten   eine {@link Map} aus {@link Beteiligter},
	 *                 {@link VerlusteBank}
	 */
	@Override
	public String texteKostenentscheidung(EntscheidungsListenElemente<KostenTragungsVerhaeltnis> elemente) {
		final StringBuilder tenor = new StringBuilder();
		final StringBuilder satz = new StringBuilder();
		final List<Beteiligter> ueberspringen = new ArrayList<>();
		final List<List<Beteiligter>> gleichlautendezeilen = findeGleichlautendeQuoten(bereinigteQuoten);
		// vollständige Kostentragung?
		boolean alleKostenGleich = alleKostenGleich(gleichlautendezeilen);
		Beteiligter glaeubiger = null;
		for (KostenTragungsVerhaeltnis verhaeltnis : elemente.prozessverhaeltnisse) {
			/*
			 * Gläubigerwechsel abfragen. Bei einem Wechsel muss ich den vorigen Satz
			 * beenden und einen neuen Satzanfang einfügen.
			 */
			final boolean gewechselt = !verhaeltnis.getGlaeubiger().equals(glaeubiger);
			glaeubiger = verhaeltnis.getGlaeubiger();
			if (ueberspringen.contains(glaeubiger)) {
				continue;
			}
			if (gewechselt) {
				if (!(satz.length() < 1)) {
					/*
					 * Wenn Gläubiger wechselt: Gesamt-Kostentenor ergänzen und neuen Satz beginnen.
					 */
					tenor.append(super.hatStattHaben(undStattKomma(letztesKommaWeg(satz.toString()))));
					tenor.append(TenorToken.LEER.toString() + TenorToken.ZU_TRAGEN + TenorToken.PUNKT_ABSATZ);
					satz.delete(0, satz.length());
				}
				List<Beteiligter> mehrere = null;
				for (List<Beteiligter> multi : gleichlautendezeilen) {
					if (multi.contains(glaeubiger)) {
						mehrere = multi;
					}
				}
				if (alleKostenGleich) {
					/*
					 * gleichlautendeQuoten beerücksichtigen: Wenn alle Kosten gleich verteilt sind,
					 * werden "Die Kosten des Rechtsstreits" verteilt ...
					 */
					satz.append(TenorToken.DIE_KOSTEN_DES_RECHTSSTREITS);
					satz.append(TenorToken.LEER);
					/*
					 * ... und alle anderen Zeilen müssen übersprungen werden, deshalb in
					 * ueberspringen aufnehmen.
					 */
					fuelleUeberspringen(this.streit.beteiligte, ueberspringen, glaeubiger);
				} else if (mehrere != null) {
					/*
					 * wenn etwas genauso verteilt ist wie die Gerichtskosten: anhängen und den
					 * Gläubiger der gleich verteilten Kosten merken.
					 */
					if (glaeubiger instanceof GerichtsKostenBeteiligter) {
						satz.append(TenorToken.DIE_GERICHTSKOSTEN);// 2
						satz.append(TenorToken.LEER.toString() + TenorToken.UND_DIE + TenorToken.LEER
								+ TenorToken.AUSSERGERICHTLICHEN_KOSTEN + TenorToken.LEER);
						final List<Beteiligter> mehrerere = new ArrayList<>();
						fuelleUeberspringen(mehrere, mehrerere, glaeubiger);
						satz.append(
								MehrfachBeteiligter.parteiBezeichnerListe(this.streit.beteiligte, mehrerere, GENITIV));
					} else {
						satz.append(TenorToken.DIE.toString() + TenorToken.LEER + TenorToken.AUSSERGERICHTLICHEN_KOSTEN
								+ TenorToken.LEER);
						satz.append(MehrfachBeteiligter.parteiBezeichnerListe(this.streit.beteiligte, mehrere, GENITIV)//
						);
					}
					fuelleUeberspringen(mehrere, ueberspringen, glaeubiger);
				} else {
					if (glaeubiger instanceof GerichtsKostenBeteiligter) {
						satz.append(TenorToken.DIE_GERICHTSKOSTEN);
					} else {
						satz.append(TenorToken.DIE + TenorToken.LEER.toString() + TenorToken.AUSSERGERICHTLICHEN_KOSTEN
								+ TenorToken.LEER);
						satz.append(glaeubiger.parteiBezeichner(GENITIV, glaeubiger.getLfdNr(),
								MehrfachBeteiligter.istEinzigerSeinerArt(glaeubiger, this.streit.beteiligte)));
					}
				}
				satz.append(TenorToken.LEER.toString() + TenorToken.HABEN + TenorToken.LEER);
			}
			satz.append(verhaeltnis.getSchuldner().parteiBezeichner(NOMINATIV, verhaeltnis.getSchuldner().getLfdNr(),
					MehrfachBeteiligter.istEinzigerSeinerArt(verhaeltnis.getSchuldner(), this.streit.beteiligte)));

			if (verhaeltnis.isDarueberHinaus())
				satz.append(TenorToken.LEER.toString() + TenorToken.DARUEBER_HINAUS + TenorToken.LEER);
			final Fraction kostenBruchteil = verhaeltnis.getKostenBruchteil();// double
			final Fraction bruch = bruchteilAus(glaeubiger, verhaeltnis.getSchuldner());
			if (kostenBruchteil.lessThan(Fraction.ONE)) {// < 1.0
				/*
				 * Wenn eine Partei 100% trägt, nur den Bezeichner ausgeben, sonst auch die
				 * Prozente.
				 */
				satz.append(TenorToken.LEER.toString() + TenorToken.ZU + TenorToken.LEER);
				if (super.alsBruch)
					satz.append(bruch);
				else
					satz.append(bruch.toPercentString());// (alsProzent(kostenBruchteil.doubleValue()));
			}
			satz.append(TenorToken.KOMMA_LEER);
		}
		/*
		 * Zum Schluss nochmal: Gesamt-Kostentenor ergänzen.
		 */
		if (!(satz.length() < 1)) {
			tenor.append(super.hatStattHaben(undStattKomma(letztesKommaWeg(satz.toString()))));
			tenor.append(TenorToken.LEER.toString() + TenorToken.ZU_TRAGEN);
			satz.delete(0, satz.length());
			tenor.append(TenorToken.PUNKT_ABSATZ);
		}
		if (!alleKostenGleich) {
			tenor.append(TenorToken.KOSTENTRAGUNG_SELBST);
			tenor.append(TenorToken.PUNKT_ABSATZ);
		}
		return tenor.toString().replace("  ", TenorToken.LEER);
	}

	/**
	 * Die Methode sucht für einen {@link Beteiligter}n den entsprechenden Bruchteil
	 * aus der {@link Map} der {@link VerlusteBank}en.
	 * 
	 * @param glaeubiger der Inhaber der {@link VerlusteBank}, also {@code key} der
	 *                   {@link Map}
	 * @param schuldner  der Schuldner als {@link Beteiligter} oder
	 *                   {@link MehrfachBeteiligter}, aus dem eine {@link List}
	 *                   gebaut wird.
	 * @return eine {@link Fraction}
	 */
	private Fraction bruchteilAus(Beteiligter glaeubiger, Beteiligter schuldner) {
		final VerlusteBank bank = bereinigteQuoten.get(glaeubiger);
		if (bank == null || bank.keySet().isEmpty())
			throw new NullPointerException("Keine Bank für " + glaeubiger);
		List<Beteiligter> schuldner2;
		if (schuldner instanceof MehrfachBeteiligter)
			schuldner2 = ((MehrfachBeteiligter) schuldner).getBeteiligte();
		else {
			schuldner2 = new ArrayList<>();
			schuldner2.add(schuldner);
		}
		if (bank.get(schuldner2) != null)
			return bank.getFractionFor(schuldner2);
		else
			throw new NullPointerException("Kein Konto für " + schuldner);
	}

	@Override
	public String texteVollstreckbarkeit(EntscheidungsListenElemente<VollstreckungsVerhaeltnis> elemente,
			VollstreckbarkeitsListe liste) {
		String schuldner = TenorToken.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER.toString();
		String glaeubiger = TenorToken.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER.toString();

		if (liste.sindAlle708iVm713()) {
			return TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR.toString() + TenorToken.PUNKT_ABSATZ;
		} else if (liste.sindAlle709()) {
			return TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709.toString() + TenorToken.PUNKT_ABSATZ;
		} else if (liste.sindAlle708iVm711()) {
			if (1 == zaehle708iVm711(elemente.prozessverhaeltnisse)) {
				/*
				 * Wenn es nur ein solches Verhältnis gibt, dann die Parteien konkret benennen.
				 */
				Beteiligter schuldnerBeteiligter = elemente.prozessverhaeltnisse.get(0).getSchuldner();
				Beteiligter glaeubigerBeteiligter = elemente.prozessverhaeltnisse.get(0).getGlaeubiger();
				schuldner = schuldnerBeteiligter.parteiBezeichner(DATIV, schuldnerBeteiligter.getLfdNr(),
						MehrfachBeteiligter.istEinzigerSeinerArt(schuldnerBeteiligter, this.streit.beteiligte));
				glaeubiger = glaeubigerBeteiligter.parteiBezeichner(NOMINATIV, glaeubigerBeteiligter.getLfdNr(),
						MehrfachBeteiligter.istEinzigerSeinerArt(glaeubigerBeteiligter, this.streit.beteiligte));
			}
			/* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
			return TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR.toString() + TenorToken.PUNKT_ABSATZ
					+ satzBeginn(schuldner) + TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_1 + glaeubiger
					+ TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_2 + TenorToken.PUNKT_ABSATZ;
		}
		StringBuffer tenor = new StringBuffer(TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
		tenor.append(TenorToken.PUNKT_ABSATZ);
		/*
		 * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
		 * ausgeben. Dafür zwei StringBuffer füllen.
		 */
		StringBuffer tenor709 = new StringBuffer();
		StringBuffer tenor708 = new StringBuffer();
		final int size709 = zaehle709(elemente.prozessverhaeltnisse);
		final int size708u711 = zaehle708iVm711(elemente.prozessverhaeltnisse);
		if (size709 > 1) {
			tenor709 = new StringBuffer(TenorToken.HINSICHTLICH_DES_JEWEILIGEN_VOLLSTRECKUNGSVERHAELTNISSES);
			tenor709.append(TenorToken.LEER);
		}
		if (size708u711 > 1) {
			tenor708 = new StringBuffer(TenorToken.HINSICHTLICH_DES_JEWEILIGEN_VOLLSTRECKUNGSVERHAELTNISSES);
			tenor708.append(TenorToken.LEER);
		}

		/*
		 * Alle VollstreckungsVerhaeltnisse durchgehen. Jeweils nach
		 * Vollstreckbarkeitsart trennen. Innerhalb der Vollstreckbarkeitsart gucken, ob
		 * es das einzige der Art ist.
		 */
		for (VollstreckungsVerhaeltnis verhaeltnis : elemente.prozessverhaeltnisse) {
			Beteiligter schuldnerBeteiligter = verhaeltnis.getSchuldner();
			Beteiligter glaeubigerBeteiligter = verhaeltnis.getGlaeubiger();
			if (verhaeltnis.getVollstreckbarkeit().isB709s1()) {
				/* Gegen Sicherheitsleistung. */
				if (size709 == 1) {
					schuldner = schuldnerBeteiligter.parteiBezeichner(DATIV, schuldnerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(schuldnerBeteiligter, this.streit.beteiligte));
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(AKKUSATIV, glaeubigerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(glaeubigerBeteiligter, this.streit.beteiligte));
					tenor709 = new StringBuffer(TenorToken.FUER.toString() + TenorToken.LEER + glaeubiger
							+ TenorToken.LEER + TenorToken.GEGENUEBER + TenorToken.LEER + schuldner);
				} else {
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(GENITIV, glaeubigerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(glaeubigerBeteiligter, this.streit.beteiligte));
					schuldner = schuldnerBeteiligter.parteiBezeichner(AKKUSATIV, schuldnerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(schuldnerBeteiligter, this.streit.beteiligte));
					tenor709.append(glaeubiger + TenorToken.LEER + TenorToken.GEGEN + TenorToken.LEER + schuldner
							+ TenorToken.KOMMA_LEER);
				}
			} else if (verhaeltnis.getVollstreckbarkeit().isB711()) {
				/* Abwendungsbefugnis */
				if (size708u711 == 1) {
					schuldner = schuldnerBeteiligter.parteiBezeichner(DATIV, schuldnerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(schuldnerBeteiligter, this.streit.beteiligte));
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(NOMINATIV, glaeubigerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(glaeubigerBeteiligter, this.streit.beteiligte));
					tenor708 = new StringBuffer(
							satzBeginn(schuldner) + TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_1//
									+ glaeubiger + TenorToken.LEER//
									+ TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_2 + TenorToken.PUNKT_ABSATZ);
				} else {
					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(GENITIV, glaeubigerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(glaeubigerBeteiligter, this.streit.beteiligte));
					schuldner = schuldnerBeteiligter.parteiBezeichner(AKKUSATIV, schuldnerBeteiligter.getLfdNr(),
							MehrfachBeteiligter.istEinzigerSeinerArt(schuldnerBeteiligter, this.streit.beteiligte));
					tenor708.append(glaeubiger + TenorToken.LEER + TenorToken.GEGEN + TenorToken.LEER + schuldner
							+ TenorToken.KOMMA_LEER);
				}
			}
		}

		/*
		 * Erst die §709-Vollstreckbarkeitsverhältnisse anhängen – hier muss immer der
		 * Abschluss dran, aber ggfs. noch Kommata weg!
		 */
		if (size709 > 0) {
			if (1 == size709) {
				tenor.append(tenor709);
			} else {
				tenor.append(TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(tenor709.toString())));
			}
			tenor.append(TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
			tenor.append(TenorToken.PUNKT_ABSATZ);
		}

		/* Dann die §708-Vollstreckbarkeitsverhältnisse */
		if (size708u711 == 1) {
			/* Bei einem ist der Abschluss schon dran */
			tenor.append(tenor708);
		} else if (size708u711 > 1) {
			/* wenn es mehrere gibt, muss noch ein Abschluss dran. */
			tenor.append(TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(tenor708.toString())) + TenorToken.LEER
					+ TenorToken.WIRD + TenorToken.LEER + TenorToken.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER
					+ TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST_1
					+ TenorToken.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER
					+ TenorToken.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST_2 + TenorToken.PUNKT_ABSATZ);
		}
		return tenor.toString();
	}

	/**
	 * Die Methode erzeugt eine Streitwertfestsetzung, einen formatierten String
	 * nach dem Muster "Der Streitwert wird auf %,.2f EUR festgesetzt.%n", bei einer
	 * Widerklage eine Zusammensetzung aus Gesamtstreitwert, Klagewert und
	 * Widerklagewert.
	 * 
	 * @param elemente {@link StreitwertEntscheidungsElemente}
	 * @return ein {@link String} für {@link StreitwertEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteStreitwert(eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente)
	 */
	@Override
	public String texteStreitwert(StreitwertEntscheidungsElemente elemente) {
		StringBuilder builder = new StringBuilder(TenorToken.STREITWERT_WIRD_AUF.toString() + TenorToken.LEER);
		if (elemente.streitwerte.size() > 1) {
			// Bei einer Widerklage müssen beide Streitwerte ausgegeben werden
			builder.append(Euro.formatRealEuros(elemente.streitwerte.get(0).longValue()));
			builder.append(
					TenorToken.LEER.toString() + TenorToken.EURO + TenorToken.KOMMA_LEER + TenorToken.DAVON_FD_KLAGE);
			builder.append(Euro.formatRealEuros(elemente.streitwerte.get(1).longValue()));
			builder.append(TenorToken.LEER.toString() + TenorToken.EURO + TenorToken.LEER + TenorToken.UND_FD_WIDERKLAGE
					+ TenorToken.LEER);
			builder.append(Euro.formatRealEuros(elemente.streitwerte.get(2).longValue()));
		} else {
			// Ohne Widerklage muss nur ein Streitwert ausgegeben werden
			builder.append(Euro.formatRealEuros(elemente.streitwerte.get(0).longValue()));
		}
		builder.append(TenorToken.LEER.toString() + TenorToken.EURO + TenorToken.LEER + TenorToken.FESTGESETZT + "\n");
		return builder.toString();
	}

	/**
	 * @param bereinigteQuoten d. {@link #bereinigteQuoten}, d. gesetzt werden soll
	 *                         als {@link Map<Beteiligter,VerlusteBank>}.
	 */
	public void setBereinigteQuoten(Map<Beteiligter, VerlusteBank> bereinigteQuoten) {
		this.bereinigteQuoten = bereinigteQuoten;
	}

	/*
	 * @return gibt {@link #gruppen} als {@link List<List<Angriff>>} zurück. public
	 * List<List<Angriff>> getGruppen() { return this.gruppen; }
	 */

	/**
	 * @param gruppen d. {@link #gruppen}, d. gesetzt werden soll als
	 *                {@link List<List<Angriff>>}.
	 */
	public void setGruppen(List<List<Angriff>> gruppen) {
		this.gruppen = gruppen;
	}

	/**
	 * @return gibt {@link #imUebrigen} als {@link List<ImUebrigen>} zurück. public
	 *         List<ImUebrigen> getImUebrigen() { return imUebrigen; }
	 */

	/**
	 * @param imUebrigen d. {@link #imUebrigen}, d. gesetzt werden soll als
	 *                   {@link List<ImUebrigen>}.
	 */
	public void setImUebrigen(List<ImUebrigen> imUebrigen) {
		this.imUebrigen = imUebrigen;
	}

	/**
	 * Wenn alle Kosten gleich verteilt sind, werden "Die Kosten des Rechtsstreits"
	 * verteilt und alle anderen Zeilen müssen übersprungen werden, deshalb in
	 * gleiche aufnehmen.
	 * 
	 * @param gleichlautendezeilen eine {@link List} aus {@link Beteiligter}
	 * @return {@code true}, wenn die geplättete {@link List} genauso viele
	 *         {@link Beteiligter} enthält wie {@link #bereinigteQuoten}
	 */
	private boolean alleKostenGleich(final List<List<Beteiligter>> gleichlautendezeilen) {
		List<Beteiligter> alleGlaeubiger = new ArrayList<>();
		for (List<Beteiligter> glaeubiger : gleichlautendezeilen) {
			for (Beteiligter einzelne : glaeubiger)
				if (!alleGlaeubiger.contains(einzelne))
					alleGlaeubiger.add(einzelne);
		}
		return alleGlaeubiger.size() == bereinigteQuoten.keySet().size();// - 1;
	}

	/**
	 * Die Methode prüft ein {@link HauptsacheVerhaeltnis} und einen {@link Angriff}
	 * auf Übereinstimmung
	 * 
	 * @param verhaeltnis ein {@link HauptsacheVerhaeltnis}
	 * @param angriff     einen {@link Angriff}
	 * @return <code>true</code>, wenn gleicher Angreifer/Gläubiger, gleicher
	 *         Gegner/Schuldner und gleiche Höhe von Erfolg/Verurteilung.
	 */
	private boolean equals(HauptsacheVerhaeltnis verhaeltnis, Angriff angriff) {
		final boolean gleicherAngreifer = equals(verhaeltnis.getGlaeubiger(), angriff.getAngreifer());
		final boolean gleicherGegner = equals(verhaeltnis.getSchuldner(), angriff.getGegner());
		final boolean gleicheHoehe = // (verhaeltnis.getVerurteilung() == angriff.getErfolg().longValue());
				verhaeltnis.getVerurteilung().equals(angriff.getErfolg());
		return gleicherAngreifer && gleicherGegner && gleicheHoehe;
	}

	/**
	 * Die Methode prüft einen {@link Beteiligter}n (ggfs.
	 * {@link MehrfachBeteiligter}n) auf Übereinstimmung mit einer {@link List} von
	 * {@link Beteiligter}n
	 * 
	 * @param beteiligter {@link Beteiligter}n (ggfs. {@link MehrfachBeteiligter}n)
	 * @param liste       {@link List} von {@link Beteiligter}n, gegen die
	 *                    verglichen werden soll
	 * @return <code>true</code>, wenn a) bei einem {@link MehrfachBeteiligter}
	 *         {@link MehrfachBeteiligter#beteiligte} dieselben {@link Beteiligter}
	 *         enthält wie die {@link List} {@code liste} oder b) wenn die
	 *         {@link List} {@code liste} nur {@link Beteiligter}
	 *         {@code beteiligter} enthält
	 */
	private boolean equals(final Beteiligter beteiligter, final List<Beteiligter> liste) {
		if (beteiligter instanceof MehrfachBeteiligter) {
			return ((MehrfachBeteiligter) beteiligter).listEquals(liste);// .beteiligte.equals(liste);
		}
		return liste.size() == 1 && liste.contains(beteiligter);
	}

	/**
	 * Die Methode prüft, ob die Gerichtskosten wie die Klägerkosten verteilt sind.
	 * 
	 * @param quoten eine {@link Map} aus {@link Beteiligter}, {@link VerlusteBank}
	 *
	 * @return eine {@link List} aus Unter- {@link List}en von {@link Beteiligter}en
	 */
	private List<List<Beteiligter>> findeGleichlautendeQuoten(Map<Beteiligter, VerlusteBank> quoten) {
		final List<List<Beteiligter>> listen = new ArrayList<>();
		if (quoten == null || quoten.isEmpty())
			return listen;
		//
		erste: for (Beteiligter a : quoten.keySet()) {
			if (a == null || quoten.get(a) == null)
				continue erste;
			zweite: for (Beteiligter b : quoten.keySet()) {
				final List<Beteiligter> beteiligte = new ArrayList<>();
				beteiligte.add(a);
				if (b == null || quoten.get(b) == null || a.equals(b))
					continue zweite;
				if (quoten.get(a).equals(quoten.get(b))) {
					beteiligte.add(b);
				}
				if (beteiligte.size() > 1) {
					Collections.sort(beteiligte, new BeteiligteInAngreiferGegnerReihenfolge());
					if (!listen.contains(beteiligte))
						listen.add(beteiligte);
				}
			}
		}
		return listen;
	}

	/**
	 * Die Methode steckt alle aus der {@link List} aus {@link Beteiligter}
	 * {@code source} in die {@link List} aus {@link Beteiligter} {@code target},
	 * mit Ausnahme des {@link Beteiligter}n {@code exception}. Sonst kommt es zu
	 * seltsamen Wendungen im Tenor
	 * 
	 * @param source    aus dieser {@link List} aus {@link Beteiligter}
	 * @param target    in diese {@link List} aus {@link Beteiligter}
	 * @param exception dieser {@link Beteiligter} kommt nicht hinein
	 */
	private void fuelleUeberspringen(List<Beteiligter> source, final List<Beteiligter> target, Beteiligter exception) {
		for (Beteiligter einzelner : source) {
			if (!exception.equals(einzelner))
				target.add(einzelner);
		}
	}

	/**
	 * Die Methode dient dazu herauszufinden, ob es nur ein Verhältnis der Art
	 * {@link Vollstreckbarkeit#isB711()} gibt.
	 * 
	 * Wenn es nur ein solches Verhältnis gibt, dann die aufrufende Methode die
	 * Parteien konkret benennen. Sonst bleibt's beim "jeweiligen"
	 * Gläubiger/Schuldner
	 * 
	 * @param verhaeltnisse eine {@link List} von
	 *                      {@link VollstreckungsVerhaeltnis}sen
	 * @return die Zahl der Verhältnisse dieser Art als {@code int}
	 */
	private int zaehle708iVm711(List<VollstreckungsVerhaeltnis> verhaeltnisse) {
		int size = 0;
		for (VollstreckungsVerhaeltnis verhaeltnis : verhaeltnisse) {
			if (verhaeltnis.getVollstreckbarkeit().isB711())
				size++;
		}
		return size;
	}

	/**
	 * Die Methode dient dazu herauszufinden, ob es nur ein Verhältnis der Art
	 * {@link Vollstreckbarkeit#isB709s1()} gibt.
	 * 
	 * Wenn es nur ein solches Verhältnis gibt, dann die aufrufende Methode die
	 * Parteien konkret benennen. Sonst bleibt's beim "jeweiligen"
	 * Gläubiger/Schuldner bzw. "Hinsichtlich des jeweiligen
	 * Vollstreckungsverhältnis"
	 * 
	 * @param verhaeltnisse eine {@link List} von
	 *                      {@link VollstreckungsVerhaeltnis}sen
	 * @return die Zahl der Verhältnisse dieser Art als {@code int}
	 */
	private int zaehle709(List<VollstreckungsVerhaeltnis> verhaeltnisse) {
		int size = 0;
		for (VollstreckungsVerhaeltnis verhaeltnis : verhaeltnisse) {
			if (verhaeltnis.getVollstreckbarkeit().isB709s1())
				size++;
		}
		return size;
	}

	//
//	private BaumbachImUebrigen imUebrigen;
//	private final BaumbachBeteiligtenListe baumbachBeteiligtenListe;
//	private double[][] quoten;
//
//	/**
//	 * @param baumbachBeteiligtenListe eine {@link BaumbachBeteiligtenListe}
//	 */
//	public BaumbachTexter(BaumbachBeteiligtenListe baumbachBeteiligtenListe) {
//		this.baumbachBeteiligtenListe = baumbachBeteiligtenListe;
//	}
//
//	/**
//	 * Die Methode baut den Text für die {@link EntscheidungsListenElemente} der
//	 * {@link HauptsacheVerhaeltnis}se
//	 * 
//	 * @param hauptsache {@link EntscheidungsListenElemente} der
//	 *                   {@link HauptsacheVerhaeltnis}se
//	 * 
//	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteHauptsache(eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer.HauptsacheEntscheidungsElemente)
//	 */
//	@Override
//	public String texteHauptsache(EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache) {
//		StringBuffer tenor = new StringBuffer();
//		/*
//		 * Ermitteln, an welche Stelle des Tenors die Abweisung im Übrigen gehört.
//		 * Jeweils an die Stelle, an der der letzte Beklagte (bzw. Widerklage) erwähnt
//		 * wird; und wo das ist, das sagt mir findeGesamtschuldnerIndices
//		 */
//		boolean letzterBeklagter = false;
//		/* die Prozessverhältnisse sind schon in der richtigen Reihenfolge */
//		for (HauptsacheVerhaeltnis verhaeltnis : hauptsache.prozessverhaeltnisse) {
//			/*
//			 * Nach darüber hinaus gucken, um es an der richtigen Stelle in den Tenor zu
//			 * stecken.
//			 */
//			String darueberHinaus;
//			if (verhaeltnis.isDarueberHinaus()) {
//				darueberHinaus = TenorTexter.DARUEBER_HINAUS;
//			} else {
//				darueberHinaus = "";
//			}
//			Beteiligter glaeubiger = verhaeltnis.getGlaeubiger();
//			Beteiligter schuldner = verhaeltnis.getSchuldner();
//			boolean einzigerSeinerArt = isEinzigerSeinerArt(schuldner);
//			if (schuldner.getTyp() == BEKLAGTE) {
//				/* erst die Verurteilungen nach der Klage */
//				tenor.append(
//						super.verbessereGesamtschuldner(String.format("%s %s verurteilt, %,.2f EUR an %s zu zahlen.%n",
//								satzBeginn(
//										schuldner.parteiBezeichner(NOMINATIV, schuldner.getLfdNr(), einzigerSeinerArt)),
//								super.getWirdWerden(schuldner.isPlural()) + darueberHinaus,
//								((BaumbachBeteiligter) schuldner).getUnterliegen(),
//								glaeubiger.parteiBezeichner(AKKUSATIV, glaeubiger.getLfdNr(), true))));
//			} else {
//				/* dann die Verurteilungen nach der Widerklage */
//				if (!letzterBeklagter) {
//					/*
//					 * wenn letzterBeklagter noch false ist, dann ausgeben, ob die Klage
//					 * ganz/imÜbrigen abgewiesen ist
//					 */
//					tenor.append(imUebrigen.toKlageAbweisungsString());
//					letzterBeklagter = true;
//				}
//				boolean einzigerGlSeinerArt = isEinzigerSeinerArt(glaeubiger);
//				tenor.append(super.verbessereGesamtschuldner(
//						String.format("Auf die Widerklage %s %s verurteilt, %,.2f EUR an %s zu zahlen.%n",
//								super.getWirdWerden(schuldner.isPlural()) + darueberHinaus,
//								schuldner.parteiBezeichner(NOMINATIV, schuldner.getLfdNr(), einzigerSeinerArt),
//								((BaumbachBeteiligter) schuldner).getUnterliegen(),
//								glaeubiger.parteiBezeichner(AKKUSATIV, glaeubiger.getLfdNr(), einzigerGlSeinerArt))));
//			}
//		}
//		/*
//		 * Zusatz für Voll- oder Teilabweisung der Klage, falls letzterBeklagter immer
//		 * noch false
//		 */
//		if (hauptsache.prozessverhaeltnisse.isEmpty() || !letzterBeklagter) {
//			tenor.append(imUebrigen.toKlageAbweisungsString());
//		}
//		/* Zusatz für Voll- oder Teilabweisung der Widerklage */
//		if (baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
//			tenor.append(imUebrigen.toWiderklageAbweisungsString());
//		}
//		return tenor.toString();
//	}
//
//	/**
//	 * Die Methode erzeugt den Kostentenor aus den übergebenen
//	 * {@link KostenGrundEntscheidungsElemente}n. Vorher muss
//	 * {@link #setQuoten(double[][])} gesetzt sein; das Array wird noch gebraucht,
//	 * um {@link #findeGleichlautendeQuoten(double[][])} nutzen zu können.
//	 * 
//	 * @param elemente die {@link EntscheidungsListenElemente} für
//	 *                 {@link KostenTragungsVerhaeltnis}
//	 * @return den Text des Kostentenors als {@link String} mit abschließendem
//	 *         Zeilenumbruch
//	 * 
//	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteKostenentscheidung(eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer.KostenGrundEntscheidungsElemente)
//	 */
//	@Override
//	public String texteKostenentscheidung(EntscheidungsListenElemente<KostenTragungsVerhaeltnis> elemente) {
//		if (quoten == null)
//			throw new NullPointerException("Das Array quoten[][] darf nicht null sein!");
//		StringBuffer tenor = new StringBuffer();
//		StringBuffer satz = new StringBuffer();
//		ArrayList<StringBuffer> kostenTenorSaetze = new ArrayList<StringBuffer>();
//		/*
//		 * Dann gucken, ob irgendwas genauso verteilt wird wie die Gerichtskosten.
//		 */
//		ArrayList<Integer> gleichlautendezeilen = findeGleichlautendeQuoten(quoten);
//		boolean alleKostenGleich = (gleichlautendezeilen.size() == quoten.length - 1);
//		BaumbachBeteiligtenListe gleiche = new BaumbachBeteiligtenListe();
//		/*
//		 * Die KostenGrundEntscheidungsElemente sind nach Gläubiger sortiert, deshalb
//		 * muss man den Wechsel merken
//		 */
//		Beteiligter glaeubiger = null;
//
//		for (int index = 0; index < elemente.prozessverhaeltnisse.size(); index++) {
//			KostenTragungsVerhaeltnis verhaeltnis = elemente.prozessverhaeltnisse.get(index);
//			/*
//			 * Gläubigerwechsel abfragen. Bei einem Wechsel muss ich den vorigen Satz
//			 * beenden und einen neuen Satzanfang einfügen.
//			 */
//			boolean gewechselt = !verhaeltnis.getGlaeubiger().equals(glaeubiger);
//			glaeubiger = verhaeltnis.getGlaeubiger();
//			/*
//			 * Wenn der Gläubiger schon mit den Gerichtskosten ausgegeben wurde, dann
//			 * überspringe ihn, zu den Kosten, die wie die Gerichtskosten verteilt werden,
//			 * keine eigene Zeile ausgeben.
//			 */
//			if (gleiche.contains(glaeubiger))
//				continue;
//			/*
//			 * Gerichtskosten ausgeben; dabei prüfen, ob die Kosten eines Gläubigers (z.B.
//			 * des Klägers) genauso verteilt sind
//			 */
//			if (verhaeltnis.getGlaeubiger() instanceof GerichtsKostenBeteiligter) {
//				if (gewechselt) {
//					if (alleKostenGleich) {
//						/*
//						 * Wenn alle Kosten gleich verteilt sind, werden "Die Kosten des Rechtsstreits"
//						 * verteilt ...
//						 */
//						satz.append(TenorTexter.DIE_KOSTEN_DES_RECHTSSTREITS_HABEN);
//
//						/*
//						 * ... und alle anderen Zeilen müssen übersprungen werden, deshalb in gleiche
//						 * aufnehmen.
//						 */
//						for (Integer i : gleichlautendezeilen) {
//							BaumbachBeteiligter gleicher = baumbachBeteiligtenListe.get(i.intValue() - 1);
//							gleiche.add(gleicher);
//						}
//					} else if (gleichlautendezeilen == null || gleichlautendezeilen.isEmpty()) {
//						/* Wenn nur die Gerichtskosten so verteilt sind: */
//						satz.append(TenorTexter.DIE_GERICHTSKOSTEN_HABEN);
//					} else {
//						/*
//						 * wenn etwas genauso verteilt ist wie die Gerichtskosten: anhängen und den
//						 * Gläubiger der gleich verteilten Kosten merken.
//						 */
//						satz.append(TenorTexter.DIE_GERICHTSKOSTEN);
//						for (Integer i : gleichlautendezeilen) {
//							BaumbachBeteiligter gleicher = baumbachBeteiligtenListe.get(i.intValue() - 1);
//							satz.append((TenorTexter.AUSSERGERICHTLICHE_KOSTEN_TEIL + gleicher.parteiBezeichner(GENITIV,
//									gleicher.getLfdNr(), isEinzigerSeinerArt(gleicher) /* false */ )));
//							gleiche.add(gleicher);
//						}
//						/*
//						 * TODO komisch, wenn einer allein die Kosten trägt: mehrfaches "und" (immer
//						 * noch?) - aber besser als nur Kommata
//						 */
//						satz = new StringBuffer(undStattKomma(satz.toString()));
//						satz.append(" haben ");
//					}
//				}
//			} else {
//				if (gewechselt) {
//					/*
//					 * Wenn Gläubiger wechselt: Gesamt-Kostentenor ergänzen und neuen Satz beginnen.
//					 */
//					kostenTenorSaetze.add(satz);
//					satz = new StringBuffer(String.format(TenorTexter.DIE_AUSSERG_KOSTEN, glaeubiger.parteiBezeichner(
//							GENITIV, glaeubiger.getLfdNr(), isEinzigerSeinerArt(glaeubiger) /* false */ )));
//				}
//			}
//			String darueberHinaus = "";
//			if (verhaeltnis.isDarueberHinaus()) {
//				darueberHinaus = TenorTexter.DARUEBER_HINAUS;
//			}
//			/*
//			 * Wenn eine Partei 100% trägt, nur den Bezeichner ausgeben, sonst auch die
//			 * Prozente; umgekehrte Abfrage wegen Double-Ungenauigkeiten.
//			 */
//			if (verhaeltnis.getKostenBruchteil() < 1.0) {
//				satz.append(String.format(TenorTexter.PARTEI_ZU_PROZENT,
//						verhaeltnis.getSchuldner().parteiBezeichner(NOMINATIV, verhaeltnis.getSchuldner().getLfdNr(),
//								isEinzigerSeinerArt(verhaeltnis.getSchuldner())/* false */ ) + darueberHinaus,
//						(100.0 * verhaeltnis.getKostenBruchteil())));
//			} else {
//				satz.append(
//						verhaeltnis.getSchuldner().parteiBezeichner(NOMINATIV, verhaeltnis.getSchuldner().getLfdNr(),
//								isEinzigerSeinerArt(verhaeltnis.getSchuldner()) /* false */ ));
//			}
//		}
//		/*
//		 * Zum Schluss nochmal: Gesamt-Kostentenor ergänzen.
//		 */
//		kostenTenorSaetze.add(satz);
//		for (StringBuffer sb : kostenTenorSaetze) {
//			// tenor.append();//kostenTenorSaetze[zeile]
//			tenor.append(super.hatStattHaben(undStattKomma(letztesKommaWeg(sb.toString()))));
//			tenor.append(TenorTexter.ZU_TRAGEN);
//		}
//		if (!alleKostenGleich)
//			tenor.append(TenorTexter.KOSTENTRAGUNG_SELBST);
//		return tenor.toString();
//	}
//
//	@Override
//	public String texteVollstreckbarkeit(EntscheidungsListenElemente<VollstreckungsVerhaeltnis> elemente,
//			VollstreckbarkeitsListe liste) {
//		String schuldner = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER;
//		String glaeubiger = TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER;
//
//		if (liste.sindAlle708iVm713()) {
//			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR;
//		} else if (liste.sindAlle709()) {
//			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709;
//		} else if (liste.sindAlle708iVm711()) {
//			if (1 == zaehle708iVm711(elemente.prozessverhaeltnisse)) {
//				// liste.werIst708iVm711().size()
//				/*
//				 * Wenn es nur ein solches Verhältnis gibt, dann die Parteien konkret benennen.
//				 */
//				Beteiligter schuldnerBeteiligter = elemente.prozessverhaeltnisse.get(0).getSchuldner();
//				Beteiligter glaeubigerBeteiligter = elemente.prozessverhaeltnisse.get(0).getGlaeubiger();
//				schuldner = schuldnerBeteiligter.parteiBezeichner(DATIV, schuldnerBeteiligter.getLfdNr(),
//						isEinzigerSeinerArt(schuldnerBeteiligter));
//				glaeubiger = glaeubigerBeteiligter.parteiBezeichner(NOMINATIV, glaeubigerBeteiligter.getLfdNr(),
//						isEinzigerSeinerArt(glaeubigerBeteiligter));
//			}
//			/* Sonst bleibt's beim "jeweiligen" Gläubiger/Schuldner */
//			return TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR + String
//					.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711, satzBeginn(schuldner), glaeubiger);
//		}
//		StringBuffer tenor = new StringBuffer(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR);
//		/*
//		 * Bei gemischtem Tenor erst die 709-er, dann die 708/711-er Verhältnisse
//		 * ausgeben. Dafür zwei StringBuffer füllen.
//		 */
//		StringBuffer tenor709 = new StringBuffer();
//		StringBuffer tenor708 = new StringBuffer();
//		final int size709 = zaehle709(elemente.prozessverhaeltnisse);// liste.werIst709().size();
//		final int size708u711 = zaehle708iVm711(elemente.prozessverhaeltnisse);// liste.werIst708iVm711().size();
//		if (size709 > 1) {
//			tenor709 = new StringBuffer("Hinsichtlich des jeweiligen Vollstreckungsverhältnisses ");
//		}
//		if (size708u711 > 1) {
//			tenor708 = new StringBuffer("Hinsichtlich des jeweiligen Vollstreckungsverhältnisses ");
//		}
//
//		/*
//		 * Alle VollstreckungsVerhaeltnisse durchgehen. Jeweils nach
//		 * Vollstreckbarkeitsart trennen. Innerhalb der Vollstreckbarkeitsart gucken, ob
//		 * es das einzige der Art ist.
//		 */
//		for (VollstreckungsVerhaeltnis verhaeltnis : elemente.prozessverhaeltnisse) {
//			Beteiligter schuldnerBeteiligter = verhaeltnis.getSchuldner();
//			Beteiligter glaeubigerBeteiligter = verhaeltnis.getGlaeubiger();
//			if (verhaeltnis.getVollstreckbarkeit().isB709s1()) {
//				/* Gegen Sicherheitsleistung. */
//				if (size709 == 1) {
//					schuldner = schuldnerBeteiligter.parteiBezeichner(DATIV, schuldnerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(schuldnerBeteiligter));
//					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(AKKUSATIV, glaeubigerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(glaeubigerBeteiligter));
//					tenor709 = new StringBuffer(String.format("Für %s gegenüber %s", glaeubiger, schuldner));
//				} else {
//					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(GENITIV, glaeubigerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(glaeubigerBeteiligter));
//					schuldner = schuldnerBeteiligter.parteiBezeichner(AKKUSATIV, schuldnerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(schuldnerBeteiligter));
//					tenor709.append(String.format("%s gegen %s, ", glaeubiger, schuldner));
//				}
//			} else if (verhaeltnis.getVollstreckbarkeit().isB711()) {
//				/* Abwendungsbefugnis */
//				if (size708u711 == 1) {
//					schuldner = schuldnerBeteiligter.parteiBezeichner(DATIV, schuldnerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(schuldnerBeteiligter));
//					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(NOMINATIV, glaeubigerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(glaeubigerBeteiligter));
//					tenor708 = new StringBuffer(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711,
//							satzBeginn(schuldner), glaeubiger));
//				} else {
//					glaeubiger = glaeubigerBeteiligter.parteiBezeichner(GENITIV, glaeubigerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(glaeubigerBeteiligter));
//					schuldner = schuldnerBeteiligter.parteiBezeichner(AKKUSATIV, schuldnerBeteiligter.getLfdNr(),
//							isEinzigerSeinerArt(schuldnerBeteiligter));
//					tenor708.append(String.format("%s gegen %s, ", glaeubiger, schuldner));
//				}
//			}
//		}
//
//		/*
//		 * Erst die §709-Vollstreckbarkeitsverhältnisse anhängen – hier muss immer der
//		 * Abschlus dran, aber ggfs. noch Kommata weg!
//		 */
//		if (size709 > 0) {
//			if (1 == size709) {
//				tenor.append(tenor709);
//			} else {
//				tenor.append(TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(tenor709.toString())));
//			}
//			tenor.append(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_709_EINZELN);
//		}
//
//		/* Dann die §708-Vollstreckbarkeitsverhältnisse */
//		if (size708u711 == 1) {
//			/* Bei einem ist der Abschluss schon dran */
//			tenor.append(tenor708);
//		} else if (size708u711 > 1) {
//			/* wenn es mehrere gibt, muss noch ein Abschluss dran. */
//			tenor.append(String.format(TenorTexter.URTEIL_VORLAEUFIG_VOLLSTRECKBAR_708_711_WERIST,
//					TenorTexter.undStattKomma(TenorTexter.letztesKommaWeg(tenor708.toString())),
//					TenorTexter.JEWEILIGER_VOLLSTRECKUNGSSCHULDNER, TenorTexter.JEWEILIGER_VOLLSTRECKUNGSGLAEUBIGER));
//		}
//		return tenor.toString();
//	}
//
//	/**
//	 * Die Methode erzeugt eine Streitwertfestsetzung, einen formatierten String
//	 * nach dem Muster "Der Streitwert wird auf %,.2f EUR festgesetzt.%n", bei einer
//	 * Widerklage eine Zusammensetzung aus Gesamtstreitwert, Klagewert und
//	 * Widerklagewert.
//	 * 
//	 * @param elemente {@link StreitwertEntscheidungsElemente}
//	 * 
//	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteStreitwert(eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer.StreitwertEntscheidungsElemente)
//	 */
//	@Override
//	public String texteStreitwert(StreitwertEntscheidungsElemente elemente) {
//		if (baumbachBeteiligtenListe.enthaeltAnWiderklageBeteiligte()) {
//			// Bei einer Widerklage müssen beide Streitwerte ausgegeben werden
//			return String.format(TenorTexter.STREITWERT_KLAGE_WIDERKLAGE, elemente.streitwerte.get(0).doubleValue(),
//					elemente.streitwerte.get(1).doubleValue(), elemente.streitwerte.get(2).doubleValue());
//		} else {
//			// Ohne Widerklage muss nur ein Streitwert ausgegeben werden
//			return String.format(TenorTexter.STREITWERT_KLAGE, elemente.streitwerte.get(0).doubleValue());
//		}
//	}
//
//	/**
//	 * @param quoten d. {@link #quoten}, d. gesetzt werden soll als
//	 *               {@link double[][]}.
//	 */
//	public void setQuoten(double[][] quoten) {
//		this.quoten = quoten;
//	}
//
//	/**
//	 * @param imUebrigen d. {@link #imUebrigen}, d. gesetzt werden soll als
//	 *                   {@link BaumbachImUebrigen}.
//	 */
//	public void setImUebrigen(BaumbachImUebrigen imUebrigen) {
//		this.imUebrigen = imUebrigen;
//	}
//
//	/**
//	 * Die Methode ermittelt, ob der Beteiligte der einzige für seinen
//	 * Beteiligten-Typ ist. Gesamtschuldner zählen nicht.
//	 * 
//	 * @param beteiligter ein {@link Beteiligter}
//	 * 
//	 * @return false, wenn es noch andere Beteiligte vom selben Typ gibt, die nicht
//	 *         BaumbachGesamtschuldner sind; sonst true.
//	 */
//	private boolean isEinzigerSeinerArt(Beteiligter beteiligter) {
//		return baumbachBeteiligtenListe.findeGesamtschuldnerIndices(beteiligter.getTyp(), false).size() < 2;
//	}
//
//	/**
//	 * Findet alle Kostenverteilungen von außergerichtlichen Kosten, die mit der
//	 * Verteilung der Gerichtskosten völlig identisch sind (z.B. werden ohne
//	 * Widerklage die Gerichtskosten und die außergerichtlichen Kosten des Klägers
//	 * oft gleich verteilt oder eine Partei muss z.B. alles tragen).
//	 * 
//	 * @param quoten ein Array double[][] mit allen Kostenquoten
//	 * @return eine ArrayList<Integer> mit allen Zeilen, die identisch sind (dürfte
//	 *         meist aber nur eine Zeile sein).
//	 */
//	private ArrayList<Integer> findeGleichlautendeQuoten(double[][] quoten) {
//		ArrayList<Integer> liste = new ArrayList<Integer>();
//		if (quoten == null || quoten.length < 1)
//			return null;
//		for (int zeile = 0; zeile < quoten.length; zeile++) {
//			if (zeile > 0 && istGleichlautendeQuote(quoten[0], quoten[zeile])) {
//				liste.add(new Integer(zeile));
//			} else if (zeile > 0) {
//			}
//		}
//		return liste;
//	}
//
//	/**
//	 * Prüft, ob die beiden double[]-Arrays identisch sind, um zu gucken, ob die
//	 * außergerichtlichen Kosten eines bestimmten Beteiligten genauso verteilt sind
//	 * wie die Gerichtskosten.
//	 * 
//	 * @param gerichtskostenQuoten double[]-Array mit der Verteilung der
//	 *                             Gerichtskosten
//	 * @param andereQuoten         double[]-Array mit der Verteilung der
//	 *                             außergerichtlichen Kosten eines Beteiligten
//	 * @return true, wenn identisch
//	 */
//	private boolean istGleichlautendeQuote(double[] gerichtskostenQuoten, double[] andereQuoten) {
//		boolean zwischen = true;
//		for (int spalte = 0; spalte < andereQuoten.length; spalte++) {
//			/*
//			 * Muss auf zwei Nachkommastellen runden, sonst erkennt er Identität nicht immer
//			 * wegen Double-Rundungsfehlern.
//			 */
//			if (Math.round(gerichtskostenQuoten[spalte] * 100) != Math.round(andereQuoten[spalte] * 100)) {
//				zwischen = false;
//			} else {
//				// Gleichlautend, es bleibt bei true, wenn es nicht vorher geändert wurde
//			}
//		}
//		return zwischen;
//	}
//
//	private int zaehle708iVm711(List<VollstreckungsVerhaeltnis> liste) {
//		int size = 0;
//		for (VollstreckungsVerhaeltnis verhaeltnis : liste) {
//			if (verhaeltnis.getVollstreckbarkeit().isB711())
//				size++;
//		}
//		return size;
//	}
//
//	private int zaehle709(List<VollstreckungsVerhaeltnis> liste) {
//		int size = 0;
//		for (VollstreckungsVerhaeltnis verhaeltnis : liste) {
//			if (verhaeltnis.getVollstreckbarkeit().isB709s1())
//				size++;
//		}
//		return size;
//	}
//
//	
}
