/**
 * AngriffsVerteidigungsmittelStaffelung.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.forderungen;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import eu.gronos.kostenrechner.data.KostenZweierquote;
import eu.gronos.kostenrechner.data.forderungen.Angriffserfolg;
import eu.gronos.kostenrechner.data.forderungen.AufrechnungsForderung;
import eu.gronos.kostenrechner.data.forderungen.Forderung;
import eu.gronos.kostenrechner.data.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.data.forderungen.KlageForderung;
import eu.gronos.kostenrechner.data.tenordaten.Allgemein;
import eu.gronos.kostenrechner.data.tenordaten.BegruendungsZahlenZeile;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.KostenTragungsVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.Vollstreckbarkeit;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckbarkeitsListe;
import eu.gronos.kostenrechner.data.tenordaten.VollstreckungsVerhaeltnis;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.interfaces.TenorVorbereitend;
import eu.gronos.kostenrechner.interfaces.UnterContainerKlasse;
import eu.gronos.kostenrechner.logic.TenorTexter;
import eu.gronos.kostenrechner.logic.VollstreckbarkeitsHelfer;
import eu.gronos.kostenrechner.util.BegruendungsZahlenTabelle;
import eu.gronos.kostenrechner.util.VerlusteBank;
import eu.gronos.kostenrechner.util.forderungen.ForderungsStaffelungPruefer;

/**
 * Eine Klasse zum Berechnen von Streitwert und von Kostenverteilung bei
 * Staffelung von Klageforderungen und Hilfsaufrechnungen
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 01.08.2014
 */
public class AngriffsVerteidigungsmittelStaffelung implements TenorVorbereitend/* , Begruendend */ {
	private final TenorDatenContainer container;

	private static final String[] COLUMN_HEADERS = new String[] { "Angriffs-/Verteidigungsmittel",
			"Höhe der Forderung (EUR)", "Erhöht den Streitwert um (EUR)", "Davon begründet (EUR)",
			"(verbleibende) Klageforderung" };
	private static final String GRUENDE_START_ANGRIFFS_VERTEIDIGUNGS_MITTEL = "Denn das Verhältnis von Obsiegen und Unterliegen bemisst sich anhand folgender Angriffs- und Verteidigungsmittel:\n";
	private static final String BESCHREIBUNG = "Tenor bei Klage/Hilfsaufrechnung (Hauptsachetenor, Kostentenor, Vollstreckbarkeitstenor und Streitwertbeschluss)";
	/**
	 * Der Beteiligter mit dem Namen klaeger speichert Genus/Numerus für die
	 * Tenorerzeugung
	 */
	private final Beteiligter klaeger;
	/**
	 * Der Beteiligter mit dem Namen beklagter speichert Genus/Numerus für die
	 * Tenorerzeugung
	 */
	private final Beteiligter beklagter;
	private Angriffserfolg hauptsache = null;
	private KostenZweierquote kosten = null;
	/**
	 * in der {@link UnterContainerKlasse} Allgemein stehen {@link Allgemein#b92ii}
	 * und {@link Allgemein#vomHundert92ii} sowie den {@link Allgemein#streitwert}.
	 */
	private Allgemein allgemein;
	/**
	 * Die {@link UnterContainerKlasse} des Typs {@link ForderungsStaffelung}
	 * enthält die {@link ArrayList}&lt;{@link Forderung}&gt; mit allen Forderungen,
	 * nach denen zu berechnen ist.
	 */
	private ForderungsStaffelung staffelung;
	private StringBuilder gruende = new StringBuilder();
	private BegruendungsZahlenTabelle zeilen;
	private VollstreckbarkeitsListe vollstreckbarkeitsListe;
	private final AngriffsTexter texter;// = new AngriffsTexter(allgemein.alsBruch);
	private final Pruefend<ForderungsStaffelung> pruefer = new ForderungsStaffelungPruefer();

	// Für VerlusteBank
	private ArrayList<Beteiligter> klaegerlein;
	private ArrayList<Beteiligter> beklagte;

	private final VerlusteBank unterliegen;

	/**
	 * Konstruktor:
	 * 
	 * 
	 * @param forderungsStaffelung ein Parameterobjekt {@link ForderungsStaffelung},
	 *                             das beinhaltet:
	 *                             {@link ForderungsStaffelung.forderungen} eine
	 *                             ArrayList<Forderung> mit allen Forderungen, nach
	 *                             denen zu berechnen ist.
	 *                             {@link ForderungsStaffelung#parteien} ein
	 *                             Beteiligter klaeger und ein Beteiligter
	 *                             beklagter, in dem Genus/Numerus für die
	 *                             Tenorerzeugung gespeichert sind
	 * @param allgemein            ein Parameterobjekt {@link Allgemein} mit b92ii =
	 *                             § 92 Abs. 2 ZPO berücksichtigen (d.h.
	 *                             unwesentlichesUnterliegenUebergehen) und ein
	 *                             Unterliegen < 10% wie kein Unterliegen behandeln
	 * @throws IllegalArgumentException Wenn forderungen null oder leer ist; oder
	 *                                  wenn die Reihenfolge (erst Klageforderungen,
	 *                                  dann Aufrechnungsforderungen) nicht
	 *                                  eingehalten ist.
	 * 
	 */
	public AngriffsVerteidigungsmittelStaffelung(VerfahrensDatenContainer verfahrensDaten)
			throws IllegalArgumentException {
		this.staffelung = verfahrensDaten.staffelung;
		pruefer.pruefeEingabe(staffelung);
		setzeForderungenZurueck(staffelung.forderungen);
		this.klaeger = staffelung.parteien.get(0);
		this.beklagter = staffelung.parteien.get(1);

		// Für die VerlusteBank

		this.klaegerlein = new ArrayList<>(staffelung.parteien.subList(0, 1));
		this.beklagte = new ArrayList<>(staffelung.parteien.subList(1, staffelung.parteien.size()));

		this.allgemein = verfahrensDaten.allgemein;
		texter = new AngriffsTexter(allgemein.alsBruch);
		this.container = new TenorDatenContainer(verfahrensDaten);
		allgemein.streitwert = null;// -1.0;
		unterliegen = new VerlusteBank();
	}

	/**
	 * Die Methode soll einen Hauptsachetenor erzeugen.
	 * 
	 * @return {@link EntscheidungsListenElemente} {@link HauptsacheVerhaeltnis}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeHauptsacheEntscheidung()
	 */
	@Override
	public EntscheidungsListenElemente<HauptsacheVerhaeltnis> erzeugeHauptsacheEntscheidung() {
		EntscheidungsListenElemente<HauptsacheVerhaeltnis> entscheidung = new EntscheidungsListenElemente<HauptsacheVerhaeltnis>();
		/* Hauptsache nur berechnen, wenn nicht schon geschehen */
		if (hauptsache == null)
			hauptsache = berechneHauptsache();
		entscheidung.prozessverhaeltnisse.clear();
		entscheidung.prozessverhaeltnisse
				.add(new HauptsacheVerhaeltnis(klaeger, beklagter, false, hauptsache.getErfolg()));
		// hauptsache[1] (long)
		texter.setGanz(hauptsache.getVolleForderung());
		entscheidung.text = texter.texteHauptsache(entscheidung);
		return entscheidung;
	}

	/**
	 * Die Methode soll einen Kostentenor erzeugen.
	 * 
	 * @return einen String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeKostenEntscheidung()
	 * 
	 * @todo TODO {@link Fraction}!
	 */
	@Override
	public EntscheidungsListenElemente<KostenTragungsVerhaeltnis> erzeugeKostenEntscheidung() {
		EntscheidungsListenElemente<KostenTragungsVerhaeltnis> kostenEntscheidung = new EntscheidungsListenElemente<KostenTragungsVerhaeltnis>();
		kostenEntscheidung.prozessverhaeltnisse.clear();
		final Fraction fract92ii = Fraction.valueOf(allgemein.vomHundert92ii, 100);
		/*
		 * Kostenquoten nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * sind sie null).
		 */
		if (kosten == null)
			kosten = berechneKostenverteilung();
		// if ((kosten.getKlaegerQuote() <= 0 && !allgemein.b92ii)/* kosten[0] */
		// || (kosten.getKlaegerQuote() < ((double) allgemein.vomHundert92ii / 100.0)/*
		// kosten[0] */&& allgemein.b92ii)) {
		if ((kosten.getKlaegerQuote().lessThanOrEqualTo(Fraction.ZERO) && !allgemein.b92ii)
				|| (kosten.getKlaegerQuote().lessThan(fract92ii)) && allgemein.b92ii) {
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false, Fraction.ONE));// 1.0
			// ((kosten.getBeklagtenQuote() <= 0 && !allgemein.b92ii)/* kosten[1] */
			// || (kosten.getBeklagtenQuote() < ((double) allgemein.vomHundert92ii /
			// 100.0)/* kosten[1] */ && allgemein.b92ii))
		} else if ((kosten.getBeklagtenQuote().lessThanOrEqualTo(Fraction.ZERO) && !allgemein.b92ii)
				|| (kosten.getBeklagtenQuote().lessThan(fract92ii) && allgemein.b92ii)) {
			kostenEntscheidung.prozessverhaeltnisse
					.add(new KostenTragungsVerhaeltnis(beklagter, klaeger, false, Fraction.ONE));// 1.0
		} else {
			kostenEntscheidung.prozessverhaeltnisse.add(
					new KostenTragungsVerhaeltnis(beklagter, klaeger, false, kosten.getKlaegerQuote()));/* kosten[0] */// .doubleValue()
			kostenEntscheidung.prozessverhaeltnisse.add(new KostenTragungsVerhaeltnis(klaeger, beklagter, false,
					kosten.getBeklagtenQuote()));/* kosten[1] */// .doubleValue()
		}
		// if (kosten.getKlaegerQuote() <= 0 || kosten.getBeklagtenQuote() <= 0)/*
		// kosten[0] *//* kosten[1] */
		if (kosten.getKlaegerQuote().lessThanOrEqualTo(Fraction.ZERO)
				|| kosten.getBeklagtenQuote().lessThanOrEqualTo(Fraction.ZERO))
			starteGruende(TenorTexter.KOSTENENTSCHEIDUNG_91);
		// else if (allgemein.b92ii && (kosten.getKlaegerQuote() <= ((double)
		// allgemein.vomHundert92ii / 100.0)
		// || kosten.getBeklagtenQuote() <= ((double) allgemein.vomHundert92ii /
		// 100.0)))/* kosten[1] *//* kosten[0] */
		else if (allgemein.b92ii && (kosten.getKlaegerQuote().lessThanOrEqualTo(fract92ii)
				|| kosten.getBeklagtenQuote().lessThanOrEqualTo(fract92ii)))
			starteGruende(TenorTexter.KOSTENENTSCHEIDUNG_91_92_II);
		else
			starteGruende(TenorTexter.KOSTENENTSCHEIDUNG_92_II);
		erweitereGruende(GRUENDE_START_ANGRIFFS_VERTEIDIGUNGS_MITTEL);
		zeilen.toStringBuilder(gruende);
		kostenEntscheidung.text = texter.texteKostenentscheidung(kostenEntscheidung);
		return kostenEntscheidung;
	}

	/**
	 * Die Methode erzeugt den Tenor zur vorläufigen Vollstreckbarkeit und die
	 * entsprechenden {@link EntscheidungsListenElemente#prozessverhaeltnisse}
	 * 
	 * @return einen {@link EntscheidungsListenElemente} für
	 *         {@link VollstreckungsVerhaeltnis}
	 */
	@Override
	public EntscheidungsListenElemente<VollstreckungsVerhaeltnis> erzeugeVollstreckbarkeitsEntscheidung() {
		EntscheidungsListenElemente<VollstreckungsVerhaeltnis> vollEntscheidung = new EntscheidungsListenElemente<VollstreckungsVerhaeltnis>();

		if (vollstreckbarkeitsListe == null)
			vollstreckbarkeitsListe = berechneVollstreckbarkeit();
		erweitereGruende(vollstreckbarkeitsListe.toString());
		VollstreckbarkeitsHelfer helfer = new VollstreckbarkeitsHelfer(
				container.hauptsacheEntscheidung.prozessverhaeltnisse,
				container.kostenEntscheidung.prozessverhaeltnisse);
		vollEntscheidung.prozessverhaeltnisse = helfer.toVollstreckbarkeitsVerhaeltnisse(klaeger, beklagter,
				vollstreckbarkeitsListe);

		vollEntscheidung.text = texter.texteVollstreckbarkeit(vollEntscheidung, vollstreckbarkeitsListe);
		return vollEntscheidung;
	}

	/**
	 * Die Methode soll eine Streitwertfestsetzung erzeugen.
	 * 
	 * @return {@link StreitwertEntscheidungsElemente}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeStreitwertEntscheidung()
	 */
	@Override
	public StreitwertEntscheidungsElemente erzeugeStreitwertEntscheidung() {
		StreitwertEntscheidungsElemente swe = new StreitwertEntscheidungsElemente();
		/*
		 * Streitwert nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * ist er null oder < 0).
		 */
		if (allgemein.streitwert == null || allgemein.streitwert.lessThan(Euro.ZERO_CENTS))//
			allgemein.streitwert = berechneStreitwert();
		// swe.streitwerte.add(new Double(allgemein.streitwert).longValue());
		swe.streitwerte.add(allgemein.streitwert);
		swe.text = texter.texteStreitwert(swe);
		return swe;
	}

	/**
	 * Die Methode erzeugt einen leeren sonstigen Tenor.
	 * 
	 * @return ein {@link EntscheidungsElemente} für Sonstiges
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeSonstigeEntscheidung()
	 */
	@Override
	public EntscheidungsElemente erzeugeSonstigeEntscheidung() {
		EntscheidungsElemente sonstige = new EntscheidungsElemente();
		sonstige.text = "";
		return sonstige;
	}

	/**
	 * Die Methode gibt die Begründung für den Kostentenor als String zurück.
	 * 
	 * @return den Text Gründe/Entscheidungsgründe als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getGruende()
	 */
	@Override
	public String getGruende() {
		return this.gruende.toString();
	}

	/**
	 * Die Methode dient dazu, den gesamten {@link TenorDatenContainer} zu erzeugen.
	 * 
	 * @return das Ergebnis von erzeugeHauptsachetenor() + erzeugeKostentenor() +
	 *         erzeugeStreitwertFestsetzung()
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#erzeugeContainer()
	 */
	@Override
	public TenorDatenContainer erzeugeContainer() {
		container.berechnungsTyp = getBerechnungsTyp();
		container.hauptsacheEntscheidung = erzeugeHauptsacheEntscheidung();
		container.kostenEntscheidung = erzeugeKostenEntscheidung();
		container.vollstreckbarkeitsEntscheidung = erzeugeVollstreckbarkeitsEntscheidung();
		container.streitwertEntscheidung = erzeugeStreitwertEntscheidung();
		container.sonstigeEntscheidung = erzeugeSonstigeEntscheidung();
		container.begruendung = zeilen.toBegruendungsElemente(getGruende());
		container.allgemein.selectedPanel = ForderungsStaffelung.TAB_ID;
		return container;
	}

	/**
	 * Die Methode soll eine Beschreibung für die Art des Tenor zurückgeben, damit
	 * andere Klassen etwa das Dokument danach benennen können.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TenorVorbereitend#getBerechnungsTyp()
	 */
	@Override
	public String getBerechnungsTyp() {
		return BESCHREIBUNG;
	}

	/**
	 * Berechnet, inwieweit sich verbliebene Klageforderung und der Wert der
	 * Aufrechnung decken, indem es den kleineren von zwei Beträgen zurückliefert.
	 * Setzt dabei auch direkt das Feld effektiverWert, so dass eine Berechnung
	 * ausreicht und sich das Ergebnis danach auch mit
	 * {@link AufrechnungsForderung#getEffektiverWert() getEffektiverWert()}
	 * erfragen lässt.
	 * 
	 * @param verbliebeneKlageforderung was von der Klageforderung übrig blieb
	 * @param aufrechnung               eine {@link AufrechnungsForderung}
	 * 
	 * @return den kleineren der Werte
	 * @see Forderung#berechneSchnittmenge(double, double)
	 */
	public static Euro berechneEffektivenWert(AufrechnungsForderung aufrechnung, Euro verbliebeneKlageforderung) {
		aufrechnung.setEffektiverWert(Euro.min(verbliebeneKlageforderung, aufrechnung.getAntrag()));// Math.min
		return aufrechnung.getEffektiverWert().get();
	}

	/**
	 * Berechnet, inwieweit sich verbliebene Klageforderung und der Erfolg der
	 * Aufrechnung decken, indem es den kleineren von zwei Beträgen zurückliefert.
	 * Setzt dabei auch direkt das Feld effektiverErfolg, so dass eine Berechnung
	 * ausreicht und sich das Ergebnis danach auch mit
	 * {@link AufrechnungsForderung#getEffektiverErfolg() getEffektiverErfolg()}
	 * erfragen lässt.
	 * 
	 * @param verbliebeneKlageforderung was von der Klageforderung übrig blieb
	 * @param aufrechnung               eine {@link AufrechnungsForderung}
	 * 
	 * @return den kleineren der Werte
	 * @see Forderung#berechneSchnittmenge(double, double)
	 */
	public static Euro berechneEffektivenErfolg(AufrechnungsForderung aufrechnung, Euro verbliebeneKlageforderung) {
		aufrechnung.setEffektiverErfolg(Euro.min(verbliebeneKlageforderung, aufrechnung.getErfolg()));// Math.min
		return aufrechnung.getEffektiverErfolg().get();
	}

	/**
	 * Die Methode dient dazu, bei allen {@link AufrechnungsForderung
	 * AufrechnungsForderungen} der <code>liste</code>
	 * {@link AufrechnungsForderung#setzeEffektivenErfolgZurueck()} und
	 * {@link AufrechnungsForderung#setzeEffektivenWertZurueck()} durchzuführen. Das
	 * muss bei Initialiiserung der Berechnung durchgeführt werden. Wenn seit der
	 * letzten Berechnung Werte verändert wurden, kommt es sonst zu Fehlern.
	 * 
	 * @param liste eine {@link ArrayList}&lt;{@link Forderung}&gt;
	 */
	private void setzeForderungenZurueck(List<Forderung> liste) {
		for (Forderung forderung : liste)
			if (forderung instanceof AufrechnungsForderung) {
				AufrechnungsForderung aufrechnung = (AufrechnungsForderung) forderung;
				aufrechnung.setzeEffektivenErfolgZurueck();
				aufrechnung.setzeEffektivenWertZurueck();
			}
		;
	}

	/**
	 * Die Methode berechnet das Obsiegen des Klägers in der Hauptsache und
	 * ermittelt dabei auch die vollständige Klageforderung.
	 * 
	 * @return ein {@link Angriffserfolg}, war: ein Array double[], bei dem index 0
	 *         die komplette Klage ist und index 1 die verbleibende Klageforderung
	 *         für den Tenor.
	 */
	private Angriffserfolg berechneHauptsache() {// double[]
		Euro verbleibend = Euro.ZERO_CENTS;// double verbleibend = 0;// erfolg
		Euro volleForderung = Euro.ZERO_CENTS;// double = 0;
		for (Forderung f : staffelung.forderungen) {
			if (f instanceof KlageForderung) {
				/* Das Obsiegen bzgl. Klageforderungen wird zusammengerechnet */

				// verbleibend += f.getErfolg();
				verbleibend = verbleibend.add(f.getErfolg());
				/* Auch der Wert, sofern nicht wirtschaftlich identisch */
				if (!((KlageForderung) f).isWirtschaftlichIdentisch())
					volleForderung = volleForderung.add(f.getAntrag());
				// volleForderung += f.getWert();
			} else if (f instanceof AufrechnungsForderung) {
				/* Aufrechnungen werden abgezogen, soweit noch etwas übrig ist. */
				AufrechnungsForderung af = (AufrechnungsForderung) f;
				/*
				 * Dabei wird abgefragt, ob der effektive Erfolg schon berechnet wurde (dann ist
				 * getEffektiverErfolg nicht mehr < 0)
				 */ // af.getEffektiverErfolg() == null ||
					// af.getEffektiverErfolg().lessThan(Euro.ZERO_CENTS)
				if (!af.getEffektiverErfolg().isPresent()) {
					verbleibend = verbleibend.subtract(berechneEffektivenErfolg(af, verbleibend));
				} else {
					verbleibend = verbleibend.subtract(af.getEffektiverErfolg().get());
				}
				// verbleibend -= af.getEffektiverErfolg() < 0 ? berechneEffektivenErfolg(af,
				// verbleibend)
				// : af.getEffektiverErfolg();
			}
		}
		return new Angriffserfolg(volleForderung, verbleibend);
	}

	/**
	 * Die Methode berechnet die Kostenverteilung, damit das Ergebnis von
	 * {@link AngriffsVerteidigungsmittelStaffelung#erzeugeKostenEntscheidung()
	 * erzeugeKostentenor()} ausgelesen werden kann.
	 * 
	 * @return eine {@link KostenZweierquote}, war ein Array double[] mit der Quote
	 *         des Klägers und des Beklagten, jeweils zwischen 0.0 und 1.0
	 * 
	 * @see AngriffsVerteidigungsmittelStaffelung#erzeugeKostenEntscheidung()
	 */
	private KostenZweierquote berechneKostenverteilung() {
		/*
		 * Streitwert nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * ist er < 0).
		 */
		if (allgemein.streitwert == null || allgemein.streitwert.lessThan(Euro.ZERO_CENTS))// < 0)
			allgemein.streitwert = berechneStreitwert();
		// Euro klaegerunterliegen = Euro.ZERO_CENTS;// double klaegerunterliegen = 0.0;
		// Euro beklagtenunterliegen = Euro.ZERO_CENTS;// double beklagtenunterliegen =
		// 0.0;
		Euro verbleibend = Euro.ZERO_CENTS;// double verbleibend = 0.0;
		int aufrechnungen = 0;
		/*
		 * Liste der Zeilen der Begründungstabelle anlegen und mit der Kopfzeile
		 * beginnen
		 */
		zeilen = new BegruendungsZahlenTabelle();
		zeilen.add(new ArrayList<String>(Arrays.asList(COLUMN_HEADERS)));
		for (Forderung f : staffelung.forderungen) {
			//DoubleDataRows doubles = new DoubleDataRows(4);
			BegruendungsZahlenZeile zahlenZeile = new BegruendungsZahlenZeile(4);
			String beschreibung;
			if (f instanceof KlageForderung) {
				beschreibung = f.getBeschreibung();
				/* Das Obsiegen bzgl. Klageforderungen wird zusammengerechnet */
				// verbleibend += f.getErfolg();
				verbleibend = verbleibend.add(f.getErfolg());
				/*
				 * Das Unterliegen bemisst sich hier allein nach Wert und Erfolg.
				 */
				// klaegerunterliegen += f.getWert() - f.getErfolg();
				// klaegerunterliegen =
				// klaegerunterliegen.add(f.getAntrag().subtract(f.getErfolg()));
				unterliegen.add(klaegerlein, f.getAntrag().subtract(f.getErfolg()));
				// beklagtenunterliegen += f.getErfolg();
				// beklagtenunterliegen = beklagtenunterliegen.add(f.getErfolg());
				unterliegen.add(beklagte, f.getErfolg());
				/*
				 * Bei den Hauptklageforderungen muss einiges doppelt in die Begründungstabelle
				 * ausgegeben werden.
				 */
				//doubles.add(f.getAntrag().doubleValue());// (new Double);
				zahlenZeile.add(f.getAntrag());
				//doubles.add(f.getAntrag().doubleValue());// (new Double);
				zahlenZeile.add(f.getAntrag());
				//doubles.add(f.getErfolg().doubleValue());// (new Double);
				zahlenZeile.add(f.getErfolg());
				//doubles.add(verbleibend.doubleValue());// (new Double);
				zahlenZeile.add(verbleibend);
				zeilen.add(beschreibung, zahlenZeile);//doubles);
			} else if (f instanceof AufrechnungsForderung) {
				aufrechnungen++;
				beschreibung = String.format("%d. %s", aufrechnungen, f.getBeschreibung());
				/* Aufrechnungen werden abgezogen, soweit noch etwas übrig ist. */
				AufrechnungsForderung af = (AufrechnungsForderung) f;
				/*
				 * Das Unterliegen bemisst sich hier nach effektivem Wert und effektivem Erfolg.
				 * Dabei wird abgefragt, ob der effektive Erfolg schon berechnet wurde (dann ist
				 * getEffektiverErfolg nicht mehr < 0)
				 */
				// double effektiverErfolg = af.getEffektiverErfolg() < 0
				// Euro.undefined(af.getEffektiverErfolg())
				final Euro effektiverErfolg = !af.getEffektiverErfolg().isPresent()
						? berechneEffektivenErfolg(af, verbleibend)
						: af.getEffektiverErfolg().get();
				// double effektiverWert = af.getEffektiverWert() < 0 ?
				// Euro.undefined(af.getEffektiverWert())
				final Euro effektiverWert = !af.getEffektiverWert().isPresent()
						? berechneEffektivenWert(af, verbleibend)
						: af.getEffektiverWert().get();
				// verbleibend -= effektiverErfolg;
				verbleibend = verbleibend.subtract(effektiverErfolg);
				// klaegerunterliegen += effektiverErfolg;//subtract
				// klaegerunterliegen = klaegerunterliegen.add(effektiverErfolg);
				unterliegen.add(klaegerlein, effektiverErfolg);
				// beklagtenunterliegen += effektiverWert - effektiverErfolg;
				// beklagtenunterliegen =
				// beklagtenunterliegen.add(effektiverWert).subtract(effektiverErfolg);
				unterliegen.add(beklagte, effektiverWert.subtract(effektiverErfolg));
				/*
				 * Bei den Aufrechnungen wird alles in der Begründungstabelle gebraucht
				 */
//				doubles.add(f.getAntrag().doubleValue());// (new Double(f.getWert()));
				zahlenZeile.add(f.getAntrag());
//				doubles.add(af.getEffektiverWert().get().doubleValue());// (new Double);
				zahlenZeile.add(af.getEffektiverWert().get());
//				doubles.add(af.getEffektiverErfolg().get().doubleValue());// (new Double);
				zahlenZeile.add(af.getEffektiverErfolg().get());
//				doubles.add(verbleibend.doubleValue());// (new Double);//
				zahlenZeile.add(verbleibend);
				zeilen.add(beschreibung, zahlenZeile);//doubles);
			}
		}
		/* Eine Ergebniszeile gibt es hier nicht */
		System.out.println("Fiktiver Streitwert:" + unterliegen.getFiktiverStreitwert() + ";Fraction Kl:"
				+ unterliegen.getFractionFor(klaegerlein) + ";Bk:" + unterliegen.getFractionFor(beklagte));
		return new KostenZweierquote(//
				// klaegerunterliegen.divide(allgemein.streitwert),
				// beklagtenunterliegen.divide(allgemein.streitwert));
				// unterliegen.getQuotaFor(klaegerlein), unterliegen.getQuotaFor(beklagte));
				unterliegen.getFractionFor(klaegerlein), unterliegen.getFractionFor(beklagte));
		// klaegerunterliegen / allgemein.streitwert,beklagtenunterliegen /
		// allgemein.streitwert);
	}

	/**
	 * Die Methode berechnet, inwieweit das Urteil vorläufig vollstreckbar ist.
	 * 
	 * @return eine {@link VollstreckbarkeitsListe}, deren 0. Element die
	 *         Unanfechtbarkeit speichert, wenn nicht unanfechtbar das 1. Element
	 *         das Verhältnis kl -> bk und das 2. Element das Verhältnis bk -> kl.
	 * 
	 */
	private VollstreckbarkeitsListe berechneVollstreckbarkeit() {
		VollstreckbarkeitsListe vollstreckbarkeitsListe = new VollstreckbarkeitsListe();
		/*
		 * Kostenquoten nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
		 * sind sie null). Die VerlusteBank kann für die Unanfechtbarkeit genutzt
		 * werden.
		 */
		if (kosten == null)
			kosten = berechneKostenverteilung();
//		Vollstreckbarkeit unanfechtbarkeit = pruefeUnanfechtbarkeit();
		final Vollstreckbarkeit unanfechtbarkeit = pruefeUnanfechtbarkeit(this.unterliegen);
		;
		vollstreckbarkeitsListe.add(unanfechtbarkeit);
		if (unanfechtbarkeit.isB713())
			return vollstreckbarkeitsListe;
		/*
		 * Beim Kläger hinsichtlich Hauptsache einsetzen, was von seiner
		 * Hauptklageforderung begründet ist (hauptsache[1]) und hinsichtlich der Kosten
		 * die üblichen Klägerkosten, soweit sie der Beklagte tragen muss (kosten[1])
		 */
		vollstreckbarkeitsListe.add(Vollstreckbarkeit.pruefeSicherheitsleistung(hauptsache.getErfolg(), // hauptsache[1],
				(TenorTexter.getKostenKlaeger().errechneGebuehrenSumme(allgemein.streitwert)
						.multiply(kosten.getBeklagtenQuote()))));/* kosten[1]* */
		/*
		 * Beim Beklagten hinsichtlich Hauptsache 0 einsetzen und hinsichtlich der
		 * Kosten die üblichen Beklagtenkosten, soweit sie der Kläger tragen muss
		 * (kosten[0])
		 */
		vollstreckbarkeitsListe.add(Vollstreckbarkeit.pruefeSicherheitsleistung(Euro.ZERO_CENTS, // 0L,
				(TenorTexter.getKostenBeklagter().errechneGebuehrenSumme(allgemein.streitwert)
						.multiply(kosten.getKlaegerQuote()))));/* kosten[0] */
		return vollstreckbarkeitsListe;
	}

	/**
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @param unterliegen
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 *         und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
	private Vollstreckbarkeit pruefeUnanfechtbarkeit(VerlusteBank unterliegen) {
		Vollstreckbarkeit unanfechtbarkeit = new Vollstreckbarkeit();
		final Euro streitwert = unterliegen.getFiktiverStreitwert();
		if (streitwert.compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0) {
			unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
		} else if (streitwert.compareTo(TenorTexter.BERUFUNGS_GRENZE.multiply(2.0)) <= 0) {
			if (unterliegen.get(klaegerlein).compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0
					&& unterliegen.get(beklagte).compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0) {
				unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
			}
		}
		return unanfechtbarkeit;
	}

	/*
	 * Die Methode prüft, ob das Urteil unanfechtbar ist (§§ 708, 713 ZPO).
	 * 
	 * @return ein {@link Vollstreckbarkeit}, das im Fall der Unanfechtbarkeit 708
	 * und 713 auf <code>true</code> setzt. Ansonsten enthaelt es nichts.
	 */
//	private Vollstreckbarkeit pruefeUnanfechtbarkeit() {
//		Vollstreckbarkeit unanfechtbarkeit = new Vollstreckbarkeit();
//		/*
//		 * Streitwert nur berechnen, wenn nicht schon geschehen (vor erster Berechnung
//		 * ist er null bzw < 0).
//		 */
//		if (Euro.undefined(allgemein.streitwert))// < 0
//			allgemein.streitwert = berechneStreitwert();
//		// allgemein.streitwert <=TenorTexter.BERUFUNGS_GRENZE
//		if (allgemein.streitwert.compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0) {
//			unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
//			// allgemein.streitwert <= TenorTexter.BERUFUNGS_GRENZE * 2
//		} else if (allgemein.streitwert.compareTo(TenorTexter.BERUFUNGS_GRENZE.multiply(2.0)) <= 0) {
//			/* Hauptsache nur berechnen, wenn nicht schon geschehen */
//			if (hauptsache == null)
//				hauptsache = berechneHauptsache();
//			/*
//			 * Wie ist der Kläger beschwert? Das, was von seiner Klage nicht durchgedrungen
//			 * ist.
//			 */
//			// double
//			Euro klaeger = hauptsache.getVolleForderung().subtract(hauptsache.getErfolg());
//			// hauptsache.getVolleForderung() - hauptsache.getErfolg();
//			// hauptsache[0] - hauptsache[1];
//			/* Wie ist der Beklagte beschwert: Das wozu er verurteilt wurde ... */
//			// double
//			Euro beklagter = hauptsache.getErfolg();// hauptsache[1];
//			/*
//			 * ... und das, was er an Aufrechnungsforderungen "verbraten" hat, ob mit oder
//			 * ohne Erfolg.
//			 */
//			for (Forderung f : staffelung.forderungen) {
//				if (f instanceof AufrechnungsForderung)
//					beklagter = beklagter.add(((AufrechnungsForderung) f).getEffektiverWert().get());
//				// beklagter += ((AufrechnungsForderung) f).getEffektiverWert();
//			}
//			// if (klaeger <= TenorTexter.BERUFUNGS_GRENZE && beklagter <=
//			// TenorTexter.BERUFUNGS_GRENZE) {
//			if (klaeger.compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0
//					&& beklagter.compareTo(TenorTexter.BERUFUNGS_GRENZE) <= 0) {
//				unanfechtbarkeit = new Vollstreckbarkeit(true, false, false, false, true);
//			}
//		}
//		return unanfechtbarkeit;
//
//	}

	/**
	 * Die Methode berechnet den Streitwert.
	 * 
	 * @return den Streitwert als double
	 */
	private Euro berechneStreitwert() {
		Euro wert = Euro.ZERO_CENTS;// double = 0.0;
		Euro verbleibend = Euro.ZERO_CENTS;// double = 0.0;
		for (Forderung f : staffelung.forderungen) {
			if (f instanceof KlageForderung) {
				/* Das Obsiegen bzgl. Klageforderungen wird zusammengerechnet */
				verbleibend // += f.getErfolg();
						= verbleibend.add(f.getErfolg());
				/* Auch der Wert, sofern nicht wirtschaftlich identisch */
				if (!((KlageForderung) f).isWirtschaftlichIdentisch())
					wert // += f.getWert();
							= wert.add(f.getAntrag());
			} else if (f instanceof AufrechnungsForderung) {
				/* Aufrechnungen werden abgezogen, soweit noch etwas übrig ist. */
				AufrechnungsForderung af = (AufrechnungsForderung) f;
				/*
				 * Dabei wird abgefragt, ob der effektive Wert und der effektive Erfolg schon
				 * berechnet wurden (dann sind getEffektiverWert bzw. getEffektiverErfolg nicht
				 * mehr < 0)
				 */
				wert // += af.getEffektiverWert() < 0 ? berechneEffektivenWert(af, verbleibend) :
						// af.getEffektiverWert(); Euro.undefined(af.getEffektiverWert())
						= wert.add(!af.getEffektiverWert().isPresent() ? berechneEffektivenWert(af, verbleibend)
								: af.getEffektiverWert().get());
				verbleibend // -= af.getEffektiverErfolg() < 0 ?
						= verbleibend.subtract(
								// Euro.undefined(af.getEffektiverErfolg())
								!af.getEffektiverErfolg().isPresent() ? berechneEffektivenErfolg(af, verbleibend)
										: af.getEffektiverErfolg().get());
			}
		}
		return wert;
	}

	/**
	 * Die Methode setzt die Entscheidungsgründe zurück (sofern nicht leer) und
	 * beginnt sie neu mit dem übergebenen Text.
	 * 
	 * @param text eine {@link CharSequence} mit dem Anfangssatz(-bestandteil) der
	 *             Gründe.
	 */
	private void starteGruende(CharSequence text) {
		if (gruende.length() > 0)
			gruende.delete(0, gruende.length());
		erweitereGruende(text);
	}

	/**
	 * Die Methode erweitert die bisherigen Gründe
	 * 
	 * @param text eine {@link CharSequence} mit den weiteren Gründen.
	 */
	private void erweitereGruende(CharSequence text) {
		gruende.append(text);
	}

}