/*
 * StreitwertReduktionTexter.java
 * eu.gronos.kostenrechner.logic.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.logic.gebuehren;

import eu.gronos.kostenrechner.data.gebuehren.Teilklageruecknahme;
import eu.gronos.kostenrechner.data.tenordaten.EntscheidungsListenElemente;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.HauptsacheVerhaeltnis;
import eu.gronos.kostenrechner.data.tenordaten.StreitwertEntscheidungsElemente;
import eu.gronos.kostenrechner.logic.TenorTexter;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.*;

/**
 * Die Klasse verfasst die Tenortexte für {@link StreitwertReduktion}
 *
 * @author Peter Schuster (setrok)
 * @date 3 Jun 2019
 *
 */
public class StreitwertReduktionTexter extends TenorTexter<Teilklageruecknahme> {
	private Euro[] streitwerte;

	public StreitwertReduktionTexter(boolean alsBruch) {
		super(alsBruch);
	}

	@Override
	public String texteHauptsache(EntscheidungsListenElemente<HauptsacheVerhaeltnis> hauptsache) {
		HauptsacheVerhaeltnis verhaeltnis = hauptsache.prozessverhaeltnisse.get(0);
		if (verhaeltnis.getVerurteilung().longValue() > 0)
			return satzBeginn(String.format(TenorTexter.VERURTEILUNG_ZAHLUNG_AN_GGF_TEILABWEISUNG_FORMAT,
					verhaeltnis.getSchuldner().parteiBezeichner(NOMINATIV, verhaeltnis.getSchuldner().getLfdNr(), true),
					getWirdWerden(verhaeltnis.getSchuldner().isPlural()),
					verhaeltnis.getGlaeubiger()
							.parteiBezeichner(AKKUSATIV, verhaeltnis.getGlaeubiger().getLfdNr(), true),
					verhaeltnis.getVerurteilung().doubleValue(),
					verhaeltnis.getVerurteilung().lessThan(streitwerte[1])
							? TenorTexter.KLAGEABWEISUNG_IM_UEBRIGEN_LEERSCHRITT
							: ""));
		else
			return TenorTexter.KLAGEABWEISUNG_ZEILE;
	}

	/**
	 * Die Methode verfasst eine Streitwertfestsetzung unter Berücksichtigung des
	 * früheren und des späteren Streitwerts. Das Datum der Reduktion muss man noch
	 * von Hand nachtragen.
	 * 
	 * @param elemente {@link StreitwertEntscheidungsElemente}, in dem
	 *                 {@link StreitwertEntscheidungsElemente#streitwerte} gefüllt
	 *                 ist.
	 * @return einen String im Format "Der Streitwert wird auf %,.2f EUR bis zum
	 *         ..., danach auf %,.2f EUR festgesetzt.%n" für
	 *         {@link StreitwertEntscheidungsElemente#streitwerte#text}
	 * @see eu.gronos.kostenrechner.logic.TenorTexter#texteStreitwert(eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer.StreitwertEntscheidungsElemente)
	 */
	@Override
	public String texteStreitwert(StreitwertEntscheidungsElemente elemente) {
		return String.format(TenorTexter.STREITWERTFESTSETZUNG_FORMAT, elemente.streitwerte.get(0).doubleValue(),
				elemente.streitwerte.get(1).doubleValue());
	}

	public void setStreitwerte(Euro[] streitwerte) {
		this.streitwerte = streitwerte;
	}
}
