/**
 * BeteiligtenTableModel.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.baumbach;

import java.util.List;

import javax.swing.table.AbstractTableModel;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter.GenusNumerus;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.RowHandler;
import eu.gronos.kostenrechner.interfaces.RowList;
import eu.gronos.kostenrechner.interfaces.TooltipLieferant;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.*;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.*;

/**
 * Ein {@link AbstractTableModel} zur Darstellung der Daten von
 * {@link BaumbachBeteiligter BaumbachBeteiligten}
 * 
 * @author Peter Schuster (setrok)
 * @date 04.05.2014
 * 
 * @see javax.swing.table.AbstractTableModel
 * 
 * @todo TODO Umstellung auf {@link Euro} vervollständigen
 */
public class BeteiligtenTableModel extends AbstractTableModel
		implements RowHandler<BaumbachBeteiligter>, RowList<BaumbachBeteiligter>, TooltipLieferant {

	private static final long serialVersionUID = -3535924357266123346L;
	protected BaumbachBeteiligtenListe values;
	protected final BeteiligtenTyp beteiligtenTyp;// int
	private final static int COLUMN_COUNT = 4;
	private final static Class<?>[] COLUMN_CLASSES = new Class<?>[] { // Double.class, Double.class,
			Beteiligter.class, Euro.class, Euro.class, Boolean.class }; // TODO Euro.class
	private static final String[] COLUMN_NAMES = new String[] { "Beteiligte(r)", "Höhe der Inanspruchnahme",
			"Verurteilung (Unterliegen)", "widerklagend/widerbeklagt" };
	private static final String[] COLUMN_TOOLTIPS = new String[] { "Parteibezeichnung für den Beteiligten?",
			"In welcher Höhe wurde der Beteiligte verklagt?",
			"In welcher Höhe war die Klage gegen diesen Beteiligten erfolgreich (Unterliegen des Beteiligten)?",
			"Ist der Beteiligte an der Widerklage beteiligt?" };
	private boolean widerKlageAnzeigen;
	private int widerKlaegerIndex = -1;
	public static final String GESAMTSCHULDNERISCH = "gesamtschuldnerisch";
	protected final static int COLUMN_BETEILIGTER = 0;
	public final static int COLUMN_STREITWERT = 1;
	protected final static int COLUMN_UNTERLIEGEN = 2;
	protected final static int COLUMN_WIDERKLAGE = 3;

	/**
	 * Konstruktor: erzeugt eine neue leere Tabelle von Beteiligten des Typs
	 * <code>beteiligtenTyp</code>.
	 * 
	 * @param beteiligtenTyp     eine der Konstanten Beteiligter.KLAEGER,
	 *                           Beteiligter.DRITTWIDERBEKLAGTE oder
	 *                           Beteiligter.BEKLAGTE
	 * @param widerKlageAnzeigen wenn true, so wird auch die Spalte "an Widerklage
	 *                           beteiligt" angezeigt
	 * 
	 */
	public BeteiligtenTableModel(BeteiligtenTyp beteiligtenTyp, boolean widerKlageAnzeigen) {
		super();
		this.beteiligtenTyp = beteiligtenTyp;
		this.values = new BaumbachBeteiligtenListe();
		this.widerKlageAnzeigen = widerKlageAnzeigen;
		setWiderklaegerIndex();
	}

	@Deprecated
	public BeteiligtenTableModel(int beteiligtenTyp, boolean widerKlageAnzeigen) {
		this(BeteiligtenTyp.values()[beteiligtenTyp], widerKlageAnzeigen);
	}

	/**
	 * Konstruktor: erzeugt eine neue Tabelle von Beteiligten des Typs
	 * <code>beteiligtenTyp</code> mit den Elementen <code>values</code>.
	 * 
	 * @param beteiligtenTyp     eine der Konstanten Beteiligter.KLAEGER,
	 *                           Beteiligter.DRITTWIDERBEKLAGTE oder
	 *                           Beteiligter.BEKLAGTE
	 * @param widerKlageAnzeigen wenn true, so wird auch die Spalte "an Widerklage
	 *                           beteiligt" angezeigt
	 * @param values             eine BaumbachBetiligtenListe mit den
	 *                           BaumbachBeteiligten eines Typs
	 */
	public BeteiligtenTableModel(BeteiligtenTyp beteiligtenTyp, boolean widerKlageAnzeigen,
			BaumbachBeteiligtenListe values) {
		this(beteiligtenTyp, widerKlageAnzeigen);
		this.values = values;
	}

	@Deprecated
	public BeteiligtenTableModel(int beteiligtenTyp, boolean widerKlageAnzeigen, BaumbachBeteiligtenListe values) {
		this(beteiligtenTyp, widerKlageAnzeigen);
		this.values = values;
	}

	/**
	 * Die Methode dient dazu, einen neuen BaumbachBeteiligter an die Werte
	 * anzuhängen.
	 * 
	 * @param bt der anzuhängende BaumbachBeteiligte
	 * @throws IllegalArgumentException es ist nur ein Kläger in einem
	 *                                  BeteiligtenTableModel erlaubt.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#addRow(java.lang.Object)
	 */
	public void addRow(BaumbachBeteiligter bt) throws IllegalArgumentException {
		if (getBeteiligtenTyp() == KLAEGER && getRowCount() > 0)
			throw new IllegalArgumentException("Derzeit nur ein Kläger erlaubt.");
		values.add(bt);
		final int row = getRowCount() - 1;
		super.fireTableRowsInserted(row, row);
		/*
		 * Wenn's gerade zwei geworden sind, muss bei der ersten Zeile nun die
		 * Nummerierung mit lfdNr aktiviert werden.
		 */
		if (getRowCount() == 2)
			super.fireTableCellUpdated(0, COLUMN_BETEILIGTER);
		setWiderklaegerIndex(row, values.get(row).isAnWiderklageBeteiligt());
	}

	/**
	 * Die Methode getRowCount dient dazu, die Zeilen des Models zu ermitteln.
	 * 
	 * A JTable uses this method to determine how many rows it should display. This
	 * method should be quick, as it is called frequently during rendering.
	 * 
	 * @return die Anzahl der Zeilen im TableModel
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	@Override
	public int getRowCount() {
		if (values != null)
			return values.size();
		else
			return 0;
	}

	/**
	 * Gibt die Anzahl der Spalten zurück. Die Anzahl variiert je nachdem, ob die
	 * Widerklage-Beteiligung angezeigt werden soll.
	 * 
	 * A JTable uses this method to determine how many columns it should create and
	 * display by default.
	 * 
	 * @return die Anzahl der Spalten als int.
	 * 
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	@Override
	public int getColumnCount() {
		int zwischen = COLUMN_COUNT;
		if (beteiligtenTyp == KLAEGER && !widerKlageAnzeigen)
			zwischen = 1;
		else if (!widerKlageAnzeigen)
			zwischen -= 1;
		return zwischen;
	}

	/**
	 * Die Methode getValueAt gibt den Wert bei Zeile rowIndex, Spalte columnIndex
	 * zurück.
	 * 
	 * @param rowIndex    die Zeile
	 * @param columnIndex die Spalte
	 * @return ein Object
	 * 
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		switch (columnIndex) {
		case COLUMN_BETEILIGTER:
			return getRow(rowIndex);
		case COLUMN_STREITWERT:
			return getStreitwertAt(rowIndex);
		case COLUMN_UNTERLIEGEN:
			return getUnterliegenAt(rowIndex);
		case COLUMN_WIDERKLAGE:
			return getWiderklageAt(rowIndex);
		default:
			return null;
		}
	}

	/**
	 * @return beteiligtenTyp gibt den beteiligtenTyp als {@link BeteiligtenTyp}
	 *         zurück.
	 */
	public BeteiligtenTyp getBeteiligtenTyp() {// int
		return beteiligtenTyp;
	}

	/**
	 * Entfernt eine Zeile aus dem Model, indem der entsprechende Eintrag aus der
	 * BaumbachBeteiligtenListe gelöscht wird.
	 * 
	 * @param rowIndex der Index der Zeile
	 * @throws IndexOutOfBoundsException if the index is out of range (index < 0 ||
	 *                                   index >= size()
	 * @throws IllegalArgumentException  wenn das TableModel bei Entfernen der Zeile
	 *                                   einen ungültigen Zustand annehmen würde
	 * @see java.util.ArrayList#remove(int)
	 * @see eu.gronos.kostenrechner.interfaces.RowHandler#removeRow(int)
	 */
	@Override
	public void removeRow(int rowIndex) throws IndexOutOfBoundsException, IllegalArgumentException {
		if (values == null || rowIndex < 0 || values.size() - 1 < rowIndex)
			return;
		if (values.enthaeltGesamtschuldner() && !values.isGesamtschuldnerschaft(rowIndex))
			throw new IllegalArgumentException(
					"Einzelne Beteiligte können nicht entfernt werden, solange noch eine gesamtschuldnerische Verurteilung vorliegt.");
		values.remove(rowIndex);
		fireTableRowsDeleted(rowIndex, rowIndex);
		/*
		 * Wenn's gerade nur noch einer geworden ist, muss bei der ersten Zeile nun die
		 * Nummerierung mit lfdNr deaktiviert werden.
		 */
		if (getRowCount() == 1)
			super.fireTableCellUpdated(0, COLUMN_BETEILIGTER);
		/* Der widerKlaegerIndex hat sich vielleicht geändert. */
		setWiderklaegerIndex();
		Kostenrechner.getLogger().info(String.format("widerKlaegerIndex: %d", widerKlaegerIndex));
	}

	/**
	 * Die Methode getBaumbachBeteiligterForRow dient dazu, den einer Zeile zugrunde
	 * zu liegenden BaumbachBeteiligten zurückzugeben.
	 * 
	 * @param rowIndex der Index der Zeile
	 * @return der einer Zeile zugrunde zu liegenden BaumbachBeteiligte
	 */
	@Override
	public BaumbachBeteiligter getRow(int rowIndex) throws IndexOutOfBoundsException {
		return values.get(rowIndex);
	}

	/**
	 * Die Methode getColumnName gibt den Namen der Spalte zurück.
	 * 
	 * @param columnIndex der Index der abgefragten Spalte
	 * @return den Spaltennamen als String
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
	 */
	@Override
	public String getColumnName(int columnIndex) {
		if (columnIndex == COLUMN_BETEILIGTER) {
			String zwischen = Beteiligter.parteiBezeichner(beteiligtenTyp, GenusNumerus.MAENNLICH_PLURAL, -1, NOMINATIV,
					true);
			return zwischen.substring(4, zwischen.length()) + "-Bezeichnung";
		} else if (columnIndex == COLUMN_WIDERKLAGE) {
			if (this.beteiligtenTyp == BEKLAGTE)
				return "widerklagend";
			else
				return "widerbeklagt";
		} else if (columnIndex < getColumnCount())
			return COLUMN_NAMES[columnIndex];
		return super.getColumnName(columnIndex);
	}

	/**
	 * Die Methode dient dazu, der JTable mitzuteilen, welcher Art die Werte der
	 * jeweiligen Tabellenspalte sind. Überschreibt
	 * {@link AbstractTableModel.getColumnClass(int)}, das immer Object.class
	 * zurückgibt.
	 * 
	 * @param columnIndex der Index der abgefragten Spalte
	 * @return die Klasse des Rückgabewerts der Spalte
	 * 
	 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
	 */
	@Override
	public Class<?> getColumnClass(int columnIndex) {
		if (columnIndex < getColumnCount())
			return COLUMN_CLASSES[columnIndex];
		return super.getColumnClass(columnIndex);
	}

	/**
	 * Mit der Methode lässt sich steuern, ob die JTable eine (oder mehrere, bei
	 * Klägern) zusätzliche Spalte(n) für die Widerklage anzeigen soll.
	 * 
	 * @param widerKlageAnzeigen true, wenn die Widerklage angezeigt werden sollen.
	 */
	public void setWiderKlageAnzeigen(boolean widerKlageAnzeigen) {
		if (this.widerKlageAnzeigen != widerKlageAnzeigen) {
			this.widerKlageAnzeigen = widerKlageAnzeigen;
			super.fireTableStructureChanged();
		}
	}

	/**
	 * Die Methode sagt der JTable, ob eine Zelle bearbeitet werden kann. Sind
	 * Gesamtschuldnerschaften vorhanden, ist die Antwort immer false. Geht
	 * ansonsten danach, ob es sich um einen Gesamtschuldner (dann immer false) oder
	 * einfachen BaumbachBeteiligten handelt und welche Zelle geändert wird. Dabei
	 * werden Plausibilitäten je nach Beteiligtentyp des TableModels berücksichtigt.
	 * 
	 * @param rowIndex    the row being queried
	 * @param columnIndex columnIndex the column being queried
	 * @return true oder (bisher nur) false
	 * 
	 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
	 */
	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		if (values.enthaeltGesamtschuldner())
			return false;
		switch (columnIndex) {
		case COLUMN_BETEILIGTER:
			return true;
		case COLUMN_STREITWERT:
			return getBeteiligtenTyp() != KLAEGER || values.isAnWiderklageBeteiligt(rowIndex);
		case COLUMN_UNTERLIEGEN:
			return getBeteiligtenTyp() != KLAEGER || values.isAnWiderklageBeteiligt(rowIndex);
		case COLUMN_WIDERKLAGE:
			return getBeteiligtenTyp() != DRITTWIDERBEKLAGTE;
		default:
			return false;
		}
	}

	/**
	 * Setzt den Wert der Zelle bei <code>rowIndex, columnIndex</code> auf
	 * <code>aValue</code>.
	 * 
	 * @todo TODO muss dann auch Gesamtschuldner berücksichtigen
	 * 
	 * @param aValue      der neue Wert für die Zelle
	 * @param rowIndex    the row whose value is to be changed
	 * @param columnIndex the column whose value is to be changed
	 * 
	 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int,
	 *      int)
	 */
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		if (!isCellEditable(rowIndex, columnIndex))
			return;
		boolean erfolgreich = false;		
		switch (columnIndex) {
		case COLUMN_BETEILIGTER:
			erfolgreich = setParteiBezeichnerAt(aValue, rowIndex);
			break;
		case COLUMN_STREITWERT:
			erfolgreich = setStreitwertAt(aValue, rowIndex);
			break;
		case COLUMN_UNTERLIEGEN:
			erfolgreich = setUnterliegenAt(aValue, rowIndex);
			break;
		case COLUMN_WIDERKLAGE:
			erfolgreich = setWiderklageAt(aValue, rowIndex);
			break;
		default:
			erfolgreich = false;
		}
		if (erfolgreich)
			super.fireTableRowsUpdated(rowIndex, rowIndex);
	}

	/**
	 * Die Methode dient dazu, den ganzen Tabelleninhalt zurückzugeben.
	 * 
	 * @return eine BaumbachBeteiligtenListe mit allen Werten der Tabelle
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#getAllValues()
	 */
	@Override
	public BaumbachBeteiligtenListe getAllValues() {
		return values;
	}

	/**
	 * Die Methode dient dazu, den ganzen Tabelleninhalt als Array
	 * BaumbachBeteiligter[] zurückzugeben.
	 * 
	 * @return den ganzen Tabelleninhalt als Array BaumbachBeteiligter[]
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#toArray()
	 */
	@Override
	public BaumbachBeteiligter[] toArray() {
		if (values != null) {
			BaumbachBeteiligter[] array = new BaumbachBeteiligter[values.size()];
			array = values.toArray(array);
			return array;
		} else
			return null;
	}

	/**
	 * Entfernt alle Werte aus der zugrundeliegenden BaumbachBeteiligtenListe oder
	 * legt eine solche an, falls noch nicht vorhanden.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#clear()
	 */
	@Override
	public void clear() {
		if (values == null)
			this.values = new BaumbachBeteiligtenListe();
		else
			values.clear();
		widerKlaegerIndex = -1;
		fireTableDataChanged();
	}

	/**
	 * Die Methode gibt den TooltipText für die angegebene Spalte zurück.
	 * 
	 * @param columnIndex die abgefragte Spalte
	 * @return den TooltipText als String
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.TooltipLieferant#getTooltipText(int)
	 */
	@Override
	public String getTooltipText(int columnIndex) {
		if (columnIndex < 0 || columnIndex > getColumnCount() - 1)
			return null;
		return COLUMN_TOOLTIPS[columnIndex];
	}

	/**
	 * Die Methode wird für dieses TableModel nicht benötigt und wirft deshalb eine
	 * Exception.
	 * 
	 * @param c ein Array mit BaumbachBeteiligten
	 * @throws UnsupportedOperationException bei jedem Aufruf, weil nicht
	 *                                       unterstützt.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.RowList#addAll(List<T>)
	 */
	@Override
	public void addAll(List<BaumbachBeteiligter> c) throws NullPointerException, UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * Die Methode ermittelt, ob der BaumbachBeteiligte bei rowIndex an einer
	 * Widerklage beteiligt ist.
	 * 
	 * @param rowIndex die Zeile
	 * @return
	 */
	private Boolean getWiderklageAt(int rowIndex) {
		if (values == null)
			return null;
		else
			return values.isAnWiderklageBeteiligt(rowIndex);
	}

	/**
	 * Die Methode getUnterliegen dient dazu, das Unterliegen für den
	 * BaumbachBeteiligten bei rowIndex zurückzugeben.
	 * 
	 * @param rowIndex die Zeile
	 * @return das Unterliegen dort
	 */
	private Euro getUnterliegenAt(int rowIndex) {// Double
		if (values == null)
			return null;
		else
			return values.getUnterliegenFor(rowIndex, false, false);// .doubleValue();
	}

	/**
	 * Die Methode dient zum Durchgriff auf die getStreitwert()-Methode des
	 * Beteiligten bei rowIndex
	 * 
	 * @param rowIndex
	 * @return
	 */
	private Euro getStreitwertAt(int rowIndex) {// Double
		if (values == null)
			return null;
		else
			return values.getStreitwertFor(rowIndex);// .doubleValue();
	}

	/**
	 * @return gibt den widerKlageAnzeigen als boolean zurück.
	 */
	boolean isWiderKlageAnzeigen() {
		return widerKlageAnzeigen;
	}

	/**
	 * Setzt die Widerklagebeteiligung für den Beteiligten in Zeile rowIndex. Prüft
	 * dabei, dass es bei Beklagten nur einen Widerkläger geben darf und dass der
	 * Wert bei Drittwiderbeklagten nicht geändert werden darf. Wird bei einem
	 * Kläger der Haken entfernt, werden auch Streitwert und Unterliegen wieder auf
	 * 0.0 gesetzt.
	 * 
	 * @param aValue   der neue Wert für die Zelle
	 * @param rowIndex the row whose value is to be changed
	 * @return true, wenn's geklappt hat.
	 */
	protected boolean setWiderklageAt(Object aValue, int rowIndex) {
		if (aValue != null && aValue instanceof Boolean && !values.isGesamtschuldnerschaft(rowIndex)
				&& beteiligtenTyp != DRITTWIDERBEKLAGTE) {
			BaumbachBeteiligter zwischen = values.get(rowIndex);
			boolean b = ((Boolean) aValue).booleanValue();
			if (!b && beteiligtenTyp == KLAEGER) {
				setValueAt(new Double(0), rowIndex, COLUMN_UNTERLIEGEN);
				setValueAt(new Double(0), rowIndex, COLUMN_STREITWERT);
			}
			if (beteiligtenTyp == BEKLAGTE) {
				setWiderklaegerIndex(rowIndex, b);
			}
			zwischen.setAnWiderklageBeteiligt(b);
			return true;
		} else
			return false;
	}

	/**
	 * Die Methode dient dazu, sicherzustellen, dass es nur einen Widerkläger gibt.
	 * Dazu aktualisiert es das Feld {@link widerKlaegerIndex} danach, ob der
	 * Beteiligte in der Zeile <code>rowIndex</code>
	 * <code>anWiderklageBeteiligt</code> ist oder nicht. Wenn ja, wird
	 * {@link widerKlaegerIndex} auf <code>rowIndex</code> gesetzt und alle anderen
	 * Beklagten auf anWiderklageBeteiligt = false. Wenn der fragliche Beklagte
	 * vorher der Widerkläger war, wird {@link widerKlaegerIndex} auf -1 gesetzt.
	 * 
	 * @param rowIndex              die Zeile
	 * @param anWiderklageBeteiligt ist der Beklagte dort Widerkläger?
	 */
	private void setWiderklaegerIndex(int rowIndex, boolean anWiderklageBeteiligt) {
		if (beteiligtenTyp != BEKLAGTE)
			return;
		/* Es kann nur einen Widerkläger geben! */
		if (anWiderklageBeteiligt) {
			widerKlaegerIndex = rowIndex;
			markierWiderklaeger();
		} else if (widerKlaegerIndex == rowIndex)
			widerKlaegerIndex = -1;
	}

	/**
	 * Die Methode dient zum Nachgucken, welcher Beklagte Widerkläger ist und
	 * widerKlaegerIndex entsprechend zu setzen. Wird beim Initialisieren des
	 * TableModels und beim Löschen von Zeilen gebraucht.
	 * 
	 */
	private void setWiderklaegerIndex() {
		if (beteiligtenTyp != BEKLAGTE) {
			widerKlaegerIndex = -1;
			return;
		}
		if (!values.enthaeltAnWiderklageBeteiligte())
			widerKlaegerIndex = -1;
		else
			for (int row = 0; row < getRowCount(); row++) {
				final BaumbachBeteiligter baumbachBeteiligter = values.get(row);
				if (baumbachBeteiligter.isAnWiderklageBeteiligt())
					widerKlaegerIndex = row;
			}
	}

	/**
	 * Die Methode dient dazu, sicherzustellen, dass es nur einen Widerkläger gibt.
	 * In allen Zeilen, die nicht <code>widerKlaegerIndex</code> sind, wird
	 * <code>anWiderklageBeteiligt</code> auf false gesetzt.
	 * 
	 */
	private void markierWiderklaeger() {
		if (beteiligtenTyp != BEKLAGTE)
			return;
		for (int row = 0; row < getRowCount(); row++) {
			boolean bisher = getWiderklageAt(row);
			boolean neu = bisher;
			final BaumbachBeteiligter baumbachBeteiligter = values.get(row);
			if (baumbachBeteiligter.getTyp() == BEKLAGTE && !baumbachBeteiligter.isGesamtschuldnerschaft()) {
				neu = (row == widerKlaegerIndex);
				baumbachBeteiligter.setAnWiderklageBeteiligt(neu);
			}
			if (bisher != neu)
				fireTableCellUpdated(row, COLUMN_WIDERKLAGE);
		}
	}

	/**
	 * Setzt das Unterliegen für den Beteiligten in Zeile rowIndex. Prüft dabei,
	 * dass das Unterliegen nicht größer als der Streitwert wird.
	 * 
	 * @param aValue   das Unterliegen als Euro
	 * @param rowIndex die fragliche Zeile
	 * @return true, wenn's geklappt hat
	 */
	protected boolean setUnterliegenAt(Object aValue, int rowIndex) {
		if (aValue != null && aValue instanceof Euro && !values.isGesamtschuldnerschaft(rowIndex)) {
			BaumbachBeteiligter zwischen = values.get(rowIndex);
//			double d = ((Double) aValue).doubleValue();
			Euro e = (Euro) aValue;
//			if (d > getStreitwertAt(rowIndex))
			if (e.greaterThan(getStreitwertAt(rowIndex)))
				return false;
			else
				zwischen.setErfolg(e);
//				zwischen.setErfolg(Euro.ofEuros(d));
			return true;
		} else
			return false;
	}

	/**
	 * Setzt den Streitwert für den Beteiligten in Zeile rowIndex. Prüft dabei, dass
	 * das Unterliegen nicht größer als der Streitwert wird.
	 * 
	 * @param aValue   der Streitwert als Double
	 * @param rowIndex die fragliche Zeile
	 * @return true, wenn's geklappt hat
	 */
	protected boolean setStreitwertAt(Object aValue, int rowIndex) {
		if (aValue != null && aValue instanceof Euro && !values.isGesamtschuldnerschaft(rowIndex)) {
			BaumbachBeteiligter zwischen = values.get(rowIndex);
//			double d = ((Double) aValue).doubleValue();
			final Euro e = (Euro) aValue;
//			if (getUnterliegenAt(rowIndex) > d)
			if (getUnterliegenAt(rowIndex).greaterThan(e))
				return false;
			else
				zwischen.setAntrag(e);
//				zwischen.setAntrag(Euro.ofEuros(d));
			return true;
		} else
			return false;
	}

	/**
	 * Setzt den GenusNumerus für den Beteiligten in Zeile rowIndex. Liest dazu den
	 * GenusNumerus aus dem übergebenen Beteiligten aus.
	 * 
	 * @param aValue   der Streitwert als Double
	 * @param rowIndex die fragliche Zeile
	 * @return true, wenn's geklappt hat
	 */
	protected boolean setParteiBezeichnerAt(Object aValue, int rowIndex) {
		BaumbachBeteiligter zwischen = values.get(rowIndex);
		if (aValue != null && aValue instanceof Beteiligter && !zwischen.isGesamtschuldnerschaft()) {
			Beteiligter value = (Beteiligter) aValue;
			values.get(rowIndex).setGenusNumerus(value.getGenusNumerus());
			return true;
		} else
			return false;
	}

}
