/**
 * StreitwertTableModel.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.model.gebuehren;

import java.util.ArrayList;

import javax.swing.table.AbstractTableModel;

import eu.gronos.kostenrechner.data.gebuehren.GebuehrenGesetzesSammlung;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.interfaces.TooltipLieferant;

/**
 * Unveränderliches TableModel, das Streitwerte auflistet
 *
 * @author Peter Schuster (setrok)
 * @date 17.08.2014
 *
 */
public class StreitwertTableModel extends AbstractTableModel implements TooltipLieferant {

    private static final long serialVersionUID = -3648963976826657812L;
    /**
     * Der int mit dem Namen COLUMN_COUNT speichert die Anzahl der Spalten:
     * "Streitwert bis \t| GKG 1,0 \t| RVG 1,0"
     */
    private static final int COLUMN_COUNT = 3;
    /**
     * Der String[] mit dem Namen COLUMN_NAMES speichert die Spaltenüberschriften
     */
    private static final String[] COLUMN_NAMES = { "Streitwert bis", "GKG 1,0",
            "RVG 1,0" };
    private static final String[] COLUMN_TOOLTIPS = { "Streitwert bis zu",
            "Höhe einer 1,0 Gebühr nach § 34 GKG",
            "Höhe einer 1,0 Gebühr nach § 13 RVG" };
    /**
     * Der Class<?>[] mit dem Namen COLUMN_CLASS speichert die Klassen der Spalten.
     * Die erste Spalte ist Long mit den Streitwerten, die anderen beiden Double mit
     * der 1,0 Gebührenhöhe
     */
    private static final Class<?>[] COLUMN_CLASS = { Long.class, Double.class,
            Double.class };
    /**
     * Der ArrayList&lt;Euro&gt; mit dem Namen values speichert die Streitwerte des
     * TableModels
     */
    private final ArrayList<Euro> values = GebuehrenGesetzesSammlung.getRvg()
            .errechneStreitwertListe();

    /**
     * Der Konstruktor muss nichts tun, TODO oder?
     * 
     */
    public StreitwertTableModel() {
        super();
    }

    /**
     * Die Methode gibt die Länge der Streitwertliste values als Zeilenanzahl zurück
     * 
     * @return die Länge der Streitwertliste values als int
     * 
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
    public int getRowCount() {
        return values.size();
    }

    /**
     * Returns the number of columns in the model. A JTable uses this method to
     * determine how many columns it should create and display by default.
     * 
     * @return the number of columns in the model
     * 
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
    public int getColumnCount() {
        return COLUMN_COUNT;
    }

    /**
     * Returns the name of the column at columnIndex. This is used to initialize the
     * table's column header name. Note: this name does not need to be unique; two
     * columns in a table can have the same name.
     * 
     * @param column die Spalte
     * @return die Überschrift der Spalte
     * 
     * @see javax.swing.table.AbstractTableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int column) {
        return COLUMN_NAMES[column];
    }

    /**
     * Gibt die genaueste Oberklasse für den Zellinhalt in der Spalte zurück. JTable
     * nutzt den Wert, um einen DefaultRenderer und einen Editor für die Spalte zu
     * initialisieren.
     * 
     * @param columnIndex die Spalte
     * @return die genaueste Oberklasse für den zurückgegebenen Wert der Spalte
     * 
     * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
        return COLUMN_CLASS[columnIndex];
    }

    /**
     * Die Methode gibt den Wert für Zeile <code>rowIndex</code>, Spalte
     * <code>columnIndex</code> zurück
     * 
     * @param rowIndex    die Zeile
     * @param columnIndex die Spalte
     * @return den Inhalt dort
     * 
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        switch (columnIndex) {
        case 0:
            return values.get(rowIndex);
        case 1:
            return GebuehrenGesetzesSammlung.getGkg()
                    .errechneGebuehr(values.get(rowIndex));
        case 2:
            return GebuehrenGesetzesSammlung.getRvg()
                    .errechneGebuehr(values.get(rowIndex));
        default:
            return null;
        }
    }

    /**
     * Die Methode ermittelt die Zeile, die für einen <code>streitwert</code>
     * einschlägig ist.
     * 
     * @param streitwert der Streitwert, für den die passende Zeile gefunden werden
     *                   soll.
     * @return den {@link #getValueAt(int, int) rowIndex} des Eintrags dafür.
     */
    public int rowAtStreitwert(Euro streitwert) {// Double
        /* Wenn die values leer sind: -1 */
        if (values == null || values.size() < 1)
            return -1;
        /* Wenn der Wert exakt enthalten ist: index davon zurückgeben. */
        if (values.contains(streitwert))
            return values.indexOf(streitwert);
        /*
         * Sonst die values durchgehen, bis ein Eintrag größer ist als der übergebene
         * Long. Wenn einer gefunden wird: dessen index zurückgeben.
         */
        for (Euro grenze : values) {// Long
            if (grenze.compareTo(streitwert)>=0)// grenze.longValue() streitwert.longValue() grenze >= streitwert
                return values.indexOf(grenze);
        }
        /*
         * Wenn kein größerer Eintrag gefunden ist, liegt der Streitwert über dem
         * Maximum (hui!). Dann der index der letzten Zeile zurück geben.
         */
        return values.size() - 1;
    }

    /**
     * Die Methode gibt den TooltipText für die angegebene Spalte zurück.
     * 
     * @param columnIndex die abgefragte Spalte
     * @return den TooltipText als String
     * 
     * @see eu.gronos.kostenrechner.interfaces.TooltipLieferant#getTooltipText(int)
     */
    @Override
    public String getTooltipText(int columnIndex) {
        return COLUMN_TOOLTIPS[columnIndex];
    }

}
