/**
 * TenorierendAdapter.java
 * eu.gronos.kostenrechner.controller (Kostenrechner)
 */
package eu.gronos.kostenrechner.util;

import eu.gronos.kostenrechner.data.tenordaten.BegruendungsElemente;
import eu.gronos.kostenrechner.data.tenordaten.CaretRange;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.interfaces.Tabulierend;
import eu.gronos.kostenrechner.interfaces.Tenorierend;
import eu.gronos.kostenrechner.view.result.TenorDialog;

/**
 * Die Klasse dient als Adapter, damit {@link TenorDialog} statt eines
 * {@link Tenorierend}, {@link Begruendend} und {@link Tabulierend} einfach ein
 * {@link TenorDatenContainer} erhalten kann.
 * 
 * Es enthält die Methoden, die {@link TenorDialog} braucht:
 * 
 * {@link #erzeugeTenor()} verkettet die einzelnen Entscheidungstexte aus dem
 * {@link TenorDatenContainer} zu einem einzigen {@link String}.
 * 
 * {@link #getGruende()} liefert die Begründung aus dem
 * {@link TenorDatenContainer} zurück.
 * 
 * Außerdem enthält es die Methoden aus {@link Tabulierend}, etwa
 * {@link #getTableCellValues()}.
 *
 * @author Peter Schuster (setrok)
 * @date 20.10.2018
 *
 */
public class TenorierendAdapter implements Tenorierend, /*Begruendend,*/ Tabulierend {
	private final TenorDatenContainer container;
	private final BegruendungsZahlenTabelle zeilen;

	/**
	 * Der Konstruktor braucht den {@link TenorDatenContainer}, um arbeiten zu
	 * können.
	 * 
	 * @param container ein befüllter {@link TenorDatenContainer}
	 */
	public TenorierendAdapter(TenorDatenContainer container) {
		this.container = container;
		BegruendungsElemente begruendung = container.begruendung;
		this.zeilen = new BegruendungsZahlenTabelle(begruendung.zeilenZahlen, begruendung.columnHeaders, begruendung.rowHeaders,
				begruendung.range);
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getTableCellValues()
	 */
	@Override
	public Object[][] getTableCellValues() {
		return zeilen.toTableCellValues();
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getColumnHeaders()
	 */
	@Override
	public String[] getColumnHeaders() {
		return zeilen.toColumnHeaders();
	}

	/**
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getRange()
	 */
	@Override
	public CaretRange getRange() {
		return zeilen.getRange();
	}
	// int[]

	/**
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getColumnClasses()
	 */
	@Override
	public Class<?>[] getColumnClasses() {
		return zeilen.toColumnClasses();
	}

	/**
	 * Die Methode dient dazu, den Begründungstext aus dem
	 * {@link TenorDatenContainer} zurückzugeben.
	 * 
	 * @return {@link TenorDatenContainer#begruendung#text}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tenorierend#getGruende()
	 */
	@Override
	public String getGruende() {
		return container.begruendung.text;
	}

	/**
	 * Die Methode baut einen vollständigen Tenor aus den einzelnen Unterklassen des
	 * {@link TenorDatenContainer}s zusammen, geordnet nach Hauptsacheentscheidung,
	 * Kostenentscheidung, Vollstreckbarkeitsentscheidung, Streitwertentscheidung,
	 * und ggfs. sonstige Entscheidung.
	 * 
	 * @return den tenor als {@link String}
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tenorierend#erzeugeTenor()
	 */
	@Override
	public String erzeugeTenor() {
		String tenor = container.hauptsacheEntscheidung.text + container.kostenEntscheidung.text
				+ container.vollstreckbarkeitsEntscheidung.text + container.streitwertEntscheidung.text
				+ container.sonstigeEntscheidung.text;
		return tenor;
	}

	/**
	 * Die Methode soll eine Beschreibung für die Art des Tenor zurückgeben, damit
	 * andere Klassen etwa das Dokument danach benennen können.
	 * 
	 * @return
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tenorierend#getTenorBeschreibung()
	 */
	@Override
	public String getTenorBeschreibung() {
		return container.berechnungsTyp;
	}

	/**
	 * @return gibt {@link #tableWeight} als {@link int} zurück, also die Summe der
	 *         Zeichen in der gesamten Tabelle.
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getTableWeight()
	 */
	@Override
	public int getTableWeight() {
		if (zeilen.getTableWeight() < 0) {
			zeilen.calculateTableWeight();
		}
		return zeilen.getTableWeight();
	}

	/**
	 * 
	 * @param column index der Spalte
	 * @return die Summe der Zeichen in allen Zellen der gesamten mit
	 *         <code>column</code> genannten Spalte
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Tabulierend#getColumnWeight(int)
	 */
	@Override
	public int getColumnWeight(int column) {
		if (zeilen.getColumnWeight(column) < 0) {
			zeilen.calculateTableWeight();
		}
		return zeilen.getColumnWeight(column);
	}

}
