/**
 * 
 */
package eu.gronos.kostenrechner.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.data.tenordaten.Fraction;
import eu.gronos.kostenrechner.interfaces.HtmlRtfFormattierend;

/**
 * Die Klasse speichert die einzelnen Verluste in einer {@link Map} zu den
 * Beteiligten bzw. Beteiligten-Mehrheiten.
 *
 * @author Peter Schuster (setrok)
 * @date 12.10.2021
 *
 */
public class VerlusteBank {

	private final Map<List<Beteiligter>, Euro> konten = new HashMap<>();
	private Euro fiktiverStreitwert = Euro.ZERO_CENTS;

	/**
	 * Die Methode fügt für das Verlustkonto der <code>beteiligte</code>n den
	 * Verlust (in Cent) hinzu und addiert den Verlust zudem dem
	 * {@link #getFiktiverStreitwertCent()} hinzu.
	 * 
	 * @param beteiligte eine {@link List} der {@link Beteiligter}n
	 * @param verlust    den Verlust als {@link Euro}
	 * @return den vorherigen Verlust der {@code beteiligte}n, also den vorherigen
	 *         Wert in der {@link Map} als {@link Euro}
	 * 
	 * @see java.util.Map#put(Object, Object)
	 */
	public HtmlRtfFormattierend add(List<Beteiligter> beteiligte, Euro verlust) {
		Euro kontostand = konten.get(beteiligte);
		fiktiverStreitwert = fiktiverStreitwert.add(verlust);
		if (kontostand == null) {
			// Neuen Eintrag erstellen
			return konten.put(beteiligte, verlust);
		} else {
			// sonst dem bestehenden Verlust hinzuaddieren
			return konten.put(beteiligte, kontostand.add(verlust));
		}
	}

	/**
	 * 
	 * @param beteiligte eine {@link List} der {@link Beteiligter}n
	 * @return den Verlust zu diesen der {@link Beteiligter}n
	 * 
	 * @see java.util.Map#get(Object)
	 */
	public Euro get(List<Beteiligter> beteiligte) {
		return konten.get(beteiligte);
	}

	/**
	 * Die Methode errechnet für ein bestimmtes Konto die Verlustquote (als Wert
	 * zwischen 0.0 und 1.0) im Verhältnis zum {@link #getFiktiverStreitwertCent()}.
	 * 
	 * @param beteiligte eine {@link List} der {@link Beteiligter}n
	 * @return die Verlustquote für das Konto als {@code double}
	 */
	public double getQuotaFor(List<Beteiligter> beteiligte) {
		if (konten.get(beteiligte) == null || getFiktiverStreitwert().getCents() < 1L)
			return 0.0;
		return getFractionFor(beteiligte).doubleValue();
	}

	/**
	 * Die Methode errechnet für ein bestimmtes Konto die Verlustquote als
	 * {@link Fraction}-Bruch im Verhältnis zum
	 * {@link #getFiktiverStreitwertCent()}.
	 * 
	 * @param beteiligte eine {@link List} der {@link Beteiligter}n
	 * @return die Verlustquote für das Konto als {@link Fraction}
	 */
	public Fraction getFractionFor(List<Beteiligter> beteiligte) {
		if (konten.get(beteiligte) == null || getFiktiverStreitwert().getCents() < 1L)
			return Fraction.ZERO;
		return konten.get(beteiligte).divideAsFraction(getFiktiverStreitwert());
	}

	/**
	 * @return das {@link Map#keySet()} als {@link Set} aus {@link List} aus
	 *         {@link Beteiligter}n
	 * 
	 * @see java.util.Map#keySet()
	 */
	public Set<List<Beteiligter>> keySet() {
		return konten.keySet();
	}

	/**
	 * @return gibt {@link #fiktiverStreitwertCent} als {@link Euro} zurück.
	 */
	public Euro getFiktiverStreitwert() {
		return this.fiktiverStreitwert;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + (int) (fiktiverStreitwert.getCents() ^ (fiktiverStreitwert.getCents() >>> 32));
		result = prime * result + ((konten == null) ? 0 : konten.hashCode());
		return result;
	}

	/**
	 * Die Methode vergleicht zwei {@link VerlusteBank}en
	 * 
	 * sind zwei VerlusteBank wohl dann gleich, wenn 1) jeder Eintrag im keySet –
	 * Beteiligter / Beteiligtenmehrheit – auch im keySet des obj enthalten ist und
	 * 2) die Beteiligten(mehrheiten) auch die gleichen Verlustsummen haben. Das
	 * scheint {@link Map} zu prüfen.
	 * 
	 * @param obj das zu vergleichende Objekt
	 * @return <code>true</code>, wenn (neben dem ganzen <code>null</code>- und
	 *         Klassengedöns) die {@link #konten} und
	 *         {@link #getFiktiverStreitwertCent()} gleich sind, sonst false
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!(obj instanceof VerlusteBank)) {
			return false;
		}
		VerlusteBank other = (VerlusteBank) obj;
		if (fiktiverStreitwert.getCents() != other.fiktiverStreitwert.getCents()) {
			return false;
		}
		// Für den Vergleich kommt es vor allem darauf an, dass die #konten
		// übereinstimmen. Das darf die Map-Implementierung prüfen.
		if (konten == null) {
			if (other.konten != null) {
				return false;
			}
		} else if (!konten.equals(other.konten)) {
			return false;
		}
		return true;
	}

	/**
	 * Die Methode soll am Schluss alle Konten löschen, die Verluste von 0L haben.
	 * 
	 * 
	 * Konten mit Verlusten von 0L entstehen, wenn dem Angreifer erst Verluste
	 * aufgebürdet werden, aber im weiteren Vorgehen durch Addieren negativer
	 * Verluste wieder neutralisiert werden.
	 */
	public void removeBlanks() {
		Set<List<Beteiligter>> keys = new HashSet<>();
		for (List<Beteiligter> key : konten.keySet()) {
			if (konten.get(key) == null || konten.get(key).getCents() == 0L) {
				keys.add(key);
			}
		}
		for (List<Beteiligter> key : keys) {
			konten.remove(key);
		}
	}

}
