package eu.gronos.kostenrechner.util.baumbach;

import java.util.function.Predicate;

import eu.gronos.kostenrechner.data.baumbach.Angriff;
import eu.gronos.kostenrechner.data.baumbach.GerichtsKostenBeteiligter;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;

/**
 * Die VerlusteFilter stellt ein {@link Predicate} zur Verfügung, der guckt, ob
 * der übergebene {@link Beteiligter} am {@link Angriff} beteiligt ist. Bei
 * einem {@link GerichtsKostenBeteiligter}n wird nicht gefiltert (alles
 * akzeptiert)
 *
 * @author Peter Schuster (setrok)
 * @date 12.10.2021
 *
 */
public  class AngriffeFilter implements Predicate<Angriff> {

    private final Beteiligter beteiligter;

    public AngriffeFilter(Beteiligter beteiligter) {
        this.beteiligter = beteiligter;
    }

    /**
     * Die Methode lässt nur Angriffe durch, an denen der dem Konstruktor übergebene
     * {@link #beteiligter} beteiligt ist. Handelt es sich dabei um einen
     * {@link GerichtsKostenBeteiligter}, werden alle Angriffe
     * akzeptiert/durchgelassen (keiner herausgefiltert). Denn für Gerichtskosten
     * werden alle Parteien betrachtet.
     * 
     * @param angriff {@link Angriff}
     * @return <code>true</code>, wenn {@link Angriff#getAngreifer()} oder
     *         {@link Angriff#getGegner()} den {@link #beteiligter} enthält, oder
     *         unabhängig vom {@link Angriff}, wenn {@link #beteiligter} vom Typ
     *         {@link GerichtsKostenBeteiligter} ist.
     * 
     * @see java.util.function.Predicate#test(java.lang.Object)
     */
    @Override
    public boolean test(Angriff angriff) {
        return this.beteiligter instanceof GerichtsKostenBeteiligter
                || angriff.getAngreifer().contains(beteiligter)
                || angriff.getGegner().contains(beteiligter);
    }

}