/**
 * TenorXmlDatei.java
 * eu.gronos.kostenrechner.controller.files (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.files;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringWriter;
import java.nio.file.Files;
import java.nio.file.Path;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.PropertyException;
import javax.xml.bind.Unmarshaller;

import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.files.RtfWerkzeugKasten;
import eu.gronos.kostenrechner.data.tenordaten.TenorDatenContainer;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;

/**
 * Klasse zum Speichern des {@link TenorDatenContainer}s als XML.
 *
 * @author Peter Schuster (setrok)
 * @date 1 Jan 2019
 *
 */
public class TenorXmlDatei extends BerechnungXmlDatei {
	private static final String FEHLER_BEIM_KONVERTIEREN_IN_XML = "Programmzustände konnten nicht in XML konvertiert werden: ";
	private static final String SCHEMA_LOCATION = "https://www.kostentenor.de schema-tenordaten.xsd";

	@Override
	public String toXmlString(VerfahrensDatenContainer container) {
		if (!(container instanceof TenorDatenContainer))
			return super.toXmlString(container);

		StringWriter sw = new StringWriter();
		try {
			/* für den XML-String sollen keine Zeilenumbrüche erstellt werden */
			Marshaller marshaller = createMarshaller(false, getSchemaLocation());
			marshaller.marshal(container, sw);
		} catch (JAXBException e) {
			e.printStackTrace();
			return FEHLER_BEIM_KONVERTIEREN_IN_XML + e.getLocalizedMessage();
		}
		return sw.toString();
	}

	@Override
	public void speicherXml(VerfahrensDatenContainer container, Path datei) throws JAXBException, IOException {
		if (!(container instanceof TenorDatenContainer))
			super.speicherXml(container, datei);

		Marshaller marshaller = createMarshaller(true, getSchemaLocation());
//		marshaller.marshal(container, datei);
		try (OutputStream out = Files.newOutputStream(datei)) {
			marshaller.marshal(container, out);
		}
	}

	@Override
	public VerfahrensDatenContainer ladeXml(Path datei) {
		VerfahrensDatenContainer container = null;
		try {
			Kostenrechner.getLogger().info("TenorXmlDatei#ladeXml");
			container = unmarshalXml(datei);
			if (container == null || //
					(((TenorDatenContainer) container).hauptsacheEntscheidung.isEmpty()
							|| ((TenorDatenContainer) container).kostenEntscheidung.isEmpty()
							|| ((TenorDatenContainer) container).vollstreckbarkeitsEntscheidung.isEmpty()) && //
							((TenorDatenContainer) container).sonstigeEntscheidung.isEmpty()) {
				throw new NullPointerException("Kein TenorDatenContainer oder veraltetes Format!");
			}
			Kostenrechner.getLogger()
					.info(String.format("Geladen als tdc: %s (%s)", datei.toAbsolutePath(), container.getClass()));
		} catch (Exception error) {
			error.printStackTrace();
			// dann doch VDC. Funktioniert nur, wenn unmarshalXml(File) private ist.
			return super.ladeXml(datei);
		}
		if (container != null) {
			container.streitgenossen = fixGesamtschuldnerschaften(container.streitgenossen);
		}
		return container;
	}

	@Override
	public String getDateiendung() {
		return DATEIENDUNG;
	}

	/**
	 * 
	 * @see eu.gronos.kostenrechner.util.files.XmlDatei#getSchemaLocation()
	 */
	@Override
	public String getSchemaLocation() {
		return SCHEMA_LOCATION;
	}

	/**
	 * Die Methode erstellt die benötigte Instanz je nach dem, ob es sich um eine
	 * Tenor-XML-Datei handelt oder eine normale Verfahrensdaten-XML. Das
	 * entscheidet {@link #isTenorXmlEndung(File)} anhand der Endung.
	 * 
	 * @param datei das {@link File}, dessen Endung geprüft wird
	 * @return eine Instanz von {@link BerechnungXmlDatei} oder
	 *         {@link TenorXmlDatei}
	 */
	public static BerechnungXmlDatei createInstanceFor(Path datei) {
		if (isTenorXmlEndung(datei)) {
			return new TenorXmlDatei();
		} else {
			return new BerechnungXmlDatei();
		}
	}

	/**
	 * Die Methode dient zum Umwandeln aus XML in einen {@link TenorDatenContainer}.
	 * 
	 * Die Methode muss <code>private</code> sein, damit
	 * {@link BerechnungXmlDatei#ladeXml(File)} sie nicht versehentlich nimmt. Daher
	 * muss man vor dem Aufruf dieser Methode prüfen, ob die Datei eine
	 * {@link TenorXmlDatei} oder eine {@link BerechnungXmlDatei} ist!
	 * 
	 * @param datei ein {@link File}, das zu laden ist
	 * @return ein {@link TenorDatenContainer} mit allen gespeicherten Zuständen
	 * @throws JAXBException Exception that represents a failure in a JAXB
	 *                       operation.
	 * @throws IOException 
	 */
	private TenorDatenContainer unmarshalXml(Path datei) throws JAXBException, IOException {
//		return JAXB.unmarshal(datei, TenorDatenContainer.class); throws JAXBException, IOException {
		JAXBContext context = JAXBContext.newInstance(TenorDatenContainer.class);
		Unmarshaller u = context.createUnmarshaller();
		try (InputStream in = Files.newInputStream(datei)) {
			return (TenorDatenContainer) u.unmarshal(in);
		}
	}

	/**
	 * Die Methode erstellt einen {@link Marshaller}, der dann zum Speichern als
	 * XML-Datei benötigt wird.
	 * 
	 * @param formattedOutput schaltet, ob der Marshaller Zeilenumbrüche und
	 *                        Einrückungen bauen soll
	 *                        {@link Marshaller#JAXB_FORMATTED_OUTPUT}
	 * @param schemaLocation  die {@link #SCHEMA_LOCATION}
	 * @return einen {@link Marshaller}
	 * @throws JAXBException
	 * @throws PropertyException
	 */
	private static Marshaller createMarshaller(boolean formattedOutput, String schemaLocation)
			throws JAXBException, PropertyException {
		JAXBContext context = JAXBContext.newInstance(TenorDatenContainer.class);
		Marshaller marshaller = context.createMarshaller();
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, new Boolean(formattedOutput));
		marshaller.setProperty(Marshaller.JAXB_SCHEMA_LOCATION, schemaLocation);
		return marshaller;
	}

	/**
	 * Die Methode findet heraus, ob es sich um eine Tenor-XML-Datei handelt.
	 * 
	 * @param datei das {@link File}, dessen Endung geprüft wird
	 * @return <code>true</code>, wenn der Name auf .rtf.skktx endet.
	 */
	private static boolean isTenorXmlEndung(Path datei) {
		return datei != null && datei.getFileName().toString().toLowerCase()
				.endsWith(RtfWerkzeugKasten.DATEI_FILTER.ganzeEndung() + DATEI_FILTER.ganzeEndung());
	}
}
