/*
 * ForderungsStaffelungPruefer.java
 * eu.gronos.kostenrechner.util.forderungen (Kostenrechner)
 */
package eu.gronos.kostenrechner.util.forderungen;

import java.util.List;

import eu.gronos.kostenrechner.data.forderungen.AufrechnungsForderung;
import eu.gronos.kostenrechner.data.forderungen.Forderung;
import eu.gronos.kostenrechner.data.forderungen.ForderungsStaffelung;
import eu.gronos.kostenrechner.data.forderungen.KlageForderung;
import eu.gronos.kostenrechner.interfaces.Pruefend;
import eu.gronos.kostenrechner.logic.TenorTexter;

/**
 * Der {@link Pruefend} kontrolliert alle Bestandteile einer
 * {@link ForderungsStaffelung}
 *
 * @author Peter Schuster (setrok)
 * @date 16.07.2020
 *
 */
public class ForderungsStaffelungPruefer implements Pruefend<ForderungsStaffelung> {

	/**
	 * Die Methode prüft eine {@link ForderungsStaffelung}
	 * 
	 * @param eingabe die zu prüfende {@link ForderungsStaffelung}
	 * @return <code>true</code>, wenn alle Plausis eingehalten werden
	 * @throws IllegalArgumentException
	 * @throws NumberFormatException
	 * @throws NullPointerException     TODO
	 * 
	 * @see eu.gronos.kostenrechner.interfaces.Pruefend#pruefeEingabe(java.lang.Object)
	 */
	@Override
	public boolean pruefeEingabe(ForderungsStaffelung eingabe)
			throws IllegalArgumentException, NumberFormatException, NullPointerException {
		if (eingabe.forderungen == null || eingabe.forderungen.size() < 1)
			throw new IllegalArgumentException(TenorTexter.FEHLER_FORDERUNG_LEER);
		if (!isReihenfolgeEingehalten(eingabe.forderungen))
			throw new IllegalArgumentException(TenorTexter.FEHLER_REIHENFOLGE_FORDERUNGEN);

		/* Jetzt auch noch die einzelnen Forderungen prüfen */
		final ForderungPruefer pruefer = new ForderungPruefer();
		for (Forderung forderung : eingabe.forderungen) {
			pruefer.pruefeEingabe(forderung);
		}

		return true;
	}

	/**
	 * Die Methode dient dazu, die Reihenfolge der Forderungen zu prüfen!
	 * 
	 * @param forderungen die zu prüfende ArrayList<Forderung>
	 * @return true, wenn die Reihenfolge eingehalten ist: Erste muss Klageforderung
	 *         sein, dann alle übrigen Klageforderungen, dann Aufrechnungen (und
	 *         keine Klageforderungen mehr)
	 */
	private boolean isReihenfolgeEingehalten(List<Forderung> forderungen) {
		boolean antwort = true;
		boolean umstiegAufAufrechnung = false;
		if (!(forderungen.get(0) instanceof KlageForderung))
			antwort = false;
		for (Forderung forderung : forderungen) {
			if (forderung instanceof AufrechnungsForderung)
				umstiegAufAufrechnung = true;
			if (umstiegAufAufrechnung && forderung instanceof KlageForderung)
				antwort = false;
		}
		return antwort;
	}
}
