/**
 * EuroCellEditor.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.Component;
import java.text.ParseException;

import javax.swing.DefaultCellEditor;
import javax.swing.JFormattedTextField;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.text.NumberFormatter;

import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.tenordaten.Euro;

/**
 * Ein Ableger von DefaultCellEditor für Double-Werte, angelehnt an
 * IntegerEditor aus den Java-Tutorials, Copyright (c) 1995, 2008, Oracle and/or
 * its affiliates. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * - Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * 
 * - Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * - Neither the name of Oracle or the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
public class EuroEditor extends DefaultCellEditor {

	private static final long serialVersionUID = -7511642419428646152L;

	/**
	 * Initialisiert einen DefaultCellEditor mit einem
	 * {@link javax.swing.JFormattedTextField JFormattedTextField} als editor, der
	 * über den übergebenen {@link javax.swing.text.NumberFormatter NumberFormatter
	 * formatter} die Eingaben validiert.
	 * 
	 * @param formatter ein NumberFormatter
	 * 
	 * @see javax.swing.DefaultCellEditor#DefaultCellEditor(JTextField)
	 */
	public EuroEditor(NumberFormatter formatter) {
		super(new JFormattedTextField(formatter));
		JFormattedTextField ftf = (JFormattedTextField) getComponent();
		ftf.setHorizontalAlignment(JTextField.TRAILING);
	}

	/**
	 * Sets an initial value for the editor. This will cause the editor to
	 * stopEditing and lose any partially edited value if the editor is editing when
	 * this method is called. Returns the component that should be added to the
	 * client's Component hierarchy. Once installed in the client's hierarchy this
	 * component will then be able to draw and receive user input.
	 * 
	 * Zusätzlich markiert er den Inhalt.
	 * 
	 * @param table      table - the JTable that is asking the editor to edit; can
	 *                   be null
	 * @param value      the value of the cell to be edited; it is up to the
	 *                   specific editor to interpret and draw the value. For
	 *                   example, if value is the string "true", it could be
	 *                   rendered as a string or it could be rendered as a check box
	 *                   that is checked. null is a valid value
	 * @param isSelected true if the cell is to be rendered with highlighting
	 * @param row        the row of the cell being edited
	 * @param column     the column of the cell being edited
	 * @return the component for editing
	 * 
	 * @url "http://stackoverflow.com/questions/19030261/jtextfield-selectall-works-strangely"
	 *      JTextField.selectAll() works strangely
	 * 
	 * @see javax.swing.DefaultCellEditor#getTableCellEditorComponent(javax.swing.JTable,
	 *      java.lang.Object, boolean, int, int)
	 */
	@Override
	public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) {
		final JFormattedTextField ftf = (JFormattedTextField) super.getTableCellEditorComponent(table, value,
				isSelected, row, column);
		String textValue = value.toString();
		if (value instanceof String)
			textValue = (String) value;
		else if (value instanceof Double)
			textValue = Double.toString((Double) value);
		ftf.setText(textValue);
		ftf.selectAll();
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				ftf.setCaretPosition(0);
				ftf.selectAll();
				ftf.requestFocus();
			}
		});
		return ftf;
	}

	/**
	 * Forwards the message from the CellEditor to the delegate. Calls
	 * fireEditingStopped and returns true.
	 * 
	 * @return true, sofern es kein falsches Eingabeformat gibt
	 * 
	 * @see javax.swing.DefaultCellEditor#stopCellEditing()
	 */
	@Override
	public boolean stopCellEditing() {
		JFormattedTextField ftf = (JFormattedTextField) getComponent();
		if (ftf.isEditValid())
			try {
				ftf.commitEdit();
				return super.stopCellEditing();
			} catch (ParseException e) {
				FehlerHelper.zeigeFehler("Falsches Eingabeformat! " + e.getLocalizedMessage(), e);
				e.printStackTrace();
				return false;
			}
		else
			return false;
	}

	/**
	 * Stellt sicher, dass der Rückgabewert Double ist
	 * 
	 * @return den Inhalt des JFormattedTextField als Double oder null, sofern er
	 *         sich par tout nicht zum Double konvertieren lässt.
	 * 
	 * @see javax.swing.DefaultCellEditor#getCellEditorValue()
	 */
	@Override
	public Object getCellEditorValue() {
		JFormattedTextField ftf = (JFormattedTextField) getComponent();
		Object value = ftf.getValue();
		System.out.println(value);
		if (value instanceof Double)
			return Euro.ofEuros((double) value);
		else if (value instanceof Number)
			return Euro.ofEuros(((Number) value).doubleValue());// new Double
		else
			return null;
	}

}
