/*
 * KostenJPanel.java
 * eu.gronos.kostenrechner.view (Kostenrechner)
 */
package eu.gronos.kostenrechner.view;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.text.NumberFormat;

import javax.swing.AbstractAction;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.text.NumberFormatter;

import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.view.BeschriebenesPanel;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.EintragEntfernenAktion;
import eu.gronos.kostenrechner.controller.NumberExtractor;
import eu.gronos.kostenrechner.controller.TabulatorBeforeAction;
import eu.gronos.kostenrechner.controller.TabulatorNextAction;
import eu.gronos.kostenrechner.controller.gebuehren.GebuehrenTransferHandler;
import eu.gronos.kostenrechner.view.gebuehren.GebuehrenTableRenderer;
import eu.gronos.kostenrechner.view.gebuehren.TooltipHeaderRenderer;

/**
 * Abstrakte Oberklasse, von denen die Klassen für die Registerkarten abgeleitet
 * werden können.
 *
 * @author Peter Schuster (setrok)
 * @date 13.06.2018
 *
 */
public abstract class KostenJPanel extends BeschriebenesPanel {

	private static final String QUIT_EDIT_MODE_ACTION_NAME = "quitEditModeAction";
	private static final String EDIT_MODE_ACTION_NAME = "editModeAction";
	private static final String TAB_BEFORE_NAME = "tabBefore";
	private static final String TAB_NEXT_NAME = "tabNext";

	/**
	 * When JTable loses focus, quit the "edit mode", On VK_ESCAPE, quit the "edit
	 * mode"
	 *
	 * @author Peter Schuster (setrok)
	 * @date 06.01.2021
	 *
	 */
	public final class QuitEditModeAction extends AbstractAction implements FocusListener {
		private final JTable table;
		private static final long serialVersionUID = 1L;

		public QuitEditModeAction(JTable table) {
			this.table = table;
		}

		@Override
		public void actionPerformed(ActionEvent event) {
			quitEditMode();
		}

		@Override
		public void focusGained(FocusEvent event) {
		}

		@Override
		public void focusLost(FocusEvent event) {
			quitEditMode();
		}

		/**
		 * On VK_ESCAPE or when JTable loses focus, quit the "edit mode"
		 * 
		 * @url https://stackoverflow.com/questions/65562456/switch-between-row-selection-mode-and-single-cell-selection-mode-in-java-jfc-swi/65563977#65563977
		 */
		private void quitEditMode() {
			System.out.println("editing de-activated");
			table.setCellSelectionEnabled(false);
			// StackOverflow
			final int selectedColumn = table.getSelectedColumn();
			final int selectedRow = table.getSelectedRow();

			table.setRowSelectionAllowed(true);

			// StackOverflow
			if (selectedColumn > -1 && selectedRow > -1) {
				table.setRowSelectionInterval(selectedRow, selectedRow);
			}

			InputMap input = table.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
			input.remove(shiftTabKey);
			input.remove(tabKey);
			input.put(shiftTabKey, TAB_BEFORE_NAME);
			input.put(tabKey, TAB_NEXT_NAME);
			input.put(enterKey, EDIT_MODE_ACTION_NAME);

		}
	}

	/**
	 * on VK_ENTER, navigate in JTable only ("edit mode")
	 *
	 * @author Peter Schuster (setrok)
	 * @date 06.01.2021
	 *
	 */
	public final class EditModeAction extends AbstractAction {
		private final JTable table;
		private static final long serialVersionUID = 1L;
		private final KeyStroke escKey = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

		/**
		 * Konstruktor: TODO (kommentieren)
		 * 
		 * @param table
		 */
		public EditModeAction(JTable table) {
			this.table = table;
		}

		@Override
		public void actionPerformed(ActionEvent event) {
			editMode();
		}

		/**
		 * on VK_ENTER, navigate in JTable only ("edit mode")
		 * 
		 * @url https://stackoverflow.com/questions/61556057/best-practise-for-handling-vk-tab-in-java-swing-to-move-between-components-as-we
		 */
		private void editMode() {
			System.out.println("editing activated");
			table.setCellSelectionEnabled(true);
			InputMap input = table.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
			input.remove(shiftTabKey);
			input.remove(tabKey);
			input.remove(enterKey);
			// StackOverflow
			input.put(shiftTabKey, "selectPreviousColumnCell");
			input.put(tabKey, "selectNextColumnCell");
			input.put(escKey, QUIT_EDIT_MODE_ACTION_NAME);
			table.getActionMap().get("selectNextColumnCell")
					.actionPerformed(new ActionEvent(table, ActionEvent.ACTION_PERFORMED, "selectNextColumnCell"));
		}
	}

	private static final long serialVersionUID = -6667909235024117172L;

	private final KeyStroke tabKey = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0);
	private final KeyStroke shiftTabKey = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, InputEvent.SHIFT_DOWN_MASK);
	private final KeyStroke enterKey = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);

	public final AbstractAction eintragEntfernenAktion = new EintragEntfernenAktion(
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(63105));

	AbstractAction aaTabNext = new TabulatorNextAction();
	AbstractAction aaTabBefore = new TabulatorBeforeAction();

	public static final NumberFormatter FORMATTER_INT = new NumberFormatter(NumberFormat.getIntegerInstance());

	/**
	 * Dank an JavaBeginners, Mit einem NumberFormatter lässt sich ein TextFeld so
	 * formatieren, dass nur Zahlen als Eingabe akzeptiert werden. Die Grundlage
	 * hierfür bildet ein JFormattedTextField.
	 * 
	 * @url "http://www.javabeginners.de/Swing/Swing-Elemente/Textkomponenten/Textfeld_fuer_Zahlen.php"
	 */
	public static final NumberFormat FORMAT = NumberFormat.getInstance();
	public static final NumberFormatter FORMATTER = new NumberFormatter(FORMAT);
	public static final TooltipHeaderRenderer HEADER_RENDERER = new TooltipHeaderRenderer();
	public static final EuroEditor EURO_EDITOR = new EuroEditor(FORMATTER);
	public static final EuroRenderer EURO_RENDERER = new EuroRenderer();

	private long start = 0;

	private long ende = 0;
	/**
	 * Konstruktor setzt das GridBagLayout als LayoutManager
	 * 
	 */
	protected KostenJPanel() {
		super(new GridBagLayout());
		setStart(Kostenrechner.systemDefaultZone.millis());
		FORMAT.setGroupingUsed(false);
		Kostenrechner.getLogger().info("Starte JPanel " + this.getClass().toString());
	}

	/**
	 * Die Methode dient dazu, die Oberflächenelemente für das Panel zu setzen, wie
	 * bisher fuelleXxxPane()
	 * 
	 */
	public abstract void buildPanel();

	/**
	 * Die Methode dient dazu, eine Referenz auf das Hauptfenster zu übermitteln
	 * 
	 * @param kostenrechner
	 */
	public abstract void setKostenrechner(Kostenrechner kostenrechner);

	/**
	 * @return gibt {@link #start} als {@link long} zurück.
	 */
	public long getStart() {
		return start;
	}

	/**
	 * @return gibt {@link #ende} als {@link long} zurück.
	 */
	public long getEnde() {
		return ende;
	}

	/**
	 * Die Methode errechnet die Dauer des Starts
	 * 
	 * @return die Dauer in Millisekunden.
	 */
	public long errechneDauer() {
		if (getEnde() == 0)
			setEnde(Kostenrechner.systemDefaultZone.millis());
		return getEnde() - getStart();
	}

	/**
	 * Verhindert, dass die übergegebene {@link JTable} <code>table</code> die
	 * Tabulatortaste und Umschalt+Tabulatortaste "frisst".
	 * 
	 * @param table die {@link JTable}
	 * @url "http://www.coderanch.com/t/342895/GUI/java/JTable-focus" aus: Big Moose
	 *      Saloon, A friendly place for programming greenhorns!
	 */
	public void entferneTastaturFalle(JTable table) {
		table.getInputMap(JComponent.WHEN_FOCUSED).put(tabKey, TAB_NEXT_NAME);
		table.getActionMap().put(TAB_NEXT_NAME, aaTabNext);
		table.getInputMap(JComponent.WHEN_FOCUSED).put(shiftTabKey, TAB_BEFORE_NAME);
		table.getActionMap().put(TAB_BEFORE_NAME, aaTabBefore);
	}

	/**
	 * Die Methode baut {@link EditModeAction} und {@link QuitEditModeAction} in die
	 * übergebene {@link JTable} ein.
	 * 
	 * @param table die {@link JTable}
	 * 
	 * @url https://stackoverflow.com/questions/65562456/switch-between-row-selection-mode-and-single-cell-selection-mode-in-java-jfc-swi/65563977#65563977
	 * @url https://stackoverflow.com/questions/61556057/best-practise-for-handling-vk-tab-in-java-swing-to-move-between-components-as-we
	 */
	public void baueEditModeEin(JTable table) {
		final AbstractAction editModeAction = new EditModeAction(table);
		table.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enterKey, EDIT_MODE_ACTION_NAME);
		table.getActionMap().put(EDIT_MODE_ACTION_NAME, editModeAction);

		final QuitEditModeAction quitEditModeAction = new QuitEditModeAction(table);
		table.getActionMap().put(QUIT_EDIT_MODE_ACTION_NAME, quitEditModeAction);
		table.addFocusListener(quitEditModeAction);
	}

	/**
	 * Die Methode dient dazu, bei den JTables für die GebührenTatbestände die
	 * Spaltenbreite einzustellen, den Renderer und den Transferhandler zu setzen.
	 * 
	 * @param table eine JTable für GebührenTatbestände
	 * 
	 */
	public void setzeGebuehrenTabelleAussehen(JTable table) {
		table.getColumnModel().getColumn(0).setMinWidth(180);
		table.getColumnModel().getColumn(0).setPreferredWidth(180);
		table.getColumnModel().getColumn(1).setPreferredWidth(40);
		table.getColumnModel().getColumn(2).setPreferredWidth(60);
		table.setDefaultRenderer(Double.class, new GebuehrenTableRenderer());
		table.getTableHeader().setDefaultRenderer(HEADER_RENDERER);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.setTransferHandler(new GebuehrenTransferHandler(table));
	}

	/**
	 * Die Methode fügt das {@link KostenJPanel} in die {@link JTabbedPane} als
	 * neuen Tab ein.
	 * 
	 * @param pane         das {@link JTabbedPane}
	 * @param beschriftung eine {@link Beschriftung}
	 * @throws IllegalArgumentException  wenn das neue {@link JPanel}
	 *                                   <code>null</code> ist.
	 * @throws IndexOutOfBoundsException wenn sich das neue {@link JPanel} nicht
	 *                                   einfügen lässt
	 */
	public void intoTabbedPane(JTabbedPane pane, Beschriftung beschriftung)
			throws IllegalArgumentException, IndexOutOfBoundsException {
		pane.addTab(beschriftung.getTitle(), null, this, beschriftung.getShortDescription());
		int index = pane.getTabCount() - 1;
		pane.setEnabledAt(index, true);
		pane.setMnemonicAt(index, beschriftung.getMnemonic());
		beschrifter.beschrifte(this, beschriftung);
	}

	/**
	 * @param start d. {@link #start}, d. gesetzt werden soll als {@link long}.
	 */
	protected void setStart(long start) {
		this.start = start;
	}

	/**
	 * @param ende d. {@link #ende}, d. gesetzt werden soll als {@link long}.
	 */
	protected void setEnde(long ende) {
		this.ende = ende;
	}

	/**
	 * Die Methode loggt die Dauer in Millisekunden
	 */
	protected void logDauer() {
		Kostenrechner.getLogger()
				.info(String.format("%s brauchte: %d millis", this.getClass().toString(), errechneDauer()));
	}

	/**
	 * Die Methode dient dazu, die formatierten Streitwerte wieder zu long zu
	 * machen, indem "," und "." aus den Zeichenketten gestrichen werden.
	 * 
	 * @param fTf          das {@link JFormattedTextField} mit dem Zahlenwert
	 * @param beschriftung die Beschriftung des {@link JLabel}s des Felds
	 * 
	 * @return ein long
	 */
	protected long longAusFormatter(JFormattedTextField fTf, String beschriftung) throws NumberFormatException {
		if (fTf == null)
			return 0L;
		return new NumberExtractor(fTf, beschriftung).extractLong();
	}
}