/**
 * BaumbachAssistentDialog.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.BEKLAGTE;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.DRITTWIDERBEKLAGTE;
import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.BeteiligtenTyp.KLAEGER;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;

import javax.swing.AbstractAction;
import javax.swing.JCheckBox;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextPane;
import javax.swing.ListSelectionModel;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.html.HTMLDocument;

import eu.gronos.beschriftungen.ComponentBeschrifter;
import eu.gronos.beschriftungen.interfaces.Vorsorgend;
import eu.gronos.beschriftungen.model.Beschriftung;
import eu.gronos.beschriftungen.model.LangBeschriftung;
import eu.gronos.beschriftungen.model.NameContainerSammlung;
import eu.gronos.beschriftungen.model.UnBeschriftung;
import eu.gronos.beschriftungen.model.VorsorgendeBeschriftung;
import eu.gronos.beschriftungen.view.TabbedPaneFactory;
import eu.gronos.kostenrechner.Kostenrechner;
import eu.gronos.kostenrechner.controller.baumbach.AssistentBkStreitwertHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentBkUnterliegenHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentWbkStreitwertHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentWbkUnterliegenHorcher;
import eu.gronos.kostenrechner.controller.baumbach.AssistentWeiterAction;
import eu.gronos.kostenrechner.controller.baumbach.StreitgenossenAssistentParser;
import eu.gronos.kostenrechner.controller.files.HtmlWerkzeugKasten;
import eu.gronos.kostenrechner.controller.files.XmlTransferHandler;
import eu.gronos.kostenrechner.controller.system.FehlerHelper;
import eu.gronos.kostenrechner.data.baumbach.Streitgenossen;
import eu.gronos.kostenrechner.data.tenordaten.Allgemein;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.data.tenordaten.VerfahrensDatenContainer;
import eu.gronos.kostenrechner.interfaces.ParsendUndBauend;
import eu.gronos.kostenrechner.model.baumbach.ProtoBeteiligtenWerte;
import eu.gronos.kostenrechner.view.HinzufuegenDialog;

/**
 * Assistent zum Hinzufügen aller Beteiligten auf einmal. Mehrere Tabs/Schritte.
 * 1.) Zuerst müssen die einfachen (Baumbach?)Beteiligten für die Klage erstellt
 * werden, ohne das Unterliegen und den Streitwert festzulegen (ggf.
 * XJustiz-Import) bzw. mit einheitlichem Streitwert. 2.) Dann können die
 * Beteiligten der Widerklage hinzugefügt werden, wieder einheitlicher
 * Streitwert. 3.) Dann kann angegeben werden, welche Beteiligten
 * gesamtschuldnerisch und zu wieviel verurteilt wurden. Erst jetzt werden die
 * eigentlichen BaumbachBeteiligten initialisiert (oder jetzt bekommen sie ihre
 * Werte).
 * 
 * @author Peter Schuster (setrok)
 * @date 22.06.2014
 * 
 * 
 */
public class BaumbachAssistentDialog extends HinzufuegenDialog<VerfahrensDatenContainer> {

	private static final long serialVersionUID = -170526643502912253L;

	private JTabbedPane tabbedPane;
	private JPanel[] panels;
	private VerfahrensDatenContainer verfahrensDatenContainer;
	private JTextPane textPane;
	private JSpinner spiKlaegerZahl;
//	private JSpinner spiBeklagtenZahl;
	private JTable tabKlaegerGenus;
//	private JScrollPane scpKlaegerGenus;
	private SimpleBeteiligtenTable tabBeklagtenGenus;
//	private JScrollPane scpBeklagteGenus;
	private JCheckBox chbBkSwEinheitlich;
	private JFormattedTextField ftfBkStreitwert;
	private SimpleBeteiligtenTable tabBeklagtenSw;
//	private JScrollPane scpBkSw;
	private JCheckBox chBkGesamtschuldner;
	private JList<Beteiligter> liBkGesamtschuldner;
//	private JScrollPane scpBeklagtenAuswahlListe;
	private JFormattedTextField ftfBkGesamtUnterliegen;
	private SimpleBeteiligtenTable tbBkUnt;
//	private JScrollPane scpBkUnt;
	private JList<Beteiligter> liBkWk;
//	private JScrollPane scpBkWk;
//	private JSpinner spiDwbkAnz;
	private SimpleBeteiligtenTable tbDwbkGenus;
//	private JScrollPane scpDwbkGenus;
	private JCheckBox chbWkEinheitlich;
	private JFormattedTextField ftfWkSw;
	private SimpleBeteiligtenTable tabKlSw;
//	private JScrollPane scpKlSw;
	private SimpleBeteiligtenTable tabDwbkSw;
//	private JScrollPane scpDwbkSw;
	private JCheckBox chbWkGesamtschuld;
	private JList<Beteiligter> liWkGesamtschuldner;
	private JFormattedTextField ftfWkUnterliegen;
	private SimpleBeteiligtenTable tbKlDarueber;
	private JScrollPane scpKlDarueber;
	private SimpleBeteiligtenTable tbDwbkDarueber;
//	private JScrollPane scpDwbkDarueber;
//	private JCheckBox chbKlWk;
//	private JLabel lblBkStreitwert;
//	private JLabel lblBkGesamtUnterliegen;
//	private JLabel lblWkSw;
//	private JLabel lblWkUnterliegen;

	public final ProtoBeteiligtenWerte primitives;
	private final ParsendUndBauend<Streitgenossen> pruefer = new StreitgenossenAssistentParser();
	private AbstractAction weiter = new AssistentWeiterAction(this,
			(LangBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38011), pruefer);

	/**
	 * Der Konstruktor für den Standardfall ohne Xjustiz-Import
	 * 
	 * @param owner         the JFrame from which the dialog is displayed
	 * @param mitWiderklage sollen auch die Schritte für die Widerklage durchgeführt
	 *                      werden?
	 */
	public BaumbachAssistentDialog(JFrame owner, boolean mitWiderklage) {
		super(owner, (VorsorgendeBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38000));
		this.primitives = new ProtoBeteiligtenWerte();
		this.primitives.mitWiderklage = mitWiderklage;
		((StreitgenossenAssistentParser) pruefer).setZustand(0);
		this.setContainer(null);
		primitives.ohneXjustiz = getVerfahrensDatenContainer() == null;
		setOkButtonToBeFocussed(false);
		Kostenrechner.getLogger().info("BaumbachAssistentDialog - ohne Xjustiz!");
		((StreitgenossenAssistentParser) pruefer).setRueckgabewert(new Streitgenossen());
		// setPrimitives darf nur einmal aufgerufen werden! - dann wohl hier
		((StreitgenossenAssistentParser) pruefer).setPrimitives(this.primitives);
	}

	/**
	 * Ein Konstruktor, der Verfahrensdaten vom XJustiz-Import annimmt und dann ein
	 * paar Felder schon einmal vorbelegen kann.
	 * 
	 * @param owner     the JFrame from which the dialog is displayed
	 * @param container ein KostenTenorContainer aus einem Xjustiz-Import
	 */
	public BaumbachAssistentDialog(JFrame owner, VerfahrensDatenContainer container) {
		super(owner, (Beschriftung) new ComponentBeschrifter()
				.changeTo(((Vorsorgend<?>) NameContainerSammlung.BESCHRIFTUNGEN.get(38000)), 1));
//		if () {}
		this.primitives = ((StreitgenossenAssistentParser) pruefer).pruefeVerfahrensDatenContainer(container,
				new ProtoBeteiligtenWerte());
		((StreitgenossenAssistentParser) pruefer).setZustand(0);
		this.setContainer(container);
		setOkButtonToBeFocussed(false);
		Kostenrechner.getLogger().info("BaumbachAssistentDialog - mit Xjustiz!");
		((StreitgenossenAssistentParser) pruefer).setRueckgabewert(container.streitgenossen);
		// setPrimitives darf nur einmal aufgerufen werden! - dann wohl hier
		((StreitgenossenAssistentParser) pruefer).setPrimitives(this.primitives);
	}

	/**
	 * Die Methode setzt alle Tabs außer den aktuellen und den Start-Tab auf deaktiv
	 * 
	 * @param index Index des zu aktivierenden Tabs
	 */
	public void zustandAnzeigen(int index) {
		for (int i = 0; i < (((StreitgenossenAssistentParser) pruefer).getHoechsterZustand())
				&& panels[i] != null; i++) {
			if (i == 0) {
				tabbedPane.setEnabledAt(i, true);
			} else if (i == index) {
				tabbedPane.setEnabledAt(i, true);
				tabbedPane.setSelectedIndex(i);
			} else {
				tabbedPane.setEnabledAt(i, false);
			}
		}
		/* Solange die Eingabe noch nicht abgeschlossen ist, Ok ausblenden */
		if (index < ((StreitgenossenAssistentParser) pruefer).getHoechsterZustand() - 1) {
			super.getOkButton().requestFocusInWindow();
		} else {
			restoreOkButton();
			super.getOkButton().requestFocusInWindow();
		}
	}

	/**
	 * Die Methode darf erst ganz zum Schluss true melden, abhängig davon also, ob
	 * eine Widerklage vorliegt oder nicht. Wenn keine Widerklage vorliegt, ist der
	 * Vorgang nach dem Panel 5 fertig, mit Widerklage nach dem Panel 10.
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#parseEingabe()
	 */
	@Override
	public boolean parseEingabe() {
		Kostenrechner.getLogger()
				.info(String.format("Zustand: %d", ((StreitgenossenAssistentParser) pruefer).getZustand()));
		boolean boole = false;
		try {
			boole = pruefer.parseEingabe();
		} catch (IllegalArgumentException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			boole = false;
		} catch (NullPointerException e) {
			FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			boole = false;
		}
		return boole;
	}

	/**
	 * @return gibt {@link #verfahrensDatenContainer} als VerfahrensDatenContainer
	 *         zurück.
	 */
	public VerfahrensDatenContainer getVerfahrensDatenContainer() {
		return verfahrensDatenContainer;
	}

	/**
	 * Diese Methode erstellt zum Abschluss BaumbachBeteiligtenListen
	 * 
	 * Erst wenn der letzte Zustand erreicht ist, darf gebaut werden (sollte
	 * eigentlich von parseEingabe() geprüft werden
	 * 
	 * @return soll einen {@link VerfahrensDatenContainer} zurückgeben
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#baueRueckgabewert()
	 */
	@Override
	protected VerfahrensDatenContainer baueRueckgabewert() {
		VerfahrensDatenContainer c = new VerfahrensDatenContainer();
		/* Den UnterContainer Streitgenossen erstellt der pruefer */
		c.streitgenossen = pruefer.getRueckgabewert();
		/* Dann die allgemeinen Teile einpacken */
		Allgemein a = c.allgemein;
		a.speicherdatum = new XmlTransferHandler().speicherDatum();
		if (!primitives.ohneXjustiz) {
			a.aktenzeichen = getVerfahrensDatenContainer().allgemein.aktenzeichen;
		} else {
			a.aktenzeichen = "";
		}
		// if (primitives.wkStreitwert > primitives.bkStreitwert)
		if (primitives.wkStreitwert.greaterThan(primitives.bkStreitwert))
			a.streitwert = primitives.wkStreitwert;
		else
			a.streitwert = primitives.bkStreitwert;
		a.selectedPanel = Streitgenossen.TAB_ID;
		Kostenrechner.getLogger().info(primitives.toString());
		return c;
	}

	/**
	 * Die Methode initialisiert die allgemeinen JComponents für den Inhaltsbereich
	 * des Dialogs und wird vom Konstruktor selbsttätig aufgerufen. In der
	 * abgeleiteten Klasse baut die Methode eine {@link javax.swing.JTabbedPane}
	 * auf, mit so vielen Tabs wie Zuständen, die in einem Array JPanel[] panels
	 * aufbewahrt werden.
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#setContentPanel()
	 */
	@Override
	protected void setContentPanel() {
		tabbedPane = new JTabbedPane(JTabbedPane.TOP);
		ComponentBeschrifter beschrifter = new ComponentBeschrifter();
		beschrifter.beschrifte(tabbedPane, NameContainerSammlung.BESCHRIFTUNGEN.get(38001));
		/* BorderLayout.NORTH führt zu Darstellungsfehlern beim Vergrößern */
		getContentPane().add(tabbedPane, BorderLayout.CENTER);
		tabbedPane.setBorder(BORDER);
		panels = new JPanel[StreitgenossenAssistentParser.HOECHSTER_ZUSTAND];
		/*
		 * for-Schleife zum Anlegen der Reiter Die tabbedPanes können hier noch nicht
		 * angelegt werden, weil mitWiderklage noch nicht bekannt ist.
		 */
		for (int i = 0; i < panels.length; i++) {
			panels[i] = new JPanel();
			panels[i].setBorder(BORDER);
			panels[i].setLayout(new GridBagLayout());
		}
	}

	/**
	 * Füllt die Tabs mit Oberflächenelemente
	 * 
	 * @see eu.gronos.kostenrechner.view.HinzufuegenDialog#fuelleContentPane()
	 */
	@Override
	protected void fuelleContentPane() {
		baueWeiterButton();
		/* TabbedPanes anlegen */
		final int[] keys = new int[] { 38002, 38020, 38100, 38200, 38300, 38400, 38500, 38600, 38700, 38800, 38900 };
		// ohne Widerklage früher aufhören!
		for (int i = 0; i < panels.length && (primitives.mitWiderklage
				|| i < StreitgenossenAssistentParser.HOECHSTER_ZUSTAND_OHNE_WIDERKLAGE); i++) {
			Beschriftung beschriftung = (Beschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(keys[i]);
			TabbedPaneFactory.intoTabbedPane(tabbedPane, panels[i], beschriftung);
		}

		/*
		 * Jeder Schritt ist ein einzelnes JPanel im tabbedPane.
		 * 
		 * Nur je eine sbtm-Variable für kl, bk, dwbk!
		 */
		// Panel 0:
		final HtmlWerkzeugKasten kit = new HtmlWerkzeugKasten();

		textPane = (JTextPane) beschrifter.beschrifteVergittert(new JTextPane(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38005), panels[0]);
		textPane.setEditorKit(kit);
		kit.read(willkommenRessource(), textPane);

		if (!primitives.ohneXjustiz) {
			baueAzEin(textPane, getVerfahrensDatenContainer().allgemein.aktenzeichen);
		}

		// panels[0].add(
//		beschrifter.testeUnVergittert(panels[0], textPane,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 1:
		// Kann bei Import übersprungen werden
//		JLabel lblKlaegerZahl = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38021), panels[1]);
//		lblKlaegerZahl.setHorizontalAlignment(SwingConstants.RIGHT);

		spiKlaegerZahl = (JSpinner) beschrifter.beschrifteGemeinsam(new JSpinner(),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38025), panels[1]);
//		lblKlaegerZahl.setLabelFor(spiKlaegerZahl);
		spiKlaegerZahl.setEnabled(false);

//		JLabel lblBeklagtenZahl = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38031), panels[1]);
//		lblBeklagtenZahl.setHorizontalAlignment(SwingConstants.RIGHT);

//		spiBeklagtenZahl = (JSpinner) 
		beschrifter.beschrifteGemeinsam(new JSpinner(),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38035), panels[1]);
//		lblBeklagtenZahl.setLabelFor(spiBeklagtenZahl);

		// .add(
//		beschrifter.testeUnVergittert(panels[1], lblKlaegerZahl,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[1], spiKlaegerZahl,
//				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[1], lblBeklagtenZahl,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[1], spiBeklagtenZahl,
//				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 2
		// kann bei Import übersprungen werden
//		JLabel lblGenusKlaeger = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38101), panels[2]);

		tabKlaegerGenus = (JTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(KLAEGER),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38111));
//		scpKlaegerGenus = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tabKlaegerGenus),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38110), panels[2]);
//		lblGenusKlaeger.setLabelFor(scpKlaegerGenus);

//		JLabel lblBeklagtenGenus = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38121), panels[2]);

		tabBeklagtenGenus = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(BEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38131));
//		scpBeklagteGenus = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tabBeklagtenGenus),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38130), panels[2]);
//		lblBeklagtenGenus.setLabelFor(scpBeklagteGenus);

//		panels[2].add(		.add(
//		beschrifter.testeUnVergittert(panels[2], lblGenusKlaeger,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[2], scpKlaegerGenus,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));
//		beschrifter.testeUnVergittert(panels[2], lblBeklagtenGenus,
//				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[2], scpBeklagteGenus,
//				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 3
		chbBkSwEinheitlich = (JCheckBox) beschrifter.beschrifteVergittert(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38201), panels[3]);

//		lblBkStreitwert = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38211), panels[3]);
//		lblBkStreitwert.setHorizontalAlignment(SwingConstants.RIGHT);

		ftfBkStreitwert = (JFormattedTextField) beschrifter.beschrifteGemeinsam(new JFormattedTextField(formatter),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38215), panels[3]);
		// Vorbelegen mit dem Wert aus Xjustiz, sofern bekannt
//		lblBkStreitwert.setLabelFor(ftfBkStreitwert);
		// "Weiter"-Action der InputMap hinzufügen und mit VK_ENTER verknüpfen
		setEnterAction(ftfBkStreitwert, weiter);

		tabBeklagtenSw = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(BEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38221));
		chbBkSwEinheitlich.addChangeListener(new AssistentBkStreitwertHorcher(ftfBkStreitwert, tabBeklagtenSw));
//		scpBkSw = (JScrollPane) 
		beschrifter.beschrifteVergittert(new JScrollPane(tabBeklagtenSw),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38220), panels[3]);

		// .add( panels[3].add(
//		beschrifter.testeUnVergittert(panels[3], chbBkSwEinheitlich,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[3], lblBkStreitwert,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[3], ftfBkStreitwert,
//				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[3], scpBkSw,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));

		// Panel 4
		chBkGesamtschuldner = (JCheckBox) beschrifter.beschrifteVergittert(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38301), panels[4]);
		chBkGesamtschuldner.setSelected(true);

//		JLabel lblBkGesamtschuldner = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38311), panels[4]);

		// Hier muss ich wohl die erste Spalte nehmen...
		liBkGesamtschuldner = new JList<Beteiligter>();
		beschrifter.beschrifte(liBkGesamtschuldner, NameContainerSammlung.BESCHRIFTUNGEN.get(38321));
		liBkGesamtschuldner.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		liBkGesamtschuldner.setCellRenderer(new BeteiligtenListRenderer(false));
//		scpBeklagtenAuswahlListe = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(liBkGesamtschuldner),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38320), panels[4]);
//		lblBkGesamtschuldner.setLabelFor(scpBeklagtenAuswahlListe);

//		lblBkGesamtUnterliegen = (JLabel) 
		beschrifter.beschrifteVergittert(new JLabel(), NameContainerSammlung.BESCHRIFTUNGEN.get(38331), panels[4]);
//		lblBkGesamtUnterliegen.setHorizontalAlignment(SwingConstants.RIGHT);

		ftfBkGesamtUnterliegen = (JFormattedTextField) beschrifter.beschrifteGemeinsam(
				new JFormattedTextField(formatter), (UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38335),
				panels[4]);
//		lblBkGesamtUnterliegen.setLabelFor(ftfBkGesamtUnterliegen);
		setEnterAction(ftfBkGesamtUnterliegen, weiter);
		/*
		 * Muss nach dem Mnemonic, weil das Zuweisen eines Mnemonic wohl das ChangeEvent
		 * auch auslöst. Und auch nach der Initialisierung der Oberflächenelemente.
		 */
		chBkGesamtschuldner
				.addChangeListener(new AssistentBkUnterliegenHorcher(liBkGesamtschuldner, ftfBkGesamtUnterliegen));

		// panels[4].add(
//		beschrifter.testeUnVergittert(panels[4], chBkGesamtschuldner,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[4], lblBkGesamtschuldner,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[4], scpBeklagtenAuswahlListe,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));
//		beschrifter.testeUnVergittert(panels[4], lblBkGesamtUnterliegen,
//				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[4], ftfBkGesamtUnterliegen,
//				GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 5
		// ==> sonst Unterliegen 0
//		JLabel lblBkUnt = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38401), panels[5]);
		tbBkUnt = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(BEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38411));
//		scpBkUnt = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tbBkUnt),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38410), panels[5]);
//		lblBkUnt.setLabelFor(scpBkUnt);

		// panels[5].add(
//		beschrifter.testeUnVergittert(panels[5], lblBkUnt,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[5], scpBkUnt,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));

		// AB HIER NUR WENN (mitWiderklage == true)
		if (primitives.mitWiderklage == false) {
			zustandAnzeigen(((StreitgenossenAssistentParser) pruefer).getZustand());
			panelsAusblendenBeiImport();
			return;
		}

		// Panel 6
		// kann bei Import übersprungen werden
//		JLabel lblBkWk = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38501), panels[6]);
		liBkWk = new JList<Beteiligter>();
		beschrifter.beschrifte(liBkWk, NameContainerSammlung.BESCHRIFTUNGEN.get(38511));
		liBkWk.setCellRenderer(new BeteiligtenListRenderer());
		liBkWk.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
//		scpBkWk = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(liBkWk),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38510), panels[6]);
//		lblBkWk.setLabelFor(scpBkWk);

//		JLabel lblDwbkAnz = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38521), panels[6]);
//		lblDwbkAnz.setHorizontalAlignment(SwingConstants.RIGHT);
//		spiDwbkAnz = (JSpinner) 
		beschrifter.beschrifteGemeinsam(new JSpinner(),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38525), panels[6]);
//		lblDwbkAnz.setLabelFor(spiDwbkAnz);
		// Abfragen, ob Wk auch gegen den Kläger geht
//		chbKlWk = (JCheckBox) 
		beschrifter.beschrifteVergittert(new JCheckBox(), NameContainerSammlung.BESCHRIFTUNGEN.get(38531), panels[6]);

		// panels[6].add(
//		beschrifter.testeUnVergittert(panels[6],lblBkWk,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 3, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[6], scpBkWk,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 3, 1, GridBagConstraints.BOTH, false));
//		beschrifter.testeUnVergittert(panels[6], lblDwbkAnz,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[6], spiDwbkAnz,
//				GitterBeutelBeschraenkungen.getInstance(1, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[6], chbKlWk,
//				GitterBeutelBeschraenkungen.getInstance(2, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 7
		// kann bei Import übersprungen werden
//		JLabel lblDwbkGenus = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38601), panels[7]);
		tbDwbkGenus = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(DRITTWIDERBEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38611));
//		scpDwbkGenus = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tbDwbkGenus),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38610), panels[7]);
//		lblDwbkGenus.setLabelFor(scpDwbkGenus);

		// panels[7].add(
//		beschrifter.testeUnVergittert(panels[7], lblDwbkGenus,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[7], scpDwbkGenus,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 8
		chbWkEinheitlich = (JCheckBox) beschrifter.beschrifteVergittert(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38701), panels[8]);

//		lblWkSw = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38711), panels[8]);
//		lblWkSw.setHorizontalAlignment(SwingConstants.RIGHT);
		ftfWkSw = (JFormattedTextField) beschrifter.beschrifteGemeinsam(new JFormattedTextField(formatter),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38715), panels[8]);
//		lblWkSw.setLabelFor(ftfWkSw);
		setEnterAction(ftfWkSw, weiter);

//		JLabel lblKlSw = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38721), panels[8]);
		tabKlSw = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(KLAEGER),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38731));
//		scpKlSw = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tabKlSw),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38730), panels[8]);
//		lblKlSw.setLabelFor(scpKlSw);

//		JLabel lblDwbkSw = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38741), panels[8]);
		tabDwbkSw = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(DRITTWIDERBEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38751));
		chbWkEinheitlich.addChangeListener(new AssistentWbkStreitwertHorcher(ftfWkSw, tabKlSw, tabDwbkSw, primitives));
//		scpDwbkSw = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tabDwbkSw),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38750), panels[8]);
//		lblDwbkSw.setLabelFor(scpDwbkSw);

		// panels[8].add(
//		beschrifter.testeUnVergittert(panels[8], chbWkEinheitlich,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[8], lblWkSw,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[8], ftfWkSw,
//				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[8], lblKlSw,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[8], scpKlSw,
//				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.BOTH, false));
//		beschrifter.testeUnVergittert(panels[8], lblDwbkSw,
//				GitterBeutelBeschraenkungen.getInstance(1, 2, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[8], scpDwbkSw,
//				GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1, GridBagConstraints.BOTH, false));

		// Panel 9
		chbWkGesamtschuld = (JCheckBox) beschrifter.beschrifteVergittert(new JCheckBox(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38801), panels[9]);

		JLabel lblWkGesamtschuldner = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38811), panels[9]);
		liWkGesamtschuldner = new JList<Beteiligter>();
		beschrifter.beschrifte(liWkGesamtschuldner, NameContainerSammlung.BESCHRIFTUNGEN.get(38821));
		liWkGesamtschuldner.setCellRenderer(new BeteiligtenListRenderer());
		lblWkGesamtschuldner.setLabelFor(liWkGesamtschuldner);
		liWkGesamtschuldner.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		// JScrollPane scpWkGesamtschuldner = (JScrollPane)
		beschrifter.beschrifteVergittert(new JScrollPane(liWkGesamtschuldner),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38820), panels[9]);

//		lblWkUnterliegen = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38831), panels[9]);
		ftfWkUnterliegen = (JFormattedTextField) beschrifter.beschrifteGemeinsam(new JFormattedTextField(formatter),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38835), panels[9]);
//		lblWkUnterliegen.setLabelFor(ftfWkUnterliegen);
		chbWkGesamtschuld.addChangeListener(
				new AssistentWbkUnterliegenHorcher(liWkGesamtschuldner, ftfWkUnterliegen, primitives));
		setEnterAction(ftfWkUnterliegen, weiter);

		// panels[9].add(
//		beschrifter.testeUnVergittert(panels[9], chbWkGesamtschuld,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 2, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[9], lblWkGesamtschuldner,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 2, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[9], scpWkGesamtschuldner,
//				GitterBeutelBeschraenkungen.getInstance(0, 2, 2, 1, GridBagConstraints.BOTH, false));
//		beschrifter.testeUnVergittert(panels[9], lblWkUnterliegen,
//				GitterBeutelBeschraenkungen.getInstance(0, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[9], ftfWkUnterliegen,
//				GitterBeutelBeschraenkungen.getInstance(1, 3, 1, 1, GridBagConstraints.HORIZONTAL, false));

		// Panel 10
		JLabel lblKlDarueber = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38901), panels[10]);
		tbKlDarueber = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(KLAEGER),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38911));
		scpKlDarueber = (JScrollPane) beschrifter.beschrifteVergittert(new JScrollPane(tbKlDarueber),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38910), panels[10]);
		lblKlDarueber.setLabelFor(scpKlDarueber);

//		JLabel lblDwbkDarueber = (JLabel) beschrifter.beschrifteVergittert(new JLabel(),
//				NameContainerSammlung.BESCHRIFTUNGEN.get(38921), panels[10]);
		tbDwbkDarueber = (SimpleBeteiligtenTable) beschrifter.beschrifte(new SimpleBeteiligtenTable(DRITTWIDERBEKLAGTE),
				NameContainerSammlung.BESCHRIFTUNGEN.get(38931));
//		scpDwbkDarueber = (JScrollPane) 
		beschrifter.beschrifteGemeinsam(new JScrollPane(tbDwbkDarueber),
				(UnBeschriftung) NameContainerSammlung.BESCHRIFTUNGEN.get(38930), panels[10]);
//		lblDwbkDarueber.setLabelFor(scpDwbkDarueber);

		// panels[10].add(
//		beschrifter.testeUnVergittert(panels[10], lblKlDarueber,
//				GitterBeutelBeschraenkungen.getInstance(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[10], scpKlDarueber,
//				GitterBeutelBeschraenkungen.getInstance(0, 1, 1, 1, GridBagConstraints.BOTH, false));
//		beschrifter.testeUnVergittert(panels[10], lblDwbkDarueber,
//				GitterBeutelBeschraenkungen.getInstance(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, false));
//		beschrifter.testeUnVergittert(panels[10], scpDwbkDarueber,
//				GitterBeutelBeschraenkungen.getInstance(1, 1, 1, 1, GridBagConstraints.BOTH, false));

		zustandAnzeigen(((StreitgenossenAssistentParser) pruefer).getZustand());
		panelsAusblendenBeiImport();
	}

	/**
	 * Die Methode entscheidet, welche HTML-Datei als Willkommensseite geladen
	 * werden soll.
	 * 
	 * @return ein {@link String} mit dem Namen der Resource
	 */
	private String willkommenRessource() {
		if (!primitives.ohneXjustiz)
			return "resources/assistent-xjustiz.html";
		else
			return "resources/assistent.html";
	}

	/**
	 * Die Methode baut das Aktenzeichen aus dem {@link VerfahrensDatenContainer} in
	 * das {@link HTMLDocument} ein.
	 * 
	 * @param jta          das {@link JTextPane}, in dem auch der Platzhalter für
	 *                     das Aktenzeichen steht.
	 * @param aktenzeichen das Aktenzeichen aus der Xjustiz-Datei.
	 */
	private void baueAzEin(JTextPane jta, String aktenzeichen) {
		final HTMLDocument htmlDoc = (HTMLDocument) jta.getDocument();
		Element span = htmlDoc.getElement("az");
		// Der Inhalt des Span-Tags
		if (span != null)
			try {
				System.err.println(htmlDoc.getText(span.getStartOffset(), span.getEndOffset() - span.getStartOffset()));
				htmlDoc.remove(span.getStartOffset(), span.getEndOffset() - span.getStartOffset());
				htmlDoc.insertString(span.getStartOffset(), " dem Verfahren " + aktenzeichen, span.getAttributes());
			} catch (BadLocationException e) {
				FehlerHelper.zeigeFehler(e.getLocalizedMessage(), e);
			}
		else
			System.err.println("Nix kann writen Aktenzeichen!");
	}

	/**
	 * Die Methode rekonstruiert den {@link HinzufuegenDialog#getOkButton()} wieder
	 * zu einer Schaltfläche "Ok". {@link HinzufuegenDialog} enthält dafür
	 * {@link HinzufuegenDialog#getOkAction()}.
	 */
	private void restoreOkButton() {
		super.getOkButton().setAction(super.getOkAction());
		super.setOkActionEnabled(true);
		removeFromInputMap(weiter);
		addToRootInputMap(super.getOkAction());
		beschrifter.beschrifte(super.getOkButton(), NameContainerSammlung.BESCHRIFTUNGEN.get(76051));
	}

	/**
	 * Die Methode ersetzt den {@link HinzufuegenDialog#getOkButton()} durch einen
	 * Weiter-Button
	 */
	private void baueWeiterButton() {
		super.setOkActionEnabled(false);
		removeFromInputMap(getOkAction());
		super.getOkButton().setAction(weiter);
		addToRootInputMap(weiter);
		beschrifter.beschrifte(super.getOkButton(), NameContainerSammlung.BESCHRIFTUNGEN.get(38011));
	}

	/**
	 * Die Methode sorgt dafür, dass bei Xjustiz-Import die nicht benötigten
	 * Panels/Tabs auch nicht stören. Zwar kann man diese Tabs nicht komplett
	 * ausblenden, aber zumindest werden die Titles auf null gesetzt, so dass sie
	 * leer sind. Der Assistent erscheint dadurch nicht mehr so groß.
	 * 
	 */
	private void panelsAusblendenBeiImport() {
		if (primitives.ohneXjustiz)
			return;
		if (primitives.tab1ueberspringen) {
			panels[1].setVisible(false);
			tabbedPane.setTitleAt(1, null);
		}
		if (primitives.tab2ueberspringen) {
			panels[2].setVisible(false);
			tabbedPane.setTitleAt(2, null);
		}
		if (primitives.tab4ueberspringen) {
			panels[4].setVisible(false);
			tabbedPane.setTitleAt(4, null);
		}
		if (primitives.mitWiderklage) {
			if (primitives.tab6ueberspringen) {
				tabbedPane.setTitleAt(6, null);
				panels[6].setVisible(false);
			}
			if (primitives.tab7ueberspringen) {
				tabbedPane.setTitleAt(7, null);
				panels[7].setVisible(false);
			}
			if (primitives.tab9ueberspringen) {
				tabbedPane.setTitleAt(9, null);
				panels[9].setVisible(false);
			}
		}
	}

	/**
	 * @param container d. {@link #verfahrensDatenContainer}, d. gesetzt werden soll
	 *                  als VerfahrensDatenContainer
	 */
	private void setContainer(VerfahrensDatenContainer container) {
		this.verfahrensDatenContainer = container;
	}

}
