package eu.gronos.kostenrechner.view.baumbach;

import static eu.gronos.kostenrechner.data.tenordaten.Beteiligter.Casus.NOMINATIV;

import java.awt.Component;

import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligtenListe;
import eu.gronos.kostenrechner.data.baumbach.BaumbachBeteiligter;
import eu.gronos.kostenrechner.data.baumbach.BaumbachGesamtschuldnerschaft;
import eu.gronos.kostenrechner.data.tenordaten.Beteiligter;
import eu.gronos.kostenrechner.model.baumbach.BeteiligtenTableModel;
import eu.gronos.kostenrechner.view.KostenTableRenderer;

/**
 * Ein Renderer für Beteiligter für JTable
 * 
 * @author Peter Schuster (setrok)
 * @date: 27.06.2014
 * 
 * @url "http://docs.oracle.com/javase/tutorial/uiswing/components/table.html"
 */
public class BeteiligtenTableRenderer extends KostenTableRenderer implements TableCellRenderer {
	private static final long serialVersionUID = -7430326486747341015L;
	private final boolean laufendeNummerImmerAusblenden;

	/**
	 * Konstruktor:
	 * 
	 * @param laufendeNummerImmerAusblenden wenn "true", so wird der Renderer
	 *                                      gezwungen, die laufende Nummer beim
	 *                                      Generieren des Parteibezeichners
	 *                                      auszublenden. Ansonsten überprüft der
	 *                                      Renderer, ob der darzustellende
	 *                                      Beteiligte einziger seiner Art ist
	 */
	public BeteiligtenTableRenderer(boolean laufendeNummerImmerAusblenden) {
		super();
		this.laufendeNummerImmerAusblenden = laufendeNummerImmerAusblenden;
	}

	/**
	 * Standardkonstruktor
	 * 
	 */
	public BeteiligtenTableRenderer() {
//		super();
//		this.laufendeNummerImmerAusblenden = false;
		this(false);
	}

	/**
	 * Gibt ein JLabel zum Darstellen der Tabelle zurück. Dabei wird der Text
	 * passend zum übergebenen Beteiligten gesetzt und der Tooltip gegebenenfalls
	 * etwas ausführlicher.
	 * 
	 * @param table      the <code>JTable</code> that is asking the renderer to
	 *                   draw; can be <code>null</code>
	 * @param value      the value of the cell to be rendered. It is up to the
	 *                   specific renderer to interpret and draw the value. For
	 *                   example, if <code>value</code> is the string "true", it
	 *                   could be rendered as a string or it could be rendered as a
	 *                   check box that is checked. <code>null</code> is a valid
	 *                   value
	 * @param isSelected true if the cell is to be rendered with the selection
	 *                   highlighted; otherwise false
	 * @param hasFocus   if true, render cell appropriately. For example, put a
	 *                   special border on the cell, if the cell can be edited,
	 *                   render in the color used to indicate editing
	 * @param row        the row index of the cell being drawn. When drawing the
	 *                   header, the value of <code>row</code> is -1
	 * @param column     the column index of the cell being drawn
	 * @return the component used for drawing the cell. This method is used to
	 *         configure the renderer appropriately before drawing.
	 * 
	 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable,
	 *      java.lang.Object, boolean, boolean, int, int)
	 */
	@Override
	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus,
			int row, int column) {
		if (value instanceof BaumbachGesamtschuldnerschaft) {
			/*
			 * Bei Gesamtschuldnerschaft ist es am einfachsten, den richtigen
			 * Parteibezeichner zu finden, weil die BaumbachGesamtschuldnerschaft ihn selbst
			 * vollständig weiß
			 */
			BaumbachGesamtschuldnerschaft bgs = (BaumbachGesamtschuldnerschaft) value;
			setText(bgs.kurzBezeichner(NOMINATIV).replace(BeteiligtenTableModel.GESAMTSCHULDNERISCH, ""));
			setToolTipText(bgs.parteiBezeichner(NOMINATIV, -1, true));
		} else if (value instanceof BaumbachBeteiligter && table.getModel() instanceof BeteiligtenTableModel) {
			/*
			 * Bei anderen BaumbachBeteiligten brauche ich einen Verweis auf die
			 * BaumbachBeteiligtenListe für die laufende Nummer. Deshalb geht das nur, wenn
			 * das TableModel ein BeteiligtenTableModel ist. Dann kann ich die Arbeit auf
			 * die BaumbachBeteiligtenListe abwälzen.
			 */
			BaumbachBeteiligtenListe bbtl = ((BeteiligtenTableModel) table.getModel()).getAllValues();
			setText(bbtl.parteiBezeichner(NOMINATIV, row));
			setToolTipText(bbtl.parteiBezeichner(NOMINATIV, row));
		} else if (value instanceof Beteiligter) {
			/*
			 * Wenn's ein "ganz einfacher" Beteiligter ist: direkt anzeigen lassen. Dieser
			 * Teil ist übernommen aus SimpleBeteiligtenRenderer
			 */
			Beteiligter b = (Beteiligter) value;
			boolean einzigerSeinerArt = laufendeNummerImmerAusblenden ? laufendeNummerImmerAusblenden
					: table.getRowCount() < 2;
			setText(b.parteiBezeichner(NOMINATIV, row + 1, einzigerSeinerArt));
			setToolTipText(b.parteiBezeichner(NOMINATIV, row + 1, einzigerSeinerArt));
		}
		setSelected(isSelected);
		return this;
	}
}
