/*
 * GebuehrenTatbestandRenderer.java
 * eu.gronos.kostenrechner.view.gebuehren (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.gebuehren;

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.ListCellRenderer;
import javax.swing.SwingConstants;

import eu.gronos.kostenrechner.data.gebuehren.AnwaltsGebuehrenTabelle;
import eu.gronos.kostenrechner.data.gebuehren.AuslagenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenAnrechnungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenErhoehungsTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenRahmenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenSatzTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.GebuehrenTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.MehrfachPauschalTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.MehrwertsteuerTatbestand;
import eu.gronos.kostenrechner.data.gebuehren.PauschalTatbestand;
import eu.gronos.kostenrechner.data.tenordaten.Euro;
import eu.gronos.kostenrechner.view.KostenListRenderer;

/**
 * Die Klasse rendert einen {@link GebuehrenTatbestand} als zweizeiliges
 * {@link JPanel}
 *
 * @author Peter Schuster (setrok)
 * @date 17 Jun 2019
 *
 */
public class GebuehrenListRenderer extends KostenListRenderer implements ListCellRenderer<GebuehrenTatbestand> {
	private static final long serialVersionUID = 511158574706627353L;
	/** Wieviel breiter als die Fontgröße muss die Spalte sein? */
	private static final int FONT_MULTIPLIKATOR = 21;// Zeichenanzahl von "Vorb. 3 Abs. 4 VV RVG"
	private final JLabel lbBezeichnung;
	private final JLabel lbGesetz;
	private final JLabel lbSatz;
	private final JLabel lbArt;

	/**
	 * Der Konstruktor setzt das {@link JPanel} schon mal zusammen.
	 * 
	 */
	public GebuehrenListRenderer() {
		super();

		/* JLabels bauen */
		lbBezeichnung = new JLabel();
		lbGesetz = new JLabel();
		lbSatz = new JLabel();
		lbArt = new JLabel();

		helfer.boldLabel(lbBezeichnung);

		lbSatz.setHorizontalAlignment(SwingConstants.RIGHT);

		helfer.sizeLabel(lbGesetz, FONT_MULTIPLIKATOR);

		/* Jetzt endlich das JPanel bauen */
		super.setLayout(new BorderLayout());
		super.add(lbBezeichnung, BorderLayout.PAGE_START);
		super.add(lbGesetz, BorderLayout.LINE_START);
		super.add(lbArt, BorderLayout.CENTER);
		super.add(lbSatz, BorderLayout.LINE_END);

		setAllOpaque(true);
	}

	/**
	 * Die Methode baut ein {@link JPanel} zur Darstellung von
	 * {@link GebuehrenTatbestand}en
	 * 
	 * @param list         The JList we're painting.
	 * @param value        the value of the cell to be rendered. It is up to the
	 *                     specific renderer to interpret and draw the value. For
	 *                     example, if value is the string "true", it could be
	 *                     rendered as a string or it could be rendered as a check
	 *                     box that is checked. null is a valid value
	 * @param index        The cells index.
	 * @param isSelected   True if the specified cell was selected.
	 * @param cellHasFocus True if the specified cell has the focus.
	 * @return ein {@link JPanel} mit vier {@link JLabel}s in zwei Zeilen, die
	 *         entweder einen Euro-Betrag mit zwei Nachkommastellen (bei einem
	 *         Auslagentatbestand) oder einen Gebührensatz mit einer Nachkommastelle
	 *         (ansonsten) darstellen.
	 * 
	 * @see javax.swing.ListCellRenderer#getListCellRendererComponent(javax.swing.JList,
	 *      java.lang.Object, int, boolean, boolean)
	 */
	@Override
	public Component getListCellRendererComponent(JList<? extends GebuehrenTatbestand> list, GebuehrenTatbestand value,
			int index, boolean isSelected, boolean cellHasFocus) {
		lbBezeichnung.setText(value.getBezeichnung());

		if (AnwaltsGebuehrenTabelle.class.equals(value.getGebuehrenKlasse())) {
			lbGesetz.setText(String.format("%s RVG", value.getVorschrift()));
		} else {
			lbGesetz.setText(String.format("%s GKG", value.getVorschrift()));
		}
		if (value instanceof GebuehrenAnrechnungsTatbestand) {
			final GebuehrenAnrechnungsTatbestand gat = (GebuehrenAnrechnungsTatbestand) value;
			lbArt.setText(GebuehrenAnrechnungsTatbestand.ART);
			lbSatz.setText(String.format("%,.2f-fach (%,.2f-%,.2f-fach)", gat.getSatz(), gat.getUntergrenze(),
					gat.getObergrenze()));
		} else if (value instanceof GebuehrenRahmenTatbestand) {
			final GebuehrenRahmenTatbestand grt = (GebuehrenRahmenTatbestand) value;
			lbArt.setText(GebuehrenRahmenTatbestand.ART);
			lbSatz.setText(String.format("%,.1f-fach (%,.1f-%,.1f-fach)", grt.getSatz(), grt.getUntergrenze(),
					grt.getObergrenze()));
		} else if (value instanceof GebuehrenErhoehungsTatbestand) {
			final GebuehrenErhoehungsTatbestand getb = (GebuehrenErhoehungsTatbestand) value;
			lbArt.setText(GebuehrenErhoehungsTatbestand.ART);
			lbSatz.setText(String.format("%,.1f-fach je %s (max. %,.1f-fach)", getb.getJeweils(), getb.getEinheit(),
					getb.getMaxSatz()));
		} else if (value instanceof GebuehrenSatzTatbestand) {
			final GebuehrenSatzTatbestand gst = (GebuehrenSatzTatbestand) value;
			lbArt.setText(GebuehrenSatzTatbestand.ART);
			lbSatz.setText(String.format("%,.1f-fach", gst.getSatz()));
		} else if (value instanceof MehrfachPauschalTatbestand) {
			final MehrfachPauschalTatbestand mptb = (MehrfachPauschalTatbestand) value;
			lbArt.setText(MehrfachPauschalTatbestand.ART);
			lbSatz.setText(String.format("%s EUR je %s", mptb.getJeweils(), mptb.getEinheit()));// "%,.2f EUR
		} else if (value instanceof AuslagenTatbestand) {
			final AuslagenTatbestand atb = (AuslagenTatbestand) value;
			lbArt.setText(AuslagenTatbestand.ART);
			// wenn nicht pauschal, dann ist getBetrag == null
			if (atb.getBetrag() == null || atb.getBetrag().compareTo(Euro.ofCents(0L)) < 0) {// < 0
				lbSatz.setText("Höhe variabel");
			} else {
				lbSatz.setText(String.format("%s EUR", atb.getBetrag()));//
			}
		} else if (value instanceof PauschalTatbestand) {
			final PauschalTatbestand ptb = (PauschalTatbestand) value;
			lbArt.setText(PauschalTatbestand.ART);
			lbSatz.setText(String.format("%s EUR", ptb.getBetrag()));// "%,.2f EUR"
		} else if (value instanceof MehrwertsteuerTatbestand) {
			final MehrwertsteuerTatbestand mwst = (MehrwertsteuerTatbestand) value;
			lbArt.setText(MehrwertsteuerTatbestand.ART);
			lbSatz.setText(String.format("%,.1f %%", 100.0 * mwst.getSteuerSatz()));
		} else {
			lbArt.setText("Sonstige");
			lbSatz.setText("");
		}
		super.setToolTipText(value.langBezeichnung());
		setSelected(isSelected);
		return this;
	}

}
