/**
 * TooltipHeaderRenderer.java
 * eu.gronos.kostenrechner (Kostenrechner)
 */
package eu.gronos.kostenrechner.view.gebuehren;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.border.Border;
import javax.swing.table.DefaultTableCellRenderer;

import eu.gronos.kostenrechner.interfaces.TooltipLieferant;

/**
 * Der Renderer versieht die Kopfzeile mit einem Tooltip. Jetzt von
 * {@link javax.swing.table.DefaultTableCellRenderer} statt von
 * {@link sun.swing.table.DefaultTableCellHeaderRenderer} abgeleitet.
 * 
 * @author Peter Felix Schuster (setrok)
 * @date 11.08.2014
 * 
 */
public class TooltipHeaderRenderer extends DefaultTableCellRenderer /* DefaultTableCellHeaderRenderer */ {
	private static final long serialVersionUID = 9155034513233553302L;

	/**
	 * Konstruktor: Standardkonstruktur der Oberklasse
	 * 
	 */
	public TooltipHeaderRenderer() {
		super();
		super.setHorizontalAlignment(JLabel.CENTER);
	}

	/**
	 * Die Methode erfragt die JComponent des und versieht ihn mit einem Tooltip.
	 * 
	 * @param table      the <code>JTable</code> that is asking the renderer to
	 *                   draw; can be <code>null</code>
	 * @param value      the value of the cell to be rendered. It is up to the
	 *                   specific renderer to interpret and draw the value. For
	 *                   example, if <code>value</code> is the string "true", it
	 *                   could be rendered as a string or it could be rendered as a
	 *                   check box that is checked. <code>null</code> is a valid
	 *                   value
	 * @param isSelected true if the cell is to be rendered with the selection
	 *                   highlighted; otherwise false
	 * @param hasFocus   if true, render cell appropriately. For example, put a
	 *                   special border on the cell, if the cell can be edited,
	 *                   render in the color used to indicate editing
	 * @param row        the row index of the cell being drawn. When drawing the
	 *                   header, the value of <code>row</code> is -1
	 * @param column     the column index of the cell being drawn
	 * @return the component used for drawing the cell. This method is used to
	 *         configure the renderer appropriately before drawing.
	 * 
	 * @see sun.swing.table.DefaultTableCellHeaderRenderer#getTableCellRendererComponent(javax.swing.JTable,
	 *      java.lang.Object, boolean, boolean, int, int)
	 */
	@Override
	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus,
			int row, int column) {
		JComponent component = (JComponent) super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row,
				column);
		if (table.getModel() instanceof TooltipLieferant) {
			TooltipLieferant lieferant = (TooltipLieferant) table.getModel();
			component.setToolTipText(lieferant.getTooltipText(column));
		} else
			setToolTipText(getText());
		setFontBold(component);
		setComponentBorder(component);
		return component;
	}

	/**
	 * Die Methode weist dem Header einen Rahmen zu.
	 * 
	 * @see {https://docs.oracle.com/javase/tutorial/uiswing/components/border.html}
	 * 
	 * @param component die {@link JComponent}
	 */
	private void setComponentBorder(JComponent component) {
		Border border = BorderFactory.createMatteBorder(0, 0, 1, 0, Color.BLACK);
		component.setBorder(border);		
	}

	private void setFontBold(JComponent component) {
		Font font = component.getFont();
		font = new Font(font.getName(), Font.BOLD, font.getSize());
		component.setFont(font);
	}
}
